package org.hildan.chrome.devtools.domains.webaudio

import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.webaudio.events.WebAudioEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [WebAudioDomain.getRealtimeData] command.
 */
@Serializable
public data class GetRealtimeDataRequest(
  public val contextId: GraphObjectId
)

/**
 * Response type for the [WebAudioDomain.getRealtimeData] command.
 */
@Serializable
public data class GetRealtimeDataResponse(
  public val realtimeData: ContextRealtimeData
)

/**
 * This domain allows inspection of Web Audio API.
 * https://webaudio.github.io/web-audio-api/
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio)
 */
@ExperimentalChromeApi
public class WebAudioDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out WebAudioEvent>> =
      mapOf(
      "WebAudio.contextCreated" to serializer<WebAudioEvent.ContextCreatedEvent>(),
      "WebAudio.contextWillBeDestroyed" to serializer<WebAudioEvent.ContextWillBeDestroyedEvent>(),
      "WebAudio.contextChanged" to serializer<WebAudioEvent.ContextChangedEvent>(),
      "WebAudio.audioListenerCreated" to serializer<WebAudioEvent.AudioListenerCreatedEvent>(),
      "WebAudio.audioListenerWillBeDestroyed" to
          serializer<WebAudioEvent.AudioListenerWillBeDestroyedEvent>(),
      "WebAudio.audioNodeCreated" to serializer<WebAudioEvent.AudioNodeCreatedEvent>(),
      "WebAudio.audioNodeWillBeDestroyed" to
          serializer<WebAudioEvent.AudioNodeWillBeDestroyedEvent>(),
      "WebAudio.audioParamCreated" to serializer<WebAudioEvent.AudioParamCreatedEvent>(),
      "WebAudio.audioParamWillBeDestroyed" to
          serializer<WebAudioEvent.AudioParamWillBeDestroyedEvent>(),
      "WebAudio.nodesConnected" to serializer<WebAudioEvent.NodesConnectedEvent>(),
      "WebAudio.nodesDisconnected" to serializer<WebAudioEvent.NodesDisconnectedEvent>(),
      "WebAudio.nodeParamConnected" to serializer<WebAudioEvent.NodeParamConnectedEvent>(),
      "WebAudio.nodeParamDisconnected" to serializer<WebAudioEvent.NodeParamDisconnectedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<WebAudioEvent> = session.events(deserializersByEventName)

  /**
   * Notifies that a new BaseAudioContext has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextCreated)
   */
  public fun contextCreated(): Flow<WebAudioEvent.ContextCreatedEvent> =
      session.events("WebAudio.contextCreated")

  /**
   * Notifies that an existing BaseAudioContext will be destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextWillBeDestroyed)
   */
  public fun contextWillBeDestroyed(): Flow<WebAudioEvent.ContextWillBeDestroyedEvent> =
      session.events("WebAudio.contextWillBeDestroyed")

  /**
   * Notifies that existing BaseAudioContext has changed some properties (id stays the same)..
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextChanged)
   */
  public fun contextChanged(): Flow<WebAudioEvent.ContextChangedEvent> =
      session.events("WebAudio.contextChanged")

  /**
   * Notifies that the construction of an AudioListener has finished.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerCreated)
   */
  public fun audioListenerCreated(): Flow<WebAudioEvent.AudioListenerCreatedEvent> =
      session.events("WebAudio.audioListenerCreated")

  /**
   * Notifies that a new AudioListener has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerWillBeDestroyed)
   */
  public fun audioListenerWillBeDestroyed(): Flow<WebAudioEvent.AudioListenerWillBeDestroyedEvent> =
      session.events("WebAudio.audioListenerWillBeDestroyed")

  /**
   * Notifies that a new AudioNode has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeCreated)
   */
  public fun audioNodeCreated(): Flow<WebAudioEvent.AudioNodeCreatedEvent> =
      session.events("WebAudio.audioNodeCreated")

  /**
   * Notifies that an existing AudioNode has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeWillBeDestroyed)
   */
  public fun audioNodeWillBeDestroyed(): Flow<WebAudioEvent.AudioNodeWillBeDestroyedEvent> =
      session.events("WebAudio.audioNodeWillBeDestroyed")

  /**
   * Notifies that a new AudioParam has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamCreated)
   */
  public fun audioParamCreated(): Flow<WebAudioEvent.AudioParamCreatedEvent> =
      session.events("WebAudio.audioParamCreated")

  /**
   * Notifies that an existing AudioParam has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamWillBeDestroyed)
   */
  public fun audioParamWillBeDestroyed(): Flow<WebAudioEvent.AudioParamWillBeDestroyedEvent> =
      session.events("WebAudio.audioParamWillBeDestroyed")

  /**
   * Notifies that two AudioNodes are connected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesConnected)
   */
  public fun nodesConnected(): Flow<WebAudioEvent.NodesConnectedEvent> =
      session.events("WebAudio.nodesConnected")

  /**
   * Notifies that AudioNodes are disconnected. The destination can be null, and it means all the
   * outgoing connections from the source are disconnected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesDisconnected)
   */
  public fun nodesDisconnected(): Flow<WebAudioEvent.NodesDisconnectedEvent> =
      session.events("WebAudio.nodesDisconnected")

  /**
   * Notifies that an AudioNode is connected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamConnected)
   */
  public fun nodeParamConnected(): Flow<WebAudioEvent.NodeParamConnectedEvent> =
      session.events("WebAudio.nodeParamConnected")

  /**
   * Notifies that an AudioNode is disconnected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamDisconnected)
   */
  public fun nodeParamDisconnected(): Flow<WebAudioEvent.NodeParamDisconnectedEvent> =
      session.events("WebAudio.nodeParamDisconnected")

  /**
   * Enables the WebAudio domain and starts sending context lifetime events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("WebAudio.enable", Unit)

  /**
   * Disables the WebAudio domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("WebAudio.disable", Unit)

  /**
   * Fetch the realtime data from the registered contexts.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-getRealtimeData)
   */
  public suspend fun getRealtimeData(input: GetRealtimeDataRequest): GetRealtimeDataResponse =
      session.request("WebAudio.getRealtimeData", input)
}
