package org.hildan.chrome.devtools.domains.input

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [InputDomain.dispatchKeyEvent] command.
 */
@Serializable
public data class DispatchKeyEventRequest(
  /**
   * Type of the key event.
   */
  public val type: String,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Time at which the event occurred.
   */
  public val timestamp: TimeSinceEpoch? = null,
  /**
   * Text as generated by processing a virtual key code with a keyboard layout. Not needed for
   * for `keyUp` and `rawKeyDown` events (default: "")
   */
  public val text: String? = null,
  /**
   * Text that would have been generated by the keyboard if no modifiers were pressed (except for
   * shift). Useful for shortcut (accelerator) key handling (default: "").
   */
  public val unmodifiedText: String? = null,
  /**
   * Unique key identifier (e.g., 'U+0041') (default: "").
   */
  public val keyIdentifier: String? = null,
  /**
   * Unique DOM defined string value for each physical key (e.g., 'KeyA') (default: "").
   */
  public val code: String? = null,
  /**
   * Unique DOM defined string value describing the meaning of the key in the context of active
   * modifiers, keyboard layout, etc (e.g., 'AltGr') (default: "").
   */
  public val key: String? = null,
  /**
   * Windows virtual key code (default: 0).
   */
  public val windowsVirtualKeyCode: Int? = null,
  /**
   * Native virtual key code (default: 0).
   */
  public val nativeVirtualKeyCode: Int? = null,
  /**
   * Whether the event was generated from auto repeat (default: false).
   */
  public val autoRepeat: Boolean? = null,
  /**
   * Whether the event was generated from the keypad (default: false).
   */
  public val isKeypad: Boolean? = null,
  /**
   * Whether the event was a system key event (default: false).
   */
  public val isSystemKey: Boolean? = null,
  /**
   * Whether the event was from the left or right side of the keyboard. 1=Left, 2=Right (default:
   * 0).
   */
  public val location: Int? = null,
  /**
   * Editing commands to send with the key event (e.g., 'selectAll') (default: []).
   * These are related to but not equal the command names used in `document.execCommand` and
   * NSStandardKeyBindingResponding.
   * See
   * https://source.chromium.org/chromium/chromium/src/+/master:third_party/blink/renderer/core/editing/commands/editor_command_names.h
   * for valid command names.
   */
  @ExperimentalChromeApi
  public val commands: List<String>? = null
)

/**
 * Request object containing input parameters for the [InputDomain.insertText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class InsertTextRequest(
  /**
   * The text to insert.
   */
  public val text: String
)

/**
 * Request object containing input parameters for the [InputDomain.dispatchMouseEvent] command.
 */
@Serializable
public data class DispatchMouseEventRequest(
  /**
   * Type of the mouse event.
   */
  public val type: String,
  /**
   * X coordinate of the event relative to the main frame's viewport in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0 refers to
   * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
   */
  public val y: Double,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Time at which the event occurred.
   */
  public val timestamp: TimeSinceEpoch? = null,
  /**
   * Mouse button (default: "none").
   */
  public val button: MouseButton? = null,
  /**
   * A number indicating which buttons are pressed on the mouse when a mouse event is triggered.
   * Left=1, Right=2, Middle=4, Back=8, Forward=16, None=0.
   */
  public val buttons: Int? = null,
  /**
   * Number of times the mouse button was clicked (default: 0).
   */
  public val clickCount: Int? = null,
  /**
   * The normalized pressure, which has a range of [0,1] (default: 0).
   */
  @ExperimentalChromeApi
  public val force: Double? = null,
  /**
   * The normalized tangential pressure, which has a range of [-1,1] (default: 0).
   */
  @ExperimentalChromeApi
  public val tangentialPressure: Double? = null,
  /**
   * The plane angle between the Y-Z plane and the plane containing both the stylus axis and the Y
   * axis, in degrees of the range [-90,90], a positive tiltX is to the right (default: 0).
   */
  @ExperimentalChromeApi
  public val tiltX: Int? = null,
  /**
   * The plane angle between the X-Z plane and the plane containing both the stylus axis and the X
   * axis, in degrees of the range [-90,90], a positive tiltY is towards the user (default: 0).
   */
  @ExperimentalChromeApi
  public val tiltY: Int? = null,
  /**
   * The clockwise rotation of a pen stylus around its own major axis, in degrees in the range
   * [0,359] (default: 0).
   */
  @ExperimentalChromeApi
  public val twist: Int? = null,
  /**
   * X delta in CSS pixels for mouse wheel event (default: 0).
   */
  public val deltaX: Double? = null,
  /**
   * Y delta in CSS pixels for mouse wheel event (default: 0).
   */
  public val deltaY: Double? = null,
  /**
   * Pointer type (default: "mouse").
   */
  public val pointerType: String? = null
)

/**
 * Request object containing input parameters for the [InputDomain.dispatchTouchEvent] command.
 */
@Serializable
public data class DispatchTouchEventRequest(
  /**
   * Type of the touch event. TouchEnd and TouchCancel must not contain any touch points, while
   * TouchStart and TouchMove must contains at least one.
   */
  public val type: String,
  /**
   * Active touch points on the touch device. One event per any changed point (compared to
   * previous touch event in a sequence) is generated, emulating pressing/moving/releasing points
   * one by one.
   */
  public val touchPoints: List<TouchPoint>,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Time at which the event occurred.
   */
  public val timestamp: TimeSinceEpoch? = null
)

/**
 * Request object containing input parameters for the [InputDomain.emulateTouchFromMouseEvent]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class EmulateTouchFromMouseEventRequest(
  /**
   * Type of the mouse event.
   */
  public val type: String,
  /**
   * X coordinate of the mouse pointer in DIP.
   */
  public val x: Int,
  /**
   * Y coordinate of the mouse pointer in DIP.
   */
  public val y: Int,
  /**
   * Mouse button. Only "none", "left", "right" are supported.
   */
  public val button: MouseButton,
  /**
   * Time at which the event occurred (default: current time).
   */
  public val timestamp: TimeSinceEpoch? = null,
  /**
   * X delta in DIP for mouse wheel event (default: 0).
   */
  public val deltaX: Double? = null,
  /**
   * Y delta in DIP for mouse wheel event (default: 0).
   */
  public val deltaY: Double? = null,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Number of times the mouse button was clicked (default: 0).
   */
  public val clickCount: Int? = null
)

/**
 * Request object containing input parameters for the [InputDomain.setIgnoreInputEvents] command.
 */
@Serializable
public data class SetIgnoreInputEventsRequest(
  /**
   * Ignores input events processing when set to true.
   */
  public val ignore: Boolean
)

/**
 * Request object containing input parameters for the [InputDomain.synthesizePinchGesture] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SynthesizePinchGestureRequest(
  /**
   * X coordinate of the start of the gesture in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the start of the gesture in CSS pixels.
   */
  public val y: Double,
  /**
   * Relative scale factor after zooming (>1.0 zooms in, <1.0 zooms out).
   */
  public val scaleFactor: Double,
  /**
   * Relative pointer speed in pixels per second (default: 800).
   */
  public val relativeSpeed: Int? = null,
  /**
   * Which type of input events to be generated (default: 'default', which queries the platform
   * for the preferred input type).
   */
  public val gestureSourceType: GestureSourceType? = null
)

/**
 * Request object containing input parameters for the [InputDomain.synthesizeScrollGesture] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SynthesizeScrollGestureRequest(
  /**
   * X coordinate of the start of the gesture in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the start of the gesture in CSS pixels.
   */
  public val y: Double,
  /**
   * The distance to scroll along the X axis (positive to scroll left).
   */
  public val xDistance: Double? = null,
  /**
   * The distance to scroll along the Y axis (positive to scroll up).
   */
  public val yDistance: Double? = null,
  /**
   * The number of additional pixels to scroll back along the X axis, in addition to the given
   * distance.
   */
  public val xOverscroll: Double? = null,
  /**
   * The number of additional pixels to scroll back along the Y axis, in addition to the given
   * distance.
   */
  public val yOverscroll: Double? = null,
  /**
   * Prevent fling (default: true).
   */
  public val preventFling: Boolean? = null,
  /**
   * Swipe speed in pixels per second (default: 800).
   */
  public val speed: Int? = null,
  /**
   * Which type of input events to be generated (default: 'default', which queries the platform
   * for the preferred input type).
   */
  public val gestureSourceType: GestureSourceType? = null,
  /**
   * The number of times to repeat the gesture (default: 0).
   */
  public val repeatCount: Int? = null,
  /**
   * The number of milliseconds delay between each repeat. (default: 250).
   */
  public val repeatDelayMs: Int? = null,
  /**
   * The name of the interaction markers to generate, if not empty (default: "").
   */
  public val interactionMarkerName: String? = null
)

/**
 * Request object containing input parameters for the [InputDomain.synthesizeTapGesture] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SynthesizeTapGestureRequest(
  /**
   * X coordinate of the start of the gesture in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the start of the gesture in CSS pixels.
   */
  public val y: Double,
  /**
   * Duration between touchdown and touchup events in ms (default: 50).
   */
  public val duration: Int? = null,
  /**
   * Number of times to perform the tap (e.g. 2 for double tap, default: 1).
   */
  public val tapCount: Int? = null,
  /**
   * Which type of input events to be generated (default: 'default', which queries the platform
   * for the preferred input type).
   */
  public val gestureSourceType: GestureSourceType? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input)
 */
public class InputDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Dispatches a key event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchKeyEvent)
   */
  public suspend fun dispatchKeyEvent(input: DispatchKeyEventRequest): Unit =
      session.request("Input.dispatchKeyEvent", input)

  /**
   * This method emulates inserting text that doesn't come from a key press,
   * for example an emoji keyboard or an IME.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-insertText)
   */
  @ExperimentalChromeApi
  public suspend fun insertText(input: InsertTextRequest): Unit =
      session.request("Input.insertText", input)

  /**
   * Dispatches a mouse event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchMouseEvent)
   */
  public suspend fun dispatchMouseEvent(input: DispatchMouseEventRequest): Unit =
      session.request("Input.dispatchMouseEvent", input)

  /**
   * Dispatches a touch event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchTouchEvent)
   */
  public suspend fun dispatchTouchEvent(input: DispatchTouchEventRequest): Unit =
      session.request("Input.dispatchTouchEvent", input)

  /**
   * Emulates touch event from the mouse event parameters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-emulateTouchFromMouseEvent)
   */
  @ExperimentalChromeApi
  public suspend fun emulateTouchFromMouseEvent(input: EmulateTouchFromMouseEventRequest): Unit =
      session.request("Input.emulateTouchFromMouseEvent", input)

  /**
   * Ignores input events (useful while auditing page).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-setIgnoreInputEvents)
   */
  public suspend fun setIgnoreInputEvents(input: SetIgnoreInputEventsRequest): Unit =
      session.request("Input.setIgnoreInputEvents", input)

  /**
   * Synthesizes a pinch gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizePinchGesture)
   */
  @ExperimentalChromeApi
  public suspend fun synthesizePinchGesture(input: SynthesizePinchGestureRequest): Unit =
      session.request("Input.synthesizePinchGesture", input)

  /**
   * Synthesizes a scroll gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizeScrollGesture)
   */
  @ExperimentalChromeApi
  public suspend fun synthesizeScrollGesture(input: SynthesizeScrollGestureRequest): Unit =
      session.request("Input.synthesizeScrollGesture", input)

  /**
   * Synthesizes a tap gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizeTapGesture)
   */
  @ExperimentalChromeApi
  public suspend fun synthesizeTapGesture(input: SynthesizeTapGestureRequest): Unit =
      session.request("Input.synthesizeTapGesture", input)
}
