@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.page

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.debugger.SearchMatch
import org.hildan.chrome.devtools.domains.dom.Rect
import org.hildan.chrome.devtools.domains.emulation.ScreenOrientation
import org.hildan.chrome.devtools.domains.io.StreamHandle
import org.hildan.chrome.devtools.domains.network.Cookie
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.page.events.PageEvent
import org.hildan.chrome.devtools.domains.runtime.ExecutionContextId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [PageDomain.addScriptToEvaluateOnLoad]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class AddScriptToEvaluateOnLoadRequest(
  public val scriptSource: String
)

/**
 * Response type for the [PageDomain.addScriptToEvaluateOnLoad] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class AddScriptToEvaluateOnLoadResponse(
  /**
   * Identifier of the added script.
   */
  public val identifier: ScriptIdentifier
)

/**
 * Request object containing input parameters for the [PageDomain.addScriptToEvaluateOnNewDocument]
 * command.
 */
@Serializable
public data class AddScriptToEvaluateOnNewDocumentRequest(
  public val source: String,
  /**
   * If specified, creates an isolated world with the given name and evaluates given script in it.
   * This world name will be used as the ExecutionContextDescription::name when the corresponding
   * event is emitted.
   */
  @ExperimentalChromeApi
  public val worldName: String? = null
)

/**
 * Response type for the [PageDomain.addScriptToEvaluateOnNewDocument] command.
 */
@Serializable
public data class AddScriptToEvaluateOnNewDocumentResponse(
  /**
   * Identifier of the added script.
   */
  public val identifier: ScriptIdentifier
)

/**
 * Request object containing input parameters for the [PageDomain.captureScreenshot] command.
 */
@Serializable
public data class CaptureScreenshotRequest(
  /**
   * Image compression format (defaults to png).
   */
  public val format: String? = null,
  /**
   * Compression quality from range [0..100] (jpeg only).
   */
  public val quality: Int? = null,
  /**
   * Capture the screenshot of a given region only.
   */
  public val clip: Viewport? = null,
  /**
   * Capture the screenshot from the surface, rather than the view. Defaults to true.
   */
  @ExperimentalChromeApi
  public val fromSurface: Boolean? = null,
  /**
   * Capture the screenshot beyond the viewport. Defaults to false.
   */
  @ExperimentalChromeApi
  public val captureBeyondViewport: Boolean? = null
)

/**
 * Response type for the [PageDomain.captureScreenshot] command.
 */
@Serializable
public data class CaptureScreenshotResponse(
  /**
   * Base64-encoded image data. (Encoded as a base64 string when passed over JSON)
   */
  public val `data`: String
)

/**
 * Request object containing input parameters for the [PageDomain.captureSnapshot] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CaptureSnapshotRequest(
  /**
   * Format (defaults to mhtml).
   */
  public val format: String? = null
)

/**
 * Response type for the [PageDomain.captureSnapshot] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CaptureSnapshotResponse(
  /**
   * Serialized page data.
   */
  public val `data`: String
)

/**
 * Request object containing input parameters for the [PageDomain.createIsolatedWorld] command.
 */
@Serializable
public data class CreateIsolatedWorldRequest(
  /**
   * Id of the frame in which the isolated world should be created.
   */
  public val frameId: FrameId,
  /**
   * An optional name which is reported in the Execution Context.
   */
  public val worldName: String? = null,
  /**
   * Whether or not universal access should be granted to the isolated world. This is a powerful
   * option, use with caution.
   */
  public val grantUniveralAccess: Boolean? = null
)

/**
 * Response type for the [PageDomain.createIsolatedWorld] command.
 */
@Serializable
public data class CreateIsolatedWorldResponse(
  /**
   * Execution context of the isolated world.
   */
  public val executionContextId: ExecutionContextId
)

/**
 * Request object containing input parameters for the [PageDomain.deleteCookie] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class DeleteCookieRequest(
  /**
   * Name of the cookie to remove.
   */
  public val cookieName: String,
  /**
   * URL to match cooke domain and path.
   */
  public val url: String
)

/**
 * Response type for the [PageDomain.getAppManifest] command.
 */
@Serializable
public data class GetAppManifestResponse(
  /**
   * Manifest location.
   */
  public val url: String,
  public val errors: List<AppManifestError>,
  /**
   * Manifest content.
   */
  public val `data`: String? = null,
  /**
   * Parsed manifest properties
   */
  @ExperimentalChromeApi
  public val parsed: AppManifestParsedProperties? = null
)

/**
 * Response type for the [PageDomain.getInstallabilityErrors] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetInstallabilityErrorsResponse(
  public val installabilityErrors: List<InstallabilityError>
)

/**
 * Response type for the [PageDomain.getManifestIcons] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetManifestIconsResponse(
  public val primaryIcon: String? = null
)

/**
 * Response type for the [PageDomain.getCookies] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class GetCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>
)

/**
 * Response type for the [PageDomain.getFrameTree] command.
 */
@Serializable
public data class GetFrameTreeResponse(
  /**
   * Present frame tree structure.
   */
  public val frameTree: FrameTree
)

/**
 * Response type for the [PageDomain.getLayoutMetrics] command.
 */
@Serializable
public data class GetLayoutMetricsResponse(
  /**
   * Metrics relating to the layout viewport.
   */
  public val layoutViewport: LayoutViewport,
  /**
   * Metrics relating to the visual viewport.
   */
  public val visualViewport: VisualViewport,
  /**
   * Size of scrollable area.
   */
  public val contentSize: Rect
)

/**
 * Response type for the [PageDomain.getNavigationHistory] command.
 */
@Serializable
public data class GetNavigationHistoryResponse(
  /**
   * Index of the current navigation history entry.
   */
  public val currentIndex: Int,
  /**
   * Array of navigation history entries.
   */
  public val entries: List<NavigationEntry>
)

/**
 * Request object containing input parameters for the [PageDomain.getResourceContent] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResourceContentRequest(
  /**
   * Frame id to get resource for.
   */
  public val frameId: FrameId,
  /**
   * URL of the resource to get content for.
   */
  public val url: String
)

/**
 * Response type for the [PageDomain.getResourceContent] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResourceContentResponse(
  /**
   * Resource content.
   */
  public val content: String,
  /**
   * True, if content was served as base64.
   */
  public val base64Encoded: Boolean
)

/**
 * Response type for the [PageDomain.getResourceTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResourceTreeResponse(
  /**
   * Present frame / resource tree structure.
   */
  public val frameTree: FrameResourceTree
)

/**
 * Request object containing input parameters for the [PageDomain.handleJavaScriptDialog] command.
 */
@Serializable
public data class HandleJavaScriptDialogRequest(
  /**
   * Whether to accept or dismiss the dialog.
   */
  public val accept: Boolean,
  /**
   * The text to enter into the dialog prompt before accepting. Used only if this is a prompt
   * dialog.
   */
  public val promptText: String? = null
)

/**
 * Request object containing input parameters for the [PageDomain.navigate] command.
 */
@Serializable
public data class NavigateRequest(
  /**
   * URL to navigate the page to.
   */
  public val url: String,
  /**
   * Referrer URL.
   */
  public val referrer: String? = null,
  /**
   * Intended transition type.
   */
  public val transitionType: TransitionType? = null,
  /**
   * Frame id to navigate, if not specified navigates the top frame.
   */
  public val frameId: FrameId? = null,
  /**
   * Referrer-policy used for the navigation.
   */
  @ExperimentalChromeApi
  public val referrerPolicy: ReferrerPolicy? = null
)

/**
 * Response type for the [PageDomain.navigate] command.
 */
@Serializable
public data class NavigateResponse(
  /**
   * Frame id that has navigated (or failed to navigate)
   */
  public val frameId: FrameId,
  /**
   * Loader identifier.
   */
  public val loaderId: LoaderId? = null,
  /**
   * User friendly error message, present if and only if navigation has failed.
   */
  public val errorText: String? = null
)

/**
 * Request object containing input parameters for the [PageDomain.navigateToHistoryEntry] command.
 */
@Serializable
public data class NavigateToHistoryEntryRequest(
  /**
   * Unique id of the entry to navigate to.
   */
  public val entryId: Int
)

/**
 * Request object containing input parameters for the [PageDomain.printToPDF] command.
 */
@Serializable
public data class PrintToPDFRequest(
  /**
   * Paper orientation. Defaults to false.
   */
  public val landscape: Boolean? = null,
  /**
   * Display header and footer. Defaults to false.
   */
  public val displayHeaderFooter: Boolean? = null,
  /**
   * Print background graphics. Defaults to false.
   */
  public val printBackground: Boolean? = null,
  /**
   * Scale of the webpage rendering. Defaults to 1.
   */
  public val scale: Double? = null,
  /**
   * Paper width in inches. Defaults to 8.5 inches.
   */
  public val paperWidth: Double? = null,
  /**
   * Paper height in inches. Defaults to 11 inches.
   */
  public val paperHeight: Double? = null,
  /**
   * Top margin in inches. Defaults to 1cm (~0.4 inches).
   */
  public val marginTop: Double? = null,
  /**
   * Bottom margin in inches. Defaults to 1cm (~0.4 inches).
   */
  public val marginBottom: Double? = null,
  /**
   * Left margin in inches. Defaults to 1cm (~0.4 inches).
   */
  public val marginLeft: Double? = null,
  /**
   * Right margin in inches. Defaults to 1cm (~0.4 inches).
   */
  public val marginRight: Double? = null,
  /**
   * Paper ranges to print, e.g., '1-5, 8, 11-13'. Defaults to the empty string, which means
   * print all pages.
   */
  public val pageRanges: String? = null,
  /**
   * Whether to silently ignore invalid but successfully parsed page ranges, such as '3-2'.
   * Defaults to false.
   */
  public val ignoreInvalidPageRanges: Boolean? = null,
  /**
   * HTML template for the print header. Should be valid HTML markup with following
   * classes used to inject printing values into them:
   * - `date`: formatted print date
   * - `title`: document title
   * - `url`: document location
   * - `pageNumber`: current page number
   * - `totalPages`: total pages in the document
   *
   * For example, `<span class=title></span>` would generate span containing the title.
   */
  public val headerTemplate: String? = null,
  /**
   * HTML template for the print footer. Should use the same format as the `headerTemplate`.
   */
  public val footerTemplate: String? = null,
  /**
   * Whether or not to prefer page size as defined by css. Defaults to false,
   * in which case the content will be scaled to fit the paper size.
   */
  public val preferCSSPageSize: Boolean? = null,
  /**
   * return as stream
   */
  @ExperimentalChromeApi
  public val transferMode: String? = null
)

/**
 * Response type for the [PageDomain.printToPDF] command.
 */
@Serializable
public data class PrintToPDFResponse(
  /**
   * Base64-encoded pdf data. Empty if |returnAsStream| is specified. (Encoded as a base64 string
   * when passed over JSON)
   */
  public val `data`: String,
  /**
   * A handle of the stream that holds resulting PDF data.
   */
  @ExperimentalChromeApi
  public val stream: StreamHandle? = null
)

/**
 * Request object containing input parameters for the [PageDomain.reload] command.
 */
@Serializable
public data class ReloadRequest(
  /**
   * If true, browser cache is ignored (as if the user pressed Shift+refresh).
   */
  public val ignoreCache: Boolean? = null,
  /**
   * If set, the script will be injected into all frames of the inspected page after reload.
   * Argument will be ignored if reloading dataURL origin.
   */
  public val scriptToEvaluateOnLoad: String? = null
)

/**
 * Request object containing input parameters for the [PageDomain.removeScriptToEvaluateOnLoad]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class RemoveScriptToEvaluateOnLoadRequest(
  public val identifier: ScriptIdentifier
)

/**
 * Request object containing input parameters for the
 * [PageDomain.removeScriptToEvaluateOnNewDocument] command.
 */
@Serializable
public data class RemoveScriptToEvaluateOnNewDocumentRequest(
  public val identifier: ScriptIdentifier
)

/**
 * Request object containing input parameters for the [PageDomain.screencastFrameAck] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ScreencastFrameAckRequest(
  /**
   * Frame number.
   */
  public val sessionId: Int
)

/**
 * Request object containing input parameters for the [PageDomain.searchInResource] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SearchInResourceRequest(
  /**
   * Frame id for resource to search in.
   */
  public val frameId: FrameId,
  /**
   * URL of the resource to search in.
   */
  public val url: String,
  /**
   * String to search for.
   */
  public val query: String,
  /**
   * If true, search is case sensitive.
   */
  public val caseSensitive: Boolean? = null,
  /**
   * If true, treats string parameter as regex.
   */
  public val isRegex: Boolean? = null
)

/**
 * Response type for the [PageDomain.searchInResource] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SearchInResourceResponse(
  /**
   * List of search matches.
   */
  public val result: List<SearchMatch>
)

/**
 * Request object containing input parameters for the [PageDomain.setAdBlockingEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAdBlockingEnabledRequest(
  /**
   * Whether to block ads.
   */
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [PageDomain.setBypassCSP] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBypassCSPRequest(
  /**
   * Whether to bypass page CSP.
   */
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [PageDomain.getPermissionsPolicyState]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetPermissionsPolicyStateRequest(
  public val frameId: FrameId
)

/**
 * Response type for the [PageDomain.getPermissionsPolicyState] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetPermissionsPolicyStateResponse(
  public val states: List<PermissionsPolicyFeatureState>
)

/**
 * Request object containing input parameters for the [PageDomain.setDeviceMetricsOverride] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetDeviceMetricsOverrideRequest(
  /**
   * Overriding width value in pixels (minimum 0, maximum 10000000). 0 disables the override.
   */
  public val width: Int,
  /**
   * Overriding height value in pixels (minimum 0, maximum 10000000). 0 disables the override.
   */
  public val height: Int,
  /**
   * Overriding device scale factor value. 0 disables the override.
   */
  public val deviceScaleFactor: Double,
  /**
   * Whether to emulate mobile device. This includes viewport meta tag, overlay scrollbars, text
   * autosizing and more.
   */
  public val mobile: Boolean,
  /**
   * Scale to apply to resulting view image.
   */
  public val scale: Double? = null,
  /**
   * Overriding screen width value in pixels (minimum 0, maximum 10000000).
   */
  public val screenWidth: Int? = null,
  /**
   * Overriding screen height value in pixels (minimum 0, maximum 10000000).
   */
  public val screenHeight: Int? = null,
  /**
   * Overriding view X position on screen in pixels (minimum 0, maximum 10000000).
   */
  public val positionX: Int? = null,
  /**
   * Overriding view Y position on screen in pixels (minimum 0, maximum 10000000).
   */
  public val positionY: Int? = null,
  /**
   * Do not set visible view size, rely upon explicit setVisibleSize call.
   */
  public val dontSetVisibleSize: Boolean? = null,
  /**
   * Screen orientation override.
   */
  public val screenOrientation: ScreenOrientation? = null,
  /**
   * The viewport dimensions and scale. If not set, the override is cleared.
   */
  public val viewport: Viewport? = null
)

/**
 * Request object containing input parameters for the [PageDomain.setDeviceOrientationOverride]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetDeviceOrientationOverrideRequest(
  /**
   * Mock alpha
   */
  public val alpha: Double,
  /**
   * Mock beta
   */
  public val beta: Double,
  /**
   * Mock gamma
   */
  public val gamma: Double
)

/**
 * Request object containing input parameters for the [PageDomain.setFontFamilies] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetFontFamiliesRequest(
  /**
   * Specifies font families to set. If a font family is not specified, it won't be changed.
   */
  public val fontFamilies: FontFamilies
)

/**
 * Request object containing input parameters for the [PageDomain.setFontSizes] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetFontSizesRequest(
  /**
   * Specifies font sizes to set. If a font size is not specified, it won't be changed.
   */
  public val fontSizes: FontSizes
)

/**
 * Request object containing input parameters for the [PageDomain.setDocumentContent] command.
 */
@Serializable
public data class SetDocumentContentRequest(
  /**
   * Frame id to set HTML for.
   */
  public val frameId: FrameId,
  /**
   * HTML content to set.
   */
  public val html: String
)

/**
 * Request object containing input parameters for the [PageDomain.setDownloadBehavior] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetDownloadBehaviorRequest(
  /**
   * Whether to allow all or deny all download requests, or use default Chrome behavior if
   * available (otherwise deny).
   */
  public val behavior: String,
  /**
   * The default path to save downloaded files to. This is requred if behavior is set to 'allow'
   */
  public val downloadPath: String? = null
)

/**
 * Request object containing input parameters for the [PageDomain.setGeolocationOverride] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class SetGeolocationOverrideRequest(
  /**
   * Mock latitude
   */
  public val latitude: Double? = null,
  /**
   * Mock longitude
   */
  public val longitude: Double? = null,
  /**
   * Mock accuracy
   */
  public val accuracy: Double? = null
)

/**
 * Request object containing input parameters for the [PageDomain.setLifecycleEventsEnabled]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetLifecycleEventsEnabledRequest(
  /**
   * If true, starts emitting lifecycle events.
   */
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [PageDomain.setTouchEmulationEnabled] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetTouchEmulationEnabledRequest(
  /**
   * Whether the touch event emulation should be enabled.
   */
  public val enabled: Boolean,
  /**
   * Touch/gesture events configuration. Default: current platform.
   */
  public val configuration: String? = null
)

/**
 * Request object containing input parameters for the [PageDomain.startScreencast] command.
 */
@Serializable
@ExperimentalChromeApi
public data class StartScreencastRequest(
  /**
   * Image compression format.
   */
  public val format: String? = null,
  /**
   * Compression quality from range [0..100].
   */
  public val quality: Int? = null,
  /**
   * Maximum screenshot width.
   */
  public val maxWidth: Int? = null,
  /**
   * Maximum screenshot height.
   */
  public val maxHeight: Int? = null,
  /**
   * Send every n-th frame.
   */
  public val everyNthFrame: Int? = null
)

/**
 * Request object containing input parameters for the [PageDomain.setWebLifecycleState] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetWebLifecycleStateRequest(
  /**
   * Target lifecycle state
   */
  public val state: String
)

/**
 * Request object containing input parameters for the [PageDomain.setProduceCompilationCache]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetProduceCompilationCacheRequest(
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [PageDomain.addCompilationCache] command.
 */
@Serializable
@ExperimentalChromeApi
public data class AddCompilationCacheRequest(
  public val url: String,
  /**
   * Base64-encoded data (Encoded as a base64 string when passed over JSON)
   */
  public val `data`: String
)

/**
 * Request object containing input parameters for the [PageDomain.generateTestReport] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GenerateTestReportRequest(
  /**
   * Message to be displayed in the report.
   */
  public val message: String,
  /**
   * Specifies the endpoint group to deliver the report to.
   */
  public val group: String? = null
)

/**
 * Request object containing input parameters for the [PageDomain.setInterceptFileChooserDialog]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInterceptFileChooserDialogRequest(
  public val enabled: Boolean
)

/**
 * Actions and events related to the inspected page belong to the page domain.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page)
 */
public class PageDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out PageEvent>> = mapOf(
      "Page.domContentEventFired" to serializer<PageEvent.DomContentEventFiredEvent>(),
      "Page.fileChooserOpened" to serializer<PageEvent.FileChooserOpenedEvent>(),
      "Page.frameAttached" to serializer<PageEvent.FrameAttachedEvent>(),
      "Page.frameClearedScheduledNavigation" to
          serializer<PageEvent.FrameClearedScheduledNavigationEvent>(),
      "Page.frameDetached" to serializer<PageEvent.FrameDetachedEvent>(),
      "Page.frameNavigated" to serializer<PageEvent.FrameNavigatedEvent>(),
      "Page.documentOpened" to serializer<PageEvent.DocumentOpenedEvent>(),
      "Page.frameResized" to serializer<PageEvent.FrameResizedEvent>(),
      "Page.frameRequestedNavigation" to serializer<PageEvent.FrameRequestedNavigationEvent>(),
      "Page.frameScheduledNavigation" to serializer<PageEvent.FrameScheduledNavigationEvent>(),
      "Page.frameStartedLoading" to serializer<PageEvent.FrameStartedLoadingEvent>(),
      "Page.frameStoppedLoading" to serializer<PageEvent.FrameStoppedLoadingEvent>(),
      "Page.downloadWillBegin" to serializer<PageEvent.DownloadWillBeginEvent>(),
      "Page.downloadProgress" to serializer<PageEvent.DownloadProgressEvent>(),
      "Page.interstitialHidden" to serializer<PageEvent.InterstitialHiddenEvent>(),
      "Page.interstitialShown" to serializer<PageEvent.InterstitialShownEvent>(),
      "Page.javascriptDialogClosed" to serializer<PageEvent.JavascriptDialogClosedEvent>(),
      "Page.javascriptDialogOpening" to serializer<PageEvent.JavascriptDialogOpeningEvent>(),
      "Page.lifecycleEvent" to serializer<PageEvent.LifecycleEventEvent>(),
      "Page.loadEventFired" to serializer<PageEvent.LoadEventFiredEvent>(),
      "Page.navigatedWithinDocument" to serializer<PageEvent.NavigatedWithinDocumentEvent>(),
      "Page.screencastFrame" to serializer<PageEvent.ScreencastFrameEvent>(),
      "Page.screencastVisibilityChanged" to
          serializer<PageEvent.ScreencastVisibilityChangedEvent>(),
      "Page.windowOpen" to serializer<PageEvent.WindowOpenEvent>(),
      "Page.compilationCacheProduced" to serializer<PageEvent.CompilationCacheProducedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<PageEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-domContentEventFired)
   */
  public fun domContentEventFired(): Flow<PageEvent.DomContentEventFiredEvent> =
      session.events("Page.domContentEventFired")

  /**
   * Emitted only when `page.interceptFileChooser` is enabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-fileChooserOpened)
   */
  public fun fileChooserOpened(): Flow<PageEvent.FileChooserOpenedEvent> =
      session.events("Page.fileChooserOpened")

  /**
   * Fired when frame has been attached to its parent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameAttached)
   */
  public fun frameAttached(): Flow<PageEvent.FrameAttachedEvent> =
      session.events("Page.frameAttached")

  /**
   * Fired when frame no longer has a scheduled navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameClearedScheduledNavigation)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public fun frameClearedScheduledNavigation(): Flow<PageEvent.FrameClearedScheduledNavigationEvent>
      = session.events("Page.frameClearedScheduledNavigation")

  /**
   * Fired when frame has been detached from its parent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameDetached)
   */
  public fun frameDetached(): Flow<PageEvent.FrameDetachedEvent> =
      session.events("Page.frameDetached")

  /**
   * Fired once navigation of the frame has completed. Frame is now associated with the new loader.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameNavigated)
   */
  public fun frameNavigated(): Flow<PageEvent.FrameNavigatedEvent> =
      session.events("Page.frameNavigated")

  /**
   * Fired when opening document to write to.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-documentOpened)
   */
  @ExperimentalChromeApi
  public fun documentOpened(): Flow<PageEvent.DocumentOpenedEvent> =
      session.events("Page.documentOpened")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameResized)
   */
  @ExperimentalChromeApi
  public fun frameResized(): Flow<PageEvent.FrameResizedEvent> = session.events("Page.frameResized")

  /**
   * Fired when a renderer-initiated navigation is requested.
   * Navigation may still be cancelled after the event is issued.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameRequestedNavigation)
   */
  @ExperimentalChromeApi
  public fun frameRequestedNavigation(): Flow<PageEvent.FrameRequestedNavigationEvent> =
      session.events("Page.frameRequestedNavigation")

  /**
   * Fired when frame schedules a potential navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameScheduledNavigation)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public fun frameScheduledNavigation(): Flow<PageEvent.FrameScheduledNavigationEvent> =
      session.events("Page.frameScheduledNavigation")

  /**
   * Fired when frame has started loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameStartedLoading)
   */
  @ExperimentalChromeApi
  public fun frameStartedLoading(): Flow<PageEvent.FrameStartedLoadingEvent> =
      session.events("Page.frameStartedLoading")

  /**
   * Fired when frame has stopped loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameStoppedLoading)
   */
  @ExperimentalChromeApi
  public fun frameStoppedLoading(): Flow<PageEvent.FrameStoppedLoadingEvent> =
      session.events("Page.frameStoppedLoading")

  /**
   * Fired when page is about to start a download.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-downloadWillBegin)
   */
  @ExperimentalChromeApi
  public fun downloadWillBegin(): Flow<PageEvent.DownloadWillBeginEvent> =
      session.events("Page.downloadWillBegin")

  /**
   * Fired when download makes progress. Last call has |done| == true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-downloadProgress)
   */
  @ExperimentalChromeApi
  public fun downloadProgress(): Flow<PageEvent.DownloadProgressEvent> =
      session.events("Page.downloadProgress")

  /**
   * Fired when interstitial page was hidden
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-interstitialHidden)
   */
  public fun interstitialHidden(): Flow<PageEvent.InterstitialHiddenEvent> =
      session.events("Page.interstitialHidden")

  /**
   * Fired when interstitial page was shown
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-interstitialShown)
   */
  public fun interstitialShown(): Flow<PageEvent.InterstitialShownEvent> =
      session.events("Page.interstitialShown")

  /**
   * Fired when a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload) has been
   * closed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-javascriptDialogClosed)
   */
  public fun javascriptDialogClosed(): Flow<PageEvent.JavascriptDialogClosedEvent> =
      session.events("Page.javascriptDialogClosed")

  /**
   * Fired when a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload) is about
   * to
   * open.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-javascriptDialogOpening)
   */
  public fun javascriptDialogOpening(): Flow<PageEvent.JavascriptDialogOpeningEvent> =
      session.events("Page.javascriptDialogOpening")

  /**
   * Fired for top level page lifecycle events such as navigation, load, paint, etc.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-lifecycleEvent)
   */
  public fun lifecycleEvent(): Flow<PageEvent.LifecycleEventEvent> =
      session.events("Page.lifecycleEvent")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-loadEventFired)
   */
  public fun loadEventFired(): Flow<PageEvent.LoadEventFiredEvent> =
      session.events("Page.loadEventFired")

  /**
   * Fired when same-document navigation happens, e.g. due to history API usage or anchor
   * navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-navigatedWithinDocument)
   */
  @ExperimentalChromeApi
  public fun navigatedWithinDocument(): Flow<PageEvent.NavigatedWithinDocumentEvent> =
      session.events("Page.navigatedWithinDocument")

  /**
   * Compressed image data requested by the `startScreencast`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-screencastFrame)
   */
  @ExperimentalChromeApi
  public fun screencastFrame(): Flow<PageEvent.ScreencastFrameEvent> =
      session.events("Page.screencastFrame")

  /**
   * Fired when the page with currently enabled screencast was shown or hidden `.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-screencastVisibilityChanged)
   */
  @ExperimentalChromeApi
  public fun screencastVisibilityChanged(): Flow<PageEvent.ScreencastVisibilityChangedEvent> =
      session.events("Page.screencastVisibilityChanged")

  /**
   * Fired when a new window is going to be opened, via window.open(), link click, form submission,
   * etc.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-windowOpen)
   */
  public fun windowOpen(): Flow<PageEvent.WindowOpenEvent> = session.events("Page.windowOpen")

  /**
   * Issued for every compilation cache generated. Is only available
   * if Page.setGenerateCompilationCache is enabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-compilationCacheProduced)
   */
  @ExperimentalChromeApi
  public fun compilationCacheProduced(): Flow<PageEvent.CompilationCacheProducedEvent> =
      session.events("Page.compilationCacheProduced")

  /**
   * Deprecated, please use addScriptToEvaluateOnNewDocument instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-addScriptToEvaluateOnLoad)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun addScriptToEvaluateOnLoad(input: AddScriptToEvaluateOnLoadRequest):
      AddScriptToEvaluateOnLoadResponse = session.request("Page.addScriptToEvaluateOnLoad", input)

  /**
   * Evaluates given script in every frame upon creation (before loading frame's scripts).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-addScriptToEvaluateOnNewDocument)
   */
  public suspend
      fun addScriptToEvaluateOnNewDocument(input: AddScriptToEvaluateOnNewDocumentRequest):
      AddScriptToEvaluateOnNewDocumentResponse =
      session.request("Page.addScriptToEvaluateOnNewDocument", input)

  /**
   * Brings page to front (activates tab).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-bringToFront)
   */
  public suspend fun bringToFront(): Unit = session.request("Page.bringToFront", Unit)

  /**
   * Capture page screenshot.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-captureScreenshot)
   */
  public suspend fun captureScreenshot(input: CaptureScreenshotRequest): CaptureScreenshotResponse =
      session.request("Page.captureScreenshot", input)

  /**
   * Returns a snapshot of the page as a string. For MHTML format, the serialization includes
   * iframes, shadow DOM, external resources, and element-inline styles.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-captureSnapshot)
   */
  @ExperimentalChromeApi
  public suspend fun captureSnapshot(input: CaptureSnapshotRequest): CaptureSnapshotResponse =
      session.request("Page.captureSnapshot", input)

  /**
   * Clears the overriden device metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-clearDeviceMetricsOverride)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun clearDeviceMetricsOverride(): Unit =
      session.request("Page.clearDeviceMetricsOverride", Unit)

  /**
   * Clears the overridden Device Orientation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-clearDeviceOrientationOverride)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun clearDeviceOrientationOverride(): Unit =
      session.request("Page.clearDeviceOrientationOverride", Unit)

  /**
   * Clears the overriden Geolocation Position and Error.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-clearGeolocationOverride)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun clearGeolocationOverride(): Unit =
      session.request("Page.clearGeolocationOverride", Unit)

  /**
   * Creates an isolated world for the given frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-createIsolatedWorld)
   */
  public suspend fun createIsolatedWorld(input: CreateIsolatedWorldRequest):
      CreateIsolatedWorldResponse = session.request("Page.createIsolatedWorld", input)

  /**
   * Deletes browser cookie with given name, domain and path.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-deleteCookie)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun deleteCookie(input: DeleteCookieRequest): Unit =
      session.request("Page.deleteCookie", input)

  /**
   * Disables page domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Page.disable", Unit)

  /**
   * Enables page domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Page.enable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getAppManifest)
   */
  public suspend fun getAppManifest(): GetAppManifestResponse =
      session.request("Page.getAppManifest", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getInstallabilityErrors)
   */
  @ExperimentalChromeApi
  public suspend fun getInstallabilityErrors(): GetInstallabilityErrorsResponse =
      session.request("Page.getInstallabilityErrors", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getManifestIcons)
   */
  @ExperimentalChromeApi
  public suspend fun getManifestIcons(): GetManifestIconsResponse =
      session.request("Page.getManifestIcons", Unit)

  /**
   * Returns all browser cookies. Depending on the backend support, will return detailed cookie
   * information in the `cookies` field.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getCookies)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun getCookies(): GetCookiesResponse = session.request("Page.getCookies", Unit)

  /**
   * Returns present frame tree structure.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getFrameTree)
   */
  public suspend fun getFrameTree(): GetFrameTreeResponse = session.request("Page.getFrameTree",
      Unit)

  /**
   * Returns metrics relating to the layouting of the page, such as viewport bounds/scale.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getLayoutMetrics)
   */
  public suspend fun getLayoutMetrics(): GetLayoutMetricsResponse =
      session.request("Page.getLayoutMetrics", Unit)

  /**
   * Returns navigation history for the current page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getNavigationHistory)
   */
  public suspend fun getNavigationHistory(): GetNavigationHistoryResponse =
      session.request("Page.getNavigationHistory", Unit)

  /**
   * Resets navigation history for the current page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-resetNavigationHistory)
   */
  public suspend fun resetNavigationHistory(): Unit = session.request("Page.resetNavigationHistory",
      Unit)

  /**
   * Returns content of the given resource.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getResourceContent)
   */
  @ExperimentalChromeApi
  public suspend fun getResourceContent(input: GetResourceContentRequest):
      GetResourceContentResponse = session.request("Page.getResourceContent", input)

  /**
   * Returns present frame / resource tree structure.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getResourceTree)
   */
  @ExperimentalChromeApi
  public suspend fun getResourceTree(): GetResourceTreeResponse =
      session.request("Page.getResourceTree", Unit)

  /**
   * Accepts or dismisses a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-handleJavaScriptDialog)
   */
  public suspend fun handleJavaScriptDialog(input: HandleJavaScriptDialogRequest): Unit =
      session.request("Page.handleJavaScriptDialog", input)

  /**
   * Navigates current page to the given URL.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-navigate)
   */
  public suspend fun navigate(input: NavigateRequest): NavigateResponse =
      session.request("Page.navigate", input)

  /**
   * Navigates current page to the given history entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-navigateToHistoryEntry)
   */
  public suspend fun navigateToHistoryEntry(input: NavigateToHistoryEntryRequest): Unit =
      session.request("Page.navigateToHistoryEntry", input)

  /**
   * Print page as PDF.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-printToPDF)
   */
  public suspend fun printToPDF(input: PrintToPDFRequest): PrintToPDFResponse =
      session.request("Page.printToPDF", input)

  /**
   * Reloads given page optionally ignoring the cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-reload)
   */
  public suspend fun reload(input: ReloadRequest): Unit = session.request("Page.reload", input)

  /**
   * Deprecated, please use removeScriptToEvaluateOnNewDocument instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-removeScriptToEvaluateOnLoad)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun removeScriptToEvaluateOnLoad(input: RemoveScriptToEvaluateOnLoadRequest): Unit
      = session.request("Page.removeScriptToEvaluateOnLoad", input)

  /**
   * Removes given script from the list.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-removeScriptToEvaluateOnNewDocument)
   */
  public suspend
      fun removeScriptToEvaluateOnNewDocument(input: RemoveScriptToEvaluateOnNewDocumentRequest):
      Unit = session.request("Page.removeScriptToEvaluateOnNewDocument", input)

  /**
   * Acknowledges that a screencast frame has been received by the frontend.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-screencastFrameAck)
   */
  @ExperimentalChromeApi
  public suspend fun screencastFrameAck(input: ScreencastFrameAckRequest): Unit =
      session.request("Page.screencastFrameAck", input)

  /**
   * Searches for given string in resource content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-searchInResource)
   */
  @ExperimentalChromeApi
  public suspend fun searchInResource(input: SearchInResourceRequest): SearchInResourceResponse =
      session.request("Page.searchInResource", input)

  /**
   * Enable Chrome's experimental ad filter on all sites.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setAdBlockingEnabled)
   */
  @ExperimentalChromeApi
  public suspend fun setAdBlockingEnabled(input: SetAdBlockingEnabledRequest): Unit =
      session.request("Page.setAdBlockingEnabled", input)

  /**
   * Enable page Content Security Policy by-passing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setBypassCSP)
   */
  @ExperimentalChromeApi
  public suspend fun setBypassCSP(input: SetBypassCSPRequest): Unit =
      session.request("Page.setBypassCSP", input)

  /**
   * Get Permissions Policy state on given frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-getPermissionsPolicyState)
   */
  @ExperimentalChromeApi
  public suspend fun getPermissionsPolicyState(input: GetPermissionsPolicyStateRequest):
      GetPermissionsPolicyStateResponse = session.request("Page.getPermissionsPolicyState", input)

  /**
   * Overrides the values of device screen dimensions (window.screen.width, window.screen.height,
   * window.innerWidth, window.innerHeight, and "device-width"/"device-height"-related CSS media
   * query results).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setDeviceMetricsOverride)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setDeviceMetricsOverride(input: SetDeviceMetricsOverrideRequest): Unit =
      session.request("Page.setDeviceMetricsOverride", input)

  /**
   * Overrides the Device Orientation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setDeviceOrientationOverride)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setDeviceOrientationOverride(input: SetDeviceOrientationOverrideRequest): Unit
      = session.request("Page.setDeviceOrientationOverride", input)

  /**
   * Set generic font families.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setFontFamilies)
   */
  @ExperimentalChromeApi
  public suspend fun setFontFamilies(input: SetFontFamiliesRequest): Unit =
      session.request("Page.setFontFamilies", input)

  /**
   * Set default font sizes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setFontSizes)
   */
  @ExperimentalChromeApi
  public suspend fun setFontSizes(input: SetFontSizesRequest): Unit =
      session.request("Page.setFontSizes", input)

  /**
   * Sets given markup as the document's HTML.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setDocumentContent)
   */
  public suspend fun setDocumentContent(input: SetDocumentContentRequest): Unit =
      session.request("Page.setDocumentContent", input)

  /**
   * Set the behavior when downloading a file.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setDownloadBehavior)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setDownloadBehavior(input: SetDownloadBehaviorRequest): Unit =
      session.request("Page.setDownloadBehavior", input)

  /**
   * Overrides the Geolocation Position or Error. Omitting any of the parameters emulates position
   * unavailable.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setGeolocationOverride)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun setGeolocationOverride(input: SetGeolocationOverrideRequest): Unit =
      session.request("Page.setGeolocationOverride", input)

  /**
   * Controls whether page will emit lifecycle events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setLifecycleEventsEnabled)
   */
  @ExperimentalChromeApi
  public suspend fun setLifecycleEventsEnabled(input: SetLifecycleEventsEnabledRequest): Unit =
      session.request("Page.setLifecycleEventsEnabled", input)

  /**
   * Toggles mouse event-based touch event emulation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setTouchEmulationEnabled)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setTouchEmulationEnabled(input: SetTouchEmulationEnabledRequest): Unit =
      session.request("Page.setTouchEmulationEnabled", input)

  /**
   * Starts sending each frame using the `screencastFrame` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-startScreencast)
   */
  @ExperimentalChromeApi
  public suspend fun startScreencast(input: StartScreencastRequest): Unit =
      session.request("Page.startScreencast", input)

  /**
   * Force the page stop all navigations and pending resource fetches.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-stopLoading)
   */
  public suspend fun stopLoading(): Unit = session.request("Page.stopLoading", Unit)

  /**
   * Crashes renderer on the IO thread, generates minidumps.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-crash)
   */
  @ExperimentalChromeApi
  public suspend fun crash(): Unit = session.request("Page.crash", Unit)

  /**
   * Tries to close page, running its beforeunload hooks, if any.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-close)
   */
  @ExperimentalChromeApi
  public suspend fun close(): Unit = session.request("Page.close", Unit)

  /**
   * Tries to update the web lifecycle state of the page.
   * It will transition the page to the given state according to:
   * https://github.com/WICG/web-lifecycle/
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setWebLifecycleState)
   */
  @ExperimentalChromeApi
  public suspend fun setWebLifecycleState(input: SetWebLifecycleStateRequest): Unit =
      session.request("Page.setWebLifecycleState", input)

  /**
   * Stops sending each frame in the `screencastFrame`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-stopScreencast)
   */
  @ExperimentalChromeApi
  public suspend fun stopScreencast(): Unit = session.request("Page.stopScreencast", Unit)

  /**
   * Forces compilation cache to be generated for every subresource script.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setProduceCompilationCache)
   */
  @ExperimentalChromeApi
  public suspend fun setProduceCompilationCache(input: SetProduceCompilationCacheRequest): Unit =
      session.request("Page.setProduceCompilationCache", input)

  /**
   * Seeds compilation cache for given url. Compilation cache does not survive
   * cross-process navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-addCompilationCache)
   */
  @ExperimentalChromeApi
  public suspend fun addCompilationCache(input: AddCompilationCacheRequest): Unit =
      session.request("Page.addCompilationCache", input)

  /**
   * Clears seeded compilation cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-clearCompilationCache)
   */
  @ExperimentalChromeApi
  public suspend fun clearCompilationCache(): Unit = session.request("Page.clearCompilationCache",
      Unit)

  /**
   * Generates a report for testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-generateTestReport)
   */
  @ExperimentalChromeApi
  public suspend fun generateTestReport(input: GenerateTestReportRequest): Unit =
      session.request("Page.generateTestReport", input)

  /**
   * Pauses page execution. Can be resumed using generic Runtime.runIfWaitingForDebugger.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-waitForDebugger)
   */
  @ExperimentalChromeApi
  public suspend fun waitForDebugger(): Unit = session.request("Page.waitForDebugger", Unit)

  /**
   * Intercept file chooser requests and transfer control to protocol clients.
   * When file chooser interception is enabled, native file chooser dialog is not shown.
   * Instead, a protocol event `Page.fileChooserOpened` is emitted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#method-setInterceptFileChooserDialog)
   */
  @ExperimentalChromeApi
  public suspend fun setInterceptFileChooserDialog(input: SetInterceptFileChooserDialogRequest):
      Unit = session.request("Page.setInterceptFileChooserDialog", input)
}
