@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.performancetimeline

import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.performancetimeline.events.PerformanceTimelineEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [PerformanceTimelineDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * The types of event to report, as specified in
   * https://w3c.github.io/performance-timeline/#dom-performanceentry-entrytype
   * The specified filter overrides any previous filters, passing empty
   * filter disables recording.
   * Note that not all types exposed to the web platform are currently supported.
   */
  public val eventTypes: List<String>
)

/**
 * Reporting of performance timeline events, as specified in
 * https://w3c.github.io/performance-timeline/#dom-performanceobserver.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline)
 */
@ExperimentalChromeApi
public class PerformanceTimelineDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out
      PerformanceTimelineEvent>> = mapOf(
      "PerformanceTimeline.timelineEventAdded" to
          serializer<PerformanceTimelineEvent.TimelineEventAddedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<PerformanceTimelineEvent> = session.events(deserializersByEventName)

  /**
   * Sent when a performance timeline event is added. See reportPerformanceTimeline method.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#event-timelineEventAdded)
   */
  public fun timelineEventAdded(): Flow<PerformanceTimelineEvent.TimelineEventAddedEvent> =
      session.events("PerformanceTimeline.timelineEventAdded")

  /**
   * Previously buffered events would be reported before method returns.
   * See also: timelineEventAdded
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#method-enable)
   */
  public suspend fun enable(input: EnableRequest): Unit =
      session.request("PerformanceTimeline.enable", input)
}
