@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.applicationcache

import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.applicationcache.events.ApplicationCacheEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the
 * [ApplicationCacheDomain.getApplicationCacheForFrame] command.
 */
@Serializable
public data class GetApplicationCacheForFrameRequest(
  /**
   * Identifier of the frame containing document whose application cache is retrieved.
   */
  public val frameId: FrameId
)

/**
 * Response type for the [ApplicationCacheDomain.getApplicationCacheForFrame] command.
 */
@Serializable
public data class GetApplicationCacheForFrameResponse(
  /**
   * Relevant application cache data for the document in given frame.
   */
  public val applicationCache: ApplicationCache
)

/**
 * Response type for the [ApplicationCacheDomain.getFramesWithManifests] command.
 */
@Serializable
public data class GetFramesWithManifestsResponse(
  /**
   * Array of frame identifiers with manifest urls for each frame containing a document
   * associated with some application cache.
   */
  public val frameIds: List<FrameWithManifest>
)

/**
 * Request object containing input parameters for the [ApplicationCacheDomain.getManifestForFrame]
 * command.
 */
@Serializable
public data class GetManifestForFrameRequest(
  /**
   * Identifier of the frame containing document whose manifest is retrieved.
   */
  public val frameId: FrameId
)

/**
 * Response type for the [ApplicationCacheDomain.getManifestForFrame] command.
 */
@Serializable
public data class GetManifestForFrameResponse(
  /**
   * Manifest URL for document in the given frame.
   */
  public val manifestURL: String
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache)
 */
@ExperimentalChromeApi
public class ApplicationCacheDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out
      ApplicationCacheEvent>> = mapOf(
      "ApplicationCache.applicationCacheStatusUpdated" to
          serializer<ApplicationCacheEvent.ApplicationCacheStatusUpdatedEvent>(),
      "ApplicationCache.networkStateUpdated" to
          serializer<ApplicationCacheEvent.NetworkStateUpdatedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<ApplicationCacheEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache/#event-applicationCacheStatusUpdated)
   */
  public fun applicationCacheStatusUpdated():
      Flow<ApplicationCacheEvent.ApplicationCacheStatusUpdatedEvent> =
      session.events("ApplicationCache.applicationCacheStatusUpdated")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache/#event-networkStateUpdated)
   */
  public fun networkStateUpdated(): Flow<ApplicationCacheEvent.NetworkStateUpdatedEvent> =
      session.events("ApplicationCache.networkStateUpdated")

  /**
   * Enables application cache domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("ApplicationCache.enable", Unit)

  /**
   * Returns relevant application cache data for the document in given frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache/#method-getApplicationCacheForFrame)
   */
  public suspend fun getApplicationCacheForFrame(input: GetApplicationCacheForFrameRequest):
      GetApplicationCacheForFrameResponse =
      session.request("ApplicationCache.getApplicationCacheForFrame", input)

  /**
   * Returns array of frame identifiers with manifest urls for each frame containing a document
   * associated with some application cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache/#method-getFramesWithManifests)
   */
  public suspend fun getFramesWithManifests(): GetFramesWithManifestsResponse =
      session.request("ApplicationCache.getFramesWithManifests", Unit)

  /**
   * Returns manifest URL for document in the given frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ApplicationCache/#method-getManifestForFrame)
   */
  public suspend fun getManifestForFrame(input: GetManifestForFrameRequest):
      GetManifestForFrameResponse = session.request("ApplicationCache.getManifestForFrame", input)
}
