@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.css

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.PseudoType
import org.hildan.chrome.devtools.domains.page.FrameId

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-StyleSheetId)
 */
public typealias StyleSheetId = String

/**
 * Stylesheet type: "injected" for stylesheets injected via extension, "user-agent" for user-agent
 * stylesheets, "inspector" for stylesheets created by the inspector (i.e. those holding the "via
 * inspector" rules), "regular" for regular stylesheets.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-StyleSheetOrigin)
 */
@Serializable
public enum class StyleSheetOrigin {
  injected,
  `user-agent`,
  inspector,
  regular,
}

/**
 * CSS rule collection for a single pseudo style.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-PseudoElementMatches)
 */
@Serializable
public data class PseudoElementMatches(
  /**
   * Pseudo element type.
   */
  public val pseudoType: PseudoType,
  /**
   * Matches of CSS rules applicable to the pseudo style.
   */
  public val matches: List<RuleMatch>
)

/**
 * Inherited CSS rule collection from ancestor node.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-InheritedStyleEntry)
 */
@Serializable
public data class InheritedStyleEntry(
  /**
   * The ancestor node's inline style, if any, in the style inheritance chain.
   */
  public val inlineStyle: CSSStyle? = null,
  /**
   * Matches of CSS rules matching the ancestor node in the style inheritance chain.
   */
  public val matchedCSSRules: List<RuleMatch>
)

/**
 * Match data for a CSS rule.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-RuleMatch)
 */
@Serializable
public data class RuleMatch(
  /**
   * CSS rule in the match.
   */
  public val rule: CSSRule,
  /**
   * Matching selector indices in the rule's selectorList selectors (0-based).
   */
  public val matchingSelectors: List<Int>
)

/**
 * Data for a simple selector (these are delimited by commas in a selector list).
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-Value)
 */
@Serializable
public data class Value(
  /**
   * Value text.
   */
  public val text: String,
  /**
   * Value range in the underlying resource (if available).
   */
  public val range: SourceRange? = null
)

/**
 * Selector list data.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-SelectorList)
 */
@Serializable
public data class SelectorList(
  /**
   * Selectors in the list.
   */
  public val selectors: List<Value>,
  /**
   * Rule selector text.
   */
  public val text: String
)

/**
 * CSS stylesheet metainformation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSStyleSheetHeader)
 */
@Serializable
public data class CSSStyleSheetHeader(
  /**
   * The stylesheet identifier.
   */
  public val styleSheetId: StyleSheetId,
  /**
   * Owner frame identifier.
   */
  public val frameId: FrameId,
  /**
   * Stylesheet resource URL.
   */
  public val sourceURL: String,
  /**
   * URL of source map associated with the stylesheet (if any).
   */
  public val sourceMapURL: String? = null,
  /**
   * Stylesheet origin.
   */
  public val origin: StyleSheetOrigin,
  /**
   * Stylesheet title.
   */
  public val title: String,
  /**
   * The backend id for the owner node of the stylesheet.
   */
  public val ownerNode: BackendNodeId? = null,
  /**
   * Denotes whether the stylesheet is disabled.
   */
  public val disabled: Boolean,
  /**
   * Whether the sourceURL field value comes from the sourceURL comment.
   */
  public val hasSourceURL: Boolean? = null,
  /**
   * Whether this stylesheet is created for STYLE tag by parser. This flag is not set for
   * document.written STYLE tags.
   */
  public val isInline: Boolean,
  /**
   * Whether this stylesheet is mutable. Inline stylesheets become mutable
   * after they have been modified via CSSOM API.
   * <link> element's stylesheets become mutable only if DevTools modifies them.
   * Constructed stylesheets (new CSSStyleSheet()) are mutable immediately after creation.
   */
  public val isMutable: Boolean,
  /**
   * Whether this stylesheet is a constructed stylesheet (created using new CSSStyleSheet()).
   */
  public val isConstructed: Boolean,
  /**
   * Line offset of the stylesheet within the resource (zero based).
   */
  public val startLine: Double,
  /**
   * Column offset of the stylesheet within the resource (zero based).
   */
  public val startColumn: Double,
  /**
   * Size of the content (in characters).
   */
  public val length: Double,
  /**
   * Line offset of the end of the stylesheet within the resource (zero based).
   */
  public val endLine: Double,
  /**
   * Column offset of the end of the stylesheet within the resource (zero based).
   */
  public val endColumn: Double
)

/**
 * CSS rule representation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSRule)
 */
@Serializable
public data class CSSRule(
  /**
   * The css style sheet identifier (absent for user agent stylesheet and user-specified
   * stylesheet rules) this rule came from.
   */
  public val styleSheetId: StyleSheetId? = null,
  /**
   * Rule selector data.
   */
  public val selectorList: SelectorList,
  /**
   * Parent stylesheet's origin.
   */
  public val origin: StyleSheetOrigin,
  /**
   * Associated style declaration.
   */
  public val style: CSSStyle,
  /**
   * Media list array (for rules involving media queries). The array enumerates media queries
   * starting with the innermost one, going outwards.
   */
  public val media: List<CSSMedia>? = null
)

/**
 * CSS coverage information.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-RuleUsage)
 */
@Serializable
public data class RuleUsage(
  /**
   * The css style sheet identifier (absent for user agent stylesheet and user-specified
   * stylesheet rules) this rule came from.
   */
  public val styleSheetId: StyleSheetId,
  /**
   * Offset of the start of the rule (including selector) from the beginning of the stylesheet.
   */
  public val startOffset: Double,
  /**
   * Offset of the end of the rule body from the beginning of the stylesheet.
   */
  public val endOffset: Double,
  /**
   * Indicates whether the rule was actually used by some element in the page.
   */
  public val used: Boolean
)

/**
 * Text range within a resource. All numbers are zero-based.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-SourceRange)
 */
@Serializable
public data class SourceRange(
  /**
   * Start line of range.
   */
  public val startLine: Int,
  /**
   * Start column of range (inclusive).
   */
  public val startColumn: Int,
  /**
   * End line of range
   */
  public val endLine: Int,
  /**
   * End column of range (exclusive).
   */
  public val endColumn: Int
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-ShorthandEntry)
 */
@Serializable
public data class ShorthandEntry(
  /**
   * Shorthand name.
   */
  public val name: String,
  /**
   * Shorthand value.
   */
  public val value: String,
  /**
   * Whether the property has "!important" annotation (implies `false` if absent).
   */
  public val important: Boolean? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSComputedStyleProperty)
 */
@Serializable
public data class CSSComputedStyleProperty(
  /**
   * Computed style property name.
   */
  public val name: String,
  /**
   * Computed style property value.
   */
  public val value: String
)

/**
 * CSS style representation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSStyle)
 */
@Serializable
public data class CSSStyle(
  /**
   * The css style sheet identifier (absent for user agent stylesheet and user-specified
   * stylesheet rules) this rule came from.
   */
  public val styleSheetId: StyleSheetId? = null,
  /**
   * CSS properties in the style.
   */
  public val cssProperties: List<CSSProperty>,
  /**
   * Computed values for all shorthands found in the style.
   */
  public val shorthandEntries: List<ShorthandEntry>,
  /**
   * Style declaration text (if available).
   */
  public val cssText: String? = null,
  /**
   * Style declaration range in the enclosing stylesheet (if available).
   */
  public val range: SourceRange? = null
)

/**
 * CSS property declaration data.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSProperty)
 */
@Serializable
public data class CSSProperty(
  /**
   * The property name.
   */
  public val name: String,
  /**
   * The property value.
   */
  public val value: String,
  /**
   * Whether the property has "!important" annotation (implies `false` if absent).
   */
  public val important: Boolean? = null,
  /**
   * Whether the property is implicit (implies `false` if absent).
   */
  public val implicit: Boolean? = null,
  /**
   * The full property text as specified in the style.
   */
  public val text: String? = null,
  /**
   * Whether the property is understood by the browser (implies `true` if absent).
   */
  public val parsedOk: Boolean? = null,
  /**
   * Whether the property is disabled by the user (present for source-based properties only).
   */
  public val disabled: Boolean? = null,
  /**
   * The entire property range in the enclosing style declaration (if available).
   */
  public val range: SourceRange? = null
)

/**
 * CSS media rule descriptor.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSMedia)
 */
@Serializable
public data class CSSMedia(
  /**
   * Media query text.
   */
  public val text: String,
  /**
   * Source of the media query: "mediaRule" if specified by a @media rule, "importRule" if
   * specified by an @import rule, "linkedSheet" if specified by a "media" attribute in a linked
   * stylesheet's LINK tag, "inlineSheet" if specified by a "media" attribute in an inline
   * stylesheet's STYLE tag.
   */
  public val source: String,
  /**
   * URL of the document containing the media query description.
   */
  public val sourceURL: String? = null,
  /**
   * The associated rule (@media or @import) header range in the enclosing stylesheet (if
   * available).
   */
  public val range: SourceRange? = null,
  /**
   * Identifier of the stylesheet containing this object (if exists).
   */
  public val styleSheetId: StyleSheetId? = null,
  /**
   * Array of media queries.
   */
  public val mediaList: List<MediaQuery>? = null
)

/**
 * Media query descriptor.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-MediaQuery)
 */
@Serializable
public data class MediaQuery(
  /**
   * Array of media query expressions.
   */
  public val expressions: List<MediaQueryExpression>,
  /**
   * Whether the media query condition is satisfied.
   */
  public val active: Boolean
)

/**
 * Media query expression descriptor.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-MediaQueryExpression)
 */
@Serializable
public data class MediaQueryExpression(
  /**
   * Media query expression value.
   */
  public val value: Double,
  /**
   * Media query expression units.
   */
  public val unit: String,
  /**
   * Media query expression feature.
   */
  public val feature: String,
  /**
   * The associated range of the value text in the enclosing stylesheet (if available).
   */
  public val valueRange: SourceRange? = null,
  /**
   * Computed length of media query expression (if applicable).
   */
  public val computedLength: Double? = null
)

/**
 * Information about amount of glyphs that were rendered with given font.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-PlatformFontUsage)
 */
@Serializable
public data class PlatformFontUsage(
  /**
   * Font's family name reported by platform.
   */
  public val familyName: String,
  /**
   * Indicates if the font was downloaded or resolved locally.
   */
  public val isCustomFont: Boolean,
  /**
   * Amount of glyphs that were rendered with this font.
   */
  public val glyphCount: Double
)

/**
 * Information about font variation axes for variable fonts
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-FontVariationAxis)
 */
@Serializable
public data class FontVariationAxis(
  /**
   * The font-variation-setting tag (a.k.a. "axis tag").
   */
  public val tag: String,
  /**
   * Human-readable variation name in the default language (normally, "en").
   */
  public val name: String,
  /**
   * The minimum value (inclusive) the font supports for this tag.
   */
  public val minValue: Double,
  /**
   * The maximum value (inclusive) the font supports for this tag.
   */
  public val maxValue: Double,
  /**
   * The default value.
   */
  public val defaultValue: Double
)

/**
 * Properties of a web font:
 * https://www.w3.org/TR/2008/REC-CSS2-20080411/fonts.html#font-descriptions
 * and additional information such as platformFontFamily and fontVariationAxes.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-FontFace)
 */
@Serializable
public data class FontFace(
  /**
   * The font-family.
   */
  public val fontFamily: String,
  /**
   * The font-style.
   */
  public val fontStyle: String,
  /**
   * The font-variant.
   */
  public val fontVariant: String,
  /**
   * The font-weight.
   */
  public val fontWeight: String,
  /**
   * The font-stretch.
   */
  public val fontStretch: String,
  /**
   * The unicode-range.
   */
  public val unicodeRange: String,
  /**
   * The src.
   */
  public val src: String,
  /**
   * The resolved platform font family
   */
  public val platformFontFamily: String,
  /**
   * Available variation settings (a.k.a. "axes").
   */
  public val fontVariationAxes: List<FontVariationAxis>? = null
)

/**
 * CSS keyframes rule representation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSKeyframesRule)
 */
@Serializable
public data class CSSKeyframesRule(
  /**
   * Animation name.
   */
  public val animationName: Value,
  /**
   * List of keyframes.
   */
  public val keyframes: List<CSSKeyframeRule>
)

/**
 * CSS keyframe rule representation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-CSSKeyframeRule)
 */
@Serializable
public data class CSSKeyframeRule(
  /**
   * The css style sheet identifier (absent for user agent stylesheet and user-specified
   * stylesheet rules) this rule came from.
   */
  public val styleSheetId: StyleSheetId? = null,
  /**
   * Parent stylesheet's origin.
   */
  public val origin: StyleSheetOrigin,
  /**
   * Associated key text.
   */
  public val keyText: Value,
  /**
   * Associated style declaration.
   */
  public val style: CSSStyle
)

/**
 * A descriptor of operation to mutate style declaration text.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#type-StyleDeclarationEdit)
 */
@Serializable
public data class StyleDeclarationEdit(
  /**
   * The css style sheet identifier.
   */
  public val styleSheetId: StyleSheetId,
  /**
   * The range of the style text in the enclosing stylesheet.
   */
  public val range: SourceRange,
  /**
   * New style text.
   */
  public val text: String
)
