@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.io

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession

/**
 * Request object containing input parameters for the [IODomain.close] command.
 */
@Serializable
public data class CloseRequest(
  /**
   * Handle of the stream to close.
   */
  public val handle: StreamHandle
)

/**
 * Request object containing input parameters for the [IODomain.read] command.
 */
@Serializable
public data class ReadRequest(
  /**
   * Handle of the stream to read.
   */
  public val handle: StreamHandle,
  /**
   * Seek to the specified offset before reading (if not specificed, proceed with offset
   * following the last read). Some types of streams may only support sequential reads.
   */
  public val offset: Int? = null,
  /**
   * Maximum number of bytes to read (left upon the agent discretion if not specified).
   */
  public val size: Int? = null
)

/**
 * Response type for the [IODomain.read] command.
 */
@Serializable
public data class ReadResponse(
  /**
   * Set if the data is base64-encoded
   */
  public val base64Encoded: Boolean? = null,
  /**
   * Data that were read.
   */
  public val `data`: String,
  /**
   * Set if the end-of-file condition occured while reading.
   */
  public val eof: Boolean
)

/**
 * Request object containing input parameters for the [IODomain.resolveBlob] command.
 */
@Serializable
public data class ResolveBlobRequest(
  /**
   * Object id of a Blob object wrapper.
   */
  public val objectId: RemoteObjectId
)

/**
 * Response type for the [IODomain.resolveBlob] command.
 */
@Serializable
public data class ResolveBlobResponse(
  /**
   * UUID of the specified Blob.
   */
  public val uuid: String
)

/**
 * Input/Output operations for streams produced by DevTools.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO)
 */
public class IODomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Close the stream, discard any temporary backing storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-close)
   */
  public suspend fun close(input: CloseRequest): Unit = session.request("IO.close", input)

  /**
   * Read a chunk of the stream
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-read)
   */
  public suspend fun read(input: ReadRequest): ReadResponse = session.request("IO.read", input)

  /**
   * Return UUID of Blob object specified by a remote object id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-resolveBlob)
   */
  public suspend fun resolveBlob(input: ResolveBlobRequest): ResolveBlobResponse =
      session.request("IO.resolveBlob", input)
}
