@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.runtime

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Unique script identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-ScriptId)
 */
public typealias ScriptId = String

/**
 * Unique object identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-RemoteObjectId)
 */
public typealias RemoteObjectId = String

/**
 * Primitive value which cannot be JSON-stringified. Includes values `-0`, `NaN`, `Infinity`,
 * `-Infinity`, and bigint literals.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-UnserializableValue)
 */
public typealias UnserializableValue = String

/**
 * Mirror object referencing original JavaScript object.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-RemoteObject)
 */
@Serializable
public data class RemoteObject(
  /**
   * Object type.
   */
  public val type: String,
  /**
   * Object subtype hint. Specified for `object` type values only.
   * NOTE: If you change anything here, make sure to also update
   * `subtype` in `ObjectPreview` and `PropertyPreview` below.
   */
  public val subtype: String? = null,
  /**
   * Object class (constructor) name. Specified for `object` type values only.
   */
  public val className: String? = null,
  /**
   * Remote object value in case of primitive values or JSON values (if it was requested).
   */
  public val value: JsonElement? = null,
  /**
   * Primitive value which can not be JSON-stringified does not have `value`, but gets this
   * property.
   */
  public val unserializableValue: UnserializableValue? = null,
  /**
   * String representation of the object.
   */
  public val description: String? = null,
  /**
   * Unique object identifier (for non-primitive values).
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Preview containing abbreviated property values. Specified for `object` type values only.
   */
  @ExperimentalChromeApi
  public val preview: ObjectPreview? = null,
  @ExperimentalChromeApi
  public val customPreview: CustomPreview? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-CustomPreview)
 */
@ExperimentalChromeApi
@Serializable
public data class CustomPreview(
  /**
   * The JSON-stringified result of formatter.header(object, config) call.
   * It contains json ML array that represents RemoteObject.
   */
  public val header: String,
  /**
   * If formatter returns true as a result of formatter.hasBody call then bodyGetterId will
   * contain RemoteObjectId for the function that returns result of formatter.body(object, config)
   * call.
   * The result value is json ML array.
   */
  public val bodyGetterId: RemoteObjectId? = null
)

/**
 * Object containing abbreviated remote object value.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-ObjectPreview)
 */
@ExperimentalChromeApi
@Serializable
public data class ObjectPreview(
  /**
   * Object type.
   */
  public val type: String,
  /**
   * Object subtype hint. Specified for `object` type values only.
   */
  public val subtype: String? = null,
  /**
   * String representation of the object.
   */
  public val description: String? = null,
  /**
   * True iff some of the properties or entries of the original object did not fit.
   */
  public val overflow: Boolean,
  /**
   * List of the properties.
   */
  public val properties: List<PropertyPreview>,
  /**
   * List of the entries. Specified for `map` and `set` subtype values only.
   */
  public val entries: List<EntryPreview>? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-PropertyPreview)
 */
@ExperimentalChromeApi
@Serializable
public data class PropertyPreview(
  /**
   * Property name.
   */
  public val name: String,
  /**
   * Object type. Accessor means that the property itself is an accessor property.
   */
  public val type: String,
  /**
   * User-friendly property value string.
   */
  public val value: String? = null,
  /**
   * Nested value preview.
   */
  public val valuePreview: ObjectPreview? = null,
  /**
   * Object subtype hint. Specified for `object` type values only.
   */
  public val subtype: String? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-EntryPreview)
 */
@ExperimentalChromeApi
@Serializable
public data class EntryPreview(
  /**
   * Preview of the key. Specified for map-like collection entries.
   */
  public val key: ObjectPreview? = null,
  /**
   * Preview of the value.
   */
  public val value: ObjectPreview
)

/**
 * Object property descriptor.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-PropertyDescriptor)
 */
@Serializable
public data class PropertyDescriptor(
  /**
   * Property name or symbol description.
   */
  public val name: String,
  /**
   * The value associated with the property.
   */
  public val value: RemoteObject? = null,
  /**
   * True if the value associated with the property may be changed (data descriptors only).
   */
  public val writable: Boolean? = null,
  /**
   * A function which serves as a getter for the property, or `undefined` if there is no getter
   * (accessor descriptors only).
   */
  public val `get`: RemoteObject? = null,
  /**
   * A function which serves as a setter for the property, or `undefined` if there is no setter
   * (accessor descriptors only).
   */
  public val `set`: RemoteObject? = null,
  /**
   * True if the type of this property descriptor may be changed and if the property may be
   * deleted from the corresponding object.
   */
  public val configurable: Boolean,
  /**
   * True if this property shows up during enumeration of the properties on the corresponding
   * object.
   */
  public val enumerable: Boolean,
  /**
   * True if the result was thrown during the evaluation.
   */
  public val wasThrown: Boolean? = null,
  /**
   * True if the property is owned for the object.
   */
  public val isOwn: Boolean? = null,
  /**
   * Property symbol object, if the property is of the `symbol` type.
   */
  public val symbol: RemoteObject? = null
)

/**
 * Object internal property descriptor. This property isn't normally visible in JavaScript code.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-InternalPropertyDescriptor)
 */
@Serializable
public data class InternalPropertyDescriptor(
  /**
   * Conventional property name.
   */
  public val name: String,
  /**
   * The value associated with the property.
   */
  public val value: RemoteObject? = null
)

/**
 * Object private field descriptor.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-PrivatePropertyDescriptor)
 */
@ExperimentalChromeApi
@Serializable
public data class PrivatePropertyDescriptor(
  /**
   * Private property name.
   */
  public val name: String,
  /**
   * The value associated with the private property.
   */
  public val value: RemoteObject? = null,
  /**
   * A function which serves as a getter for the private property,
   * or `undefined` if there is no getter (accessor descriptors only).
   */
  public val `get`: RemoteObject? = null,
  /**
   * A function which serves as a setter for the private property,
   * or `undefined` if there is no setter (accessor descriptors only).
   */
  public val `set`: RemoteObject? = null
)

/**
 * Represents function call argument. Either remote object id `objectId`, primitive `value`,
 * unserializable primitive value or neither of (for undefined) them should be specified.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-CallArgument)
 */
@Serializable
public data class CallArgument(
  /**
   * Primitive value or serializable javascript object.
   */
  public val value: JsonElement? = null,
  /**
   * Primitive value which can not be JSON-stringified.
   */
  public val unserializableValue: UnserializableValue? = null,
  /**
   * Remote object handle.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Id of an execution context.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-ExecutionContextId)
 */
public typealias ExecutionContextId = Int

/**
 * Description of an isolated world.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-ExecutionContextDescription)
 */
@Serializable
public data class ExecutionContextDescription(
  /**
   * Unique id of the execution context. It can be used to specify in which execution context
   * script evaluation should be performed.
   */
  public val id: ExecutionContextId,
  /**
   * Execution context origin.
   */
  public val origin: String,
  /**
   * Human readable name describing given context.
   */
  public val name: String,
  /**
   * A system-unique execution context identifier. Unlike the id, this is unique accross
   * multiple processes, so can be reliably used to identify specific context while backend
   * performs a cross-process navigation.
   */
  @ExperimentalChromeApi
  public val uniqueId: String? = null,
  /**
   * Embedder-specific auxiliary data.
   */
  public val auxData: JsonElement? = null
)

/**
 * Detailed information about exception (or error) that was thrown during script compilation or
 * execution.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-ExceptionDetails)
 */
@Serializable
public data class ExceptionDetails(
  /**
   * Exception id.
   */
  public val exceptionId: Int,
  /**
   * Exception text, which should be used together with exception object when available.
   */
  public val text: String,
  /**
   * Line number of the exception location (0-based).
   */
  public val lineNumber: Int,
  /**
   * Column number of the exception location (0-based).
   */
  public val columnNumber: Int,
  /**
   * Script ID of the exception location.
   */
  public val scriptId: ScriptId? = null,
  /**
   * URL of the exception location, to be used when the script was not reported.
   */
  public val url: String? = null,
  /**
   * JavaScript stack trace if available.
   */
  public val stackTrace: StackTrace? = null,
  /**
   * Exception object if available.
   */
  public val exception: RemoteObject? = null,
  /**
   * Identifier of the context where exception happened.
   */
  public val executionContextId: ExecutionContextId? = null
)

/**
 * Number of milliseconds since epoch.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-Timestamp)
 */
public typealias Timestamp = Double

/**
 * Number of milliseconds.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-TimeDelta)
 */
public typealias TimeDelta = Double

/**
 * Stack entry for runtime errors and assertions.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-CallFrame)
 */
@Serializable
public data class CallFrame(
  /**
   * JavaScript function name.
   */
  public val functionName: String,
  /**
   * JavaScript script id.
   */
  public val scriptId: ScriptId,
  /**
   * JavaScript script name or url.
   */
  public val url: String,
  /**
   * JavaScript script line number (0-based).
   */
  public val lineNumber: Int,
  /**
   * JavaScript script column number (0-based).
   */
  public val columnNumber: Int
)

/**
 * Call frames for assertions or error messages.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-StackTrace)
 */
@Serializable
public data class StackTrace(
  /**
   * String label of this stack trace. For async traces this may be a name of the function that
   * initiated the async call.
   */
  public val description: String? = null,
  /**
   * JavaScript function name.
   */
  public val callFrames: List<CallFrame>,
  /**
   * Asynchronous JavaScript stack trace that preceded this stack, if available.
   */
  public val parent: StackTrace? = null,
  /**
   * Asynchronous JavaScript stack trace that preceded this stack, if available.
   */
  @ExperimentalChromeApi
  public val parentId: StackTraceId? = null
)

/**
 * Unique identifier of current debugger.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-UniqueDebuggerId)
 */
@ExperimentalChromeApi
public typealias UniqueDebuggerId = String

/**
 * If `debuggerId` is set stack trace comes from another debugger and can be resolved there. This
 * allows to track cross-debugger calls. See `Runtime.StackTrace` and `Debugger.paused` for usages.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#type-StackTraceId)
 */
@ExperimentalChromeApi
@Serializable
public data class StackTraceId(
  public val id: String,
  public val debuggerId: UniqueDebuggerId? = null
)
