@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.target.events

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.target.SessionID
import org.hildan.chrome.devtools.domains.target.TargetID
import org.hildan.chrome.devtools.domains.target.TargetInfo
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class TargetEvent {
  /**
   * Issued when attached to target because of auto-attach or `attachToTarget` command.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-attachedToTarget)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class AttachedToTargetEvent(
    /**
     * Identifier assigned to the session used to send/receive messages.
     */
    public val sessionId: SessionID,
    public val targetInfo: TargetInfo,
    public val waitingForDebugger: Boolean
  ) : TargetEvent()

  /**
   * Issued when detached from target for any reason (including `detachFromTarget` command). Can be
   * issued multiple times per target if multiple sessions have been attached to it.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-detachedFromTarget)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class DetachedFromTargetEvent(
    /**
     * Detached session identifier.
     */
    public val sessionId: SessionID,
    /**
     * Deprecated.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public val targetId: TargetID? = null
  ) : TargetEvent()

  /**
   * Notifies about a new protocol message received from the session (as reported in
   * `attachedToTarget` event).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-receivedMessageFromTarget)
   */
  @Serializable
  public data class ReceivedMessageFromTargetEvent(
    /**
     * Identifier of a session which sends a message.
     */
    public val sessionId: SessionID,
    public val message: String,
    /**
     * Deprecated.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public val targetId: TargetID? = null
  ) : TargetEvent()

  /**
   * Issued when a possible inspection target is created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetCreated)
   */
  @Serializable
  public data class TargetCreatedEvent(
    public val targetInfo: TargetInfo
  ) : TargetEvent()

  /**
   * Issued when a target is destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetDestroyed)
   */
  @Serializable
  public data class TargetDestroyedEvent(
    public val targetId: TargetID
  ) : TargetEvent()

  /**
   * Issued when a target has crashed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetCrashed)
   */
  @Serializable
  public data class TargetCrashedEvent(
    public val targetId: TargetID,
    /**
     * Termination status type.
     */
    public val status: String,
    /**
     * Termination error code.
     */
    public val errorCode: Int
  ) : TargetEvent()

  /**
   * Issued when some information about a target has changed. This only happens between
   * `targetCreated` and `targetDestroyed`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetInfoChanged)
   */
  @Serializable
  public data class TargetInfoChangedEvent(
    public val targetInfo: TargetInfo
  ) : TargetEvent()
}
