@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.accessibility

import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import org.hildan.chrome.devtools.domains.dom.BackendNodeId

/**
 * Unique accessibility node identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXNodeId)
 */
public typealias AXNodeId = String

/**
 * Enum of possible property types.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXValueType)
 */
@Serializable
public enum class AXValueType {
  boolean,
  tristate,
  booleanOrUndefined,
  idref,
  idrefList,
  integer,
  node,
  nodeList,
  number,
  string,
  computedString,
  token,
  tokenList,
  domRelation,
  role,
  internalRole,
  valueUndefined,
}

/**
 * Enum of possible property sources.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXValueSourceType)
 */
@Serializable
public enum class AXValueSourceType {
  attribute,
  implicit,
  style,
  contents,
  placeholder,
  relatedElement,
}

/**
 * Enum of possible native property sources (as a subtype of a particular AXValueSourceType).
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXValueNativeSourceType)
 */
@Serializable
public enum class AXValueNativeSourceType {
  figcaption,
  label,
  labelfor,
  labelwrapped,
  legend,
  rubyannotation,
  tablecaption,
  title,
  other,
}

/**
 * A single source for a computed AX property.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXValueSource)
 */
@Serializable
public data class AXValueSource(
  /**
   * What type of source this is.
   */
  public val type: AXValueSourceType,
  /**
   * The value of this property source.
   */
  public val value: AXValue? = null,
  /**
   * The name of the relevant attribute, if any.
   */
  public val attribute: String? = null,
  /**
   * The value of the relevant attribute, if any.
   */
  public val attributeValue: AXValue? = null,
  /**
   * Whether this source is superseded by a higher priority source.
   */
  public val superseded: Boolean? = null,
  /**
   * The native markup source for this value, e.g. a <label> element.
   */
  public val nativeSource: AXValueNativeSourceType? = null,
  /**
   * The value, such as a node or node list, of the native source.
   */
  public val nativeSourceValue: AXValue? = null,
  /**
   * Whether the value for this property is invalid.
   */
  public val invalid: Boolean? = null,
  /**
   * Reason for the value being invalid, if it is.
   */
  public val invalidReason: String? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXRelatedNode)
 */
@Serializable
public data class AXRelatedNode(
  /**
   * The BackendNodeId of the related DOM node.
   */
  public val backendDOMNodeId: BackendNodeId,
  /**
   * The IDRef value provided, if any.
   */
  public val idref: String? = null,
  /**
   * The text alternative of this node in the current context.
   */
  public val text: String? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXProperty)
 */
@Serializable
public data class AXProperty(
  /**
   * The name of this property.
   */
  public val name: AXPropertyName,
  /**
   * The value of this property.
   */
  public val value: AXValue
)

/**
 * A single computed AX property.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXValue)
 */
@Serializable
public data class AXValue(
  /**
   * The type of this value.
   */
  public val type: AXValueType,
  /**
   * The computed value of this property.
   */
  public val value: JsonElement? = null,
  /**
   * One or more related nodes, if applicable.
   */
  public val relatedNodes: List<AXRelatedNode>? = null,
  /**
   * The sources which contributed to the computation of this property.
   */
  public val sources: List<AXValueSource>? = null
)

/**
 * Values of AXProperty name:
 * - from 'busy' to 'roledescription': states which apply to every AX node
 * - from 'live' to 'root': attributes which apply to nodes in live regions
 * - from 'autocomplete' to 'valuetext': attributes which apply to widgets
 * - from 'checked' to 'selected': states which apply to widgets
 * - from 'activedescendant' to 'owns' - relationships between elements other than
 * parent/child/sibling.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXPropertyName)
 */
@Serializable
public enum class AXPropertyName {
  busy,
  disabled,
  editable,
  focusable,
  focused,
  hidden,
  hiddenRoot,
  invalid,
  keyshortcuts,
  settable,
  roledescription,
  live,
  atomic,
  relevant,
  root,
  autocomplete,
  hasPopup,
  level,
  multiselectable,
  orientation,
  multiline,
  readonly,
  required,
  valuemin,
  valuemax,
  valuetext,
  checked,
  expanded,
  modal,
  pressed,
  selected,
  activedescendant,
  controls,
  describedby,
  details,
  errormessage,
  flowto,
  labelledby,
  owns,
}

/**
 * A node in the accessibility tree.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#type-AXNode)
 */
@Serializable
public data class AXNode(
  /**
   * Unique identifier for this node.
   */
  public val nodeId: AXNodeId,
  /**
   * Whether this node is ignored for accessibility
   */
  public val ignored: Boolean,
  /**
   * Collection of reasons why this node is hidden.
   */
  public val ignoredReasons: List<AXProperty>? = null,
  /**
   * This `Node`'s role, whether explicit or implicit.
   */
  public val role: AXValue? = null,
  /**
   * The accessible name for this `Node`.
   */
  public val name: AXValue? = null,
  /**
   * The accessible description for this `Node`.
   */
  public val description: AXValue? = null,
  /**
   * The value for this `Node`.
   */
  public val value: AXValue? = null,
  /**
   * All other properties
   */
  public val properties: List<AXProperty>? = null,
  /**
   * IDs for each of this node's child nodes.
   */
  public val childIds: List<AXNodeId>? = null,
  /**
   * The backend ID for the associated DOM node, if any.
   */
  public val backendDOMNodeId: BackendNodeId? = null
)
