@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.accessibility

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.accessibility.events.AccessibilityEvent
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [AccessibilityDomain.disable] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [AccessibilityDomain.enable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [AccessibilityDomain.getPartialAXTree]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetPartialAXTreeRequest(
  /**
   * Identifier of the node to get the partial accessibility tree for.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to get the partial accessibility tree for.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper to get the partial accessibility tree for.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Whether to fetch this nodes ancestors, siblings and children. Defaults to true.
   */
  public val fetchRelatives: Boolean? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node to get the partial accessibility tree for.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node to get the partial accessibility tree for.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper to get the partial accessibility tree for.
     */
    public var objectId: RemoteObjectId? = null

    /**
     * Whether to fetch this nodes ancestors, siblings and children. Defaults to true.
     */
    public var fetchRelatives: Boolean? = null

    public fun build(): GetPartialAXTreeRequest = GetPartialAXTreeRequest(nodeId, backendNodeId,
        objectId, fetchRelatives)
  }
}

/**
 * Response type for the [AccessibilityDomain.getPartialAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetPartialAXTreeResponse(
  /**
   * The `Accessibility.AXNode` for this DOM node, if it exists, plus its ancestors, siblings and
   * children, if requested.
   */
  public val nodes: List<AXNode>,
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.getFullAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFullAXTreeRequest(
  /**
   * The maximum depth at which descendants of the root node should be retrieved.
   * If omitted, the full tree is returned.
   */
  public val depth: Int? = null,
  /**
   * The frame for whose document the AX tree should be retrieved.
   * If omited, the root frame is used.
   */
  public val frameId: FrameId? = null,
) {
  public class Builder() {
    /**
     * The maximum depth at which descendants of the root node should be retrieved.
     * If omitted, the full tree is returned.
     */
    public var depth: Int? = null

    /**
     * The frame for whose document the AX tree should be retrieved.
     * If omited, the root frame is used.
     */
    public var frameId: FrameId? = null

    public fun build(): GetFullAXTreeRequest = GetFullAXTreeRequest(depth, frameId)
  }
}

/**
 * Response type for the [AccessibilityDomain.getFullAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFullAXTreeResponse(
  public val nodes: List<AXNode>,
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.getRootAXNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRootAXNodeRequest(
  /**
   * The frame in whose document the node resides.
   * If omitted, the root frame is used.
   */
  public val frameId: FrameId? = null,
) {
  public class Builder() {
    /**
     * The frame in whose document the node resides.
     * If omitted, the root frame is used.
     */
    public var frameId: FrameId? = null

    public fun build(): GetRootAXNodeRequest = GetRootAXNodeRequest(frameId)
  }
}

/**
 * Response type for the [AccessibilityDomain.getRootAXNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRootAXNodeResponse(
  public val node: AXNode,
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.getAXNodeAndAncestors]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetAXNodeAndAncestorsRequest(
  /**
   * Identifier of the node to get.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to get.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper to get.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node to get.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node to get.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper to get.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): GetAXNodeAndAncestorsRequest = GetAXNodeAndAncestorsRequest(nodeId,
        backendNodeId, objectId)
  }
}

/**
 * Response type for the [AccessibilityDomain.getAXNodeAndAncestors] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetAXNodeAndAncestorsResponse(
  public val nodes: List<AXNode>,
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.getChildAXNodes] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetChildAXNodesRequest(
  public val id: AXNodeId,
  /**
   * The frame in whose document the node resides.
   * If omitted, the root frame is used.
   */
  public val frameId: FrameId? = null,
) {
  public class Builder(
    public val id: AXNodeId,
  ) {
    /**
     * The frame in whose document the node resides.
     * If omitted, the root frame is used.
     */
    public var frameId: FrameId? = null

    public fun build(): GetChildAXNodesRequest = GetChildAXNodesRequest(id, frameId)
  }
}

/**
 * Response type for the [AccessibilityDomain.getChildAXNodes] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetChildAXNodesResponse(
  public val nodes: List<AXNode>,
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.queryAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class QueryAXTreeRequest(
  /**
   * Identifier of the node for the root to query.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node for the root to query.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper for the root to query.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Find nodes with this computed name.
   */
  public val accessibleName: String? = null,
  /**
   * Find nodes with this computed role.
   */
  public val role: String? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node for the root to query.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node for the root to query.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper for the root to query.
     */
    public var objectId: RemoteObjectId? = null

    /**
     * Find nodes with this computed name.
     */
    public var accessibleName: String? = null

    /**
     * Find nodes with this computed role.
     */
    public var role: String? = null

    public fun build(): QueryAXTreeRequest = QueryAXTreeRequest(nodeId, backendNodeId, objectId,
        accessibleName, role)
  }
}

/**
 * Response type for the [AccessibilityDomain.queryAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class QueryAXTreeResponse(
  /**
   * A list of `Accessibility.AXNode` matching the specified attributes,
   * including nodes that are ignored for accessibility.
   */
  public val nodes: List<AXNode>,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility)
 */
@ExperimentalChromeApi
public class AccessibilityDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out AccessibilityEvent>>
      = mapOf(
      "Accessibility.loadComplete" to serializer<AccessibilityEvent.LoadComplete>(),
      "Accessibility.nodesUpdated" to serializer<AccessibilityEvent.NodesUpdated>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<AccessibilityEvent> = session.typedEvents(deserializersByEventName)

  /**
   * The loadComplete event mirrors the load complete event sent by the browser to assistive
   * technology when the web page has finished loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#event-loadComplete)
   */
  @ExperimentalChromeApi
  public fun loadCompleteEvents(): Flow<AccessibilityEvent.LoadComplete> =
      session.typedEvents("Accessibility.loadComplete")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("loadCompleteEvents()"),
  )
  public fun loadComplete(): Flow<AccessibilityEvent.LoadComplete> = loadCompleteEvents()

  /**
   * The nodesUpdated event is sent every time a previously requested node has changed the in tree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#event-nodesUpdated)
   */
  @ExperimentalChromeApi
  public fun nodesUpdatedEvents(): Flow<AccessibilityEvent.NodesUpdated> =
      session.typedEvents("Accessibility.nodesUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("nodesUpdatedEvents()"),
  )
  public fun nodesUpdated(): Flow<AccessibilityEvent.NodesUpdated> = nodesUpdatedEvents()

  /**
   * Disables the accessibility domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Accessibility.disable", Unit)

  /**
   * Enables the accessibility domain which causes `AXNodeId`s to remain consistent between method
   * calls.
   * This turns on accessibility for the page, which can impact performance until accessibility is
   * disabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Accessibility.enable", Unit)

  /**
   * Fetches the accessibility node and partial accessibility tree for this DOM node, if it exists.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getPartialAXTree)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getPartialAXTree(input: GetPartialAXTreeRequest): GetPartialAXTreeResponse =
      session.request("Accessibility.getPartialAXTree", input)

  /**
   * Fetches the accessibility node and partial accessibility tree for this DOM node, if it exists.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getPartialAXTree)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun getPartialAXTree(optionalArgs: GetPartialAXTreeRequest.Builder.() -> Unit = {}):
      GetPartialAXTreeResponse {
    val builder = GetPartialAXTreeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getPartialAXTree(input)
  }

  /**
   * Fetches the entire accessibility tree for the root Document
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getFullAXTree)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getFullAXTree(input: GetFullAXTreeRequest): GetFullAXTreeResponse =
      session.request("Accessibility.getFullAXTree", input)

  /**
   * Fetches the entire accessibility tree for the root Document
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getFullAXTree)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getFullAXTree(optionalArgs: GetFullAXTreeRequest.Builder.() -> Unit =
      {}): GetFullAXTreeResponse {
    val builder = GetFullAXTreeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getFullAXTree(input)
  }

  /**
   * Fetches the root node.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getRootAXNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getRootAXNode(input: GetRootAXNodeRequest): GetRootAXNodeResponse =
      session.request("Accessibility.getRootAXNode", input)

  /**
   * Fetches the root node.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getRootAXNode)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getRootAXNode(optionalArgs: GetRootAXNodeRequest.Builder.() -> Unit =
      {}): GetRootAXNodeResponse {
    val builder = GetRootAXNodeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getRootAXNode(input)
  }

  /**
   * Fetches a node and all ancestors up to and including the root.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getAXNodeAndAncestors)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getAXNodeAndAncestors(input: GetAXNodeAndAncestorsRequest):
      GetAXNodeAndAncestorsResponse = session.request("Accessibility.getAXNodeAndAncestors", input)

  /**
   * Fetches a node and all ancestors up to and including the root.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getAXNodeAndAncestors)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun getAXNodeAndAncestors(optionalArgs: GetAXNodeAndAncestorsRequest.Builder.() -> Unit = {}):
      GetAXNodeAndAncestorsResponse {
    val builder = GetAXNodeAndAncestorsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getAXNodeAndAncestors(input)
  }

  /**
   * Fetches a particular accessibility node by AXNodeId.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getChildAXNodes)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getChildAXNodes(input: GetChildAXNodesRequest): GetChildAXNodesResponse =
      session.request("Accessibility.getChildAXNodes", input)

  /**
   * Fetches a particular accessibility node by AXNodeId.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getChildAXNodes)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getChildAXNodes(id: AXNodeId,
      optionalArgs: GetChildAXNodesRequest.Builder.() -> Unit = {}): GetChildAXNodesResponse {
    val builder = GetChildAXNodesRequest.Builder(id)
    val input = builder.apply(optionalArgs).build()
    return getChildAXNodes(input)
  }

  /**
   * Query a DOM node's accessibility subtree for accessible name and role.
   * This command computes the name and role for all nodes in the subtree, including those that are
   * ignored for accessibility, and returns those that mactch the specified name and role. If no DOM
   * node is specified, or the DOM node does not exist, the command returns an error. If neither
   * `accessibleName` or `role` is specified, it returns all the accessibility nodes in the subtree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-queryAXTree)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun queryAXTree(input: QueryAXTreeRequest): QueryAXTreeResponse =
      session.request("Accessibility.queryAXTree", input)

  /**
   * Query a DOM node's accessibility subtree for accessible name and role.
   * This command computes the name and role for all nodes in the subtree, including those that are
   * ignored for accessibility, and returns those that mactch the specified name and role. If no DOM
   * node is specified, or the DOM node does not exist, the command returns an error. If neither
   * `accessibleName` or `role` is specified, it returns all the accessibility nodes in the subtree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-queryAXTree)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun queryAXTree(optionalArgs: QueryAXTreeRequest.Builder.() -> Unit = {}):
      QueryAXTreeResponse {
    val builder = QueryAXTreeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return queryAXTree(input)
  }
}
