@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.animation

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.animation.events.AnimationEvent
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [AnimationDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [AnimationDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [AnimationDomain.getCurrentTime] command.
 */
@Serializable
public data class GetCurrentTimeRequest(
  /**
   * Id of animation.
   */
  public val id: String,
)

/**
 * Response type for the [AnimationDomain.getCurrentTime] command.
 */
@Serializable
public data class GetCurrentTimeResponse(
  /**
   * Current time of the page.
   */
  public val currentTime: Double,
)

/**
 * Response type for the [AnimationDomain.getPlaybackRate] command.
 */
@Serializable
public data class GetPlaybackRateResponse(
  /**
   * Playback rate for animations on page.
   */
  public val playbackRate: Double,
)

/**
 * Request object containing input parameters for the [AnimationDomain.releaseAnimations] command.
 */
@Serializable
public data class ReleaseAnimationsRequest(
  /**
   * List of animation ids to seek.
   */
  public val animations: List<String>,
)

/**
 * A dummy response object for the [AnimationDomain.releaseAnimations] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ReleaseAnimationsResponse

/**
 * Request object containing input parameters for the [AnimationDomain.resolveAnimation] command.
 */
@Serializable
public data class ResolveAnimationRequest(
  /**
   * Animation id.
   */
  public val animationId: String,
)

/**
 * Response type for the [AnimationDomain.resolveAnimation] command.
 */
@Serializable
public data class ResolveAnimationResponse(
  /**
   * Corresponding remote object.
   */
  public val remoteObject: RemoteObject,
)

/**
 * Request object containing input parameters for the [AnimationDomain.seekAnimations] command.
 */
@Serializable
public data class SeekAnimationsRequest(
  /**
   * List of animation ids to seek.
   */
  public val animations: List<String>,
  /**
   * Set the current time of each animation.
   */
  public val currentTime: Double,
)

/**
 * A dummy response object for the [AnimationDomain.seekAnimations] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SeekAnimationsResponse

/**
 * Request object containing input parameters for the [AnimationDomain.setPaused] command.
 */
@Serializable
public data class SetPausedRequest(
  /**
   * Animations to set the pause state of.
   */
  public val animations: List<String>,
  /**
   * Paused state to set to.
   */
  public val paused: Boolean,
)

/**
 * A dummy response object for the [AnimationDomain.setPaused] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object SetPausedResponse

/**
 * Request object containing input parameters for the [AnimationDomain.setPlaybackRate] command.
 */
@Serializable
public data class SetPlaybackRateRequest(
  /**
   * Playback rate for animations on page
   */
  public val playbackRate: Double,
)

/**
 * A dummy response object for the [AnimationDomain.setPlaybackRate] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetPlaybackRateResponse

/**
 * Request object containing input parameters for the [AnimationDomain.setTiming] command.
 */
@Serializable
public data class SetTimingRequest(
  /**
   * Animation id.
   */
  public val animationId: String,
  /**
   * Duration of the animation.
   */
  public val duration: Double,
  /**
   * Delay of the animation.
   */
  public val delay: Double,
)

/**
 * A dummy response object for the [AnimationDomain.setTiming] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object SetTimingResponse

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation)
 */
@ExperimentalChromeApi
public class AnimationDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out AnimationEvent>> =
      mapOf(
      "Animation.animationCanceled" to serializer<AnimationEvent.AnimationCanceled>(),
      "Animation.animationCreated" to serializer<AnimationEvent.AnimationCreated>(),
      "Animation.animationStarted" to serializer<AnimationEvent.AnimationStarted>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<AnimationEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Event for when an animation has been cancelled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#event-animationCanceled)
   */
  public fun animationCanceledEvents(): Flow<AnimationEvent.AnimationCanceled> =
      session.typedEvents("Animation.animationCanceled")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("animationCanceledEvents()"),
  )
  public fun animationCanceled(): Flow<AnimationEvent.AnimationCanceled> = animationCanceledEvents()

  /**
   * Event for each animation that has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#event-animationCreated)
   */
  public fun animationCreatedEvents(): Flow<AnimationEvent.AnimationCreated> =
      session.typedEvents("Animation.animationCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("animationCreatedEvents()"),
  )
  public fun animationCreated(): Flow<AnimationEvent.AnimationCreated> = animationCreatedEvents()

  /**
   * Event for animation that has been started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#event-animationStarted)
   */
  public fun animationStartedEvents(): Flow<AnimationEvent.AnimationStarted> =
      session.typedEvents("Animation.animationStarted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("animationStartedEvents()"),
  )
  public fun animationStarted(): Flow<AnimationEvent.AnimationStarted> = animationStartedEvents()

  /**
   * Disables animation domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Animation.disable", Unit)

  /**
   * Enables animation domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Animation.enable", Unit)

  /**
   * Returns the current time of the an animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-getCurrentTime)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getCurrentTime(input: GetCurrentTimeRequest): GetCurrentTimeResponse =
      session.request("Animation.getCurrentTime", input)

  /**
   * Returns the current time of the an animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-getCurrentTime)
   *
   * @param id Id of animation.
   */
  public suspend fun getCurrentTime(id: String): GetCurrentTimeResponse {
    val input = GetCurrentTimeRequest(id)
    return getCurrentTime(input)
  }

  /**
   * Gets the playback rate of the document timeline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-getPlaybackRate)
   */
  public suspend fun getPlaybackRate(): GetPlaybackRateResponse =
      session.request("Animation.getPlaybackRate", Unit)

  /**
   * Releases a set of animations to no longer be manipulated.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-releaseAnimations)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun releaseAnimations(input: ReleaseAnimationsRequest): ReleaseAnimationsResponse =
      session.request("Animation.releaseAnimations", input)

  /**
   * Releases a set of animations to no longer be manipulated.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-releaseAnimations)
   *
   * @param animations List of animation ids to seek.
   */
  public suspend fun releaseAnimations(animations: List<String>): ReleaseAnimationsResponse {
    val input = ReleaseAnimationsRequest(animations)
    return releaseAnimations(input)
  }

  /**
   * Gets the remote object of the Animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-resolveAnimation)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun resolveAnimation(input: ResolveAnimationRequest): ResolveAnimationResponse =
      session.request("Animation.resolveAnimation", input)

  /**
   * Gets the remote object of the Animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-resolveAnimation)
   *
   * @param animationId Animation id.
   */
  public suspend fun resolveAnimation(animationId: String): ResolveAnimationResponse {
    val input = ResolveAnimationRequest(animationId)
    return resolveAnimation(input)
  }

  /**
   * Seek a set of animations to a particular time within each animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-seekAnimations)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun seekAnimations(input: SeekAnimationsRequest): SeekAnimationsResponse =
      session.request("Animation.seekAnimations", input)

  /**
   * Seek a set of animations to a particular time within each animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-seekAnimations)
   *
   * @param animations List of animation ids to seek.
   * @param currentTime Set the current time of each animation.
   */
  public suspend fun seekAnimations(animations: List<String>, currentTime: Double):
      SeekAnimationsResponse {
    val input = SeekAnimationsRequest(animations, currentTime)
    return seekAnimations(input)
  }

  /**
   * Sets the paused state of a set of animations.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setPaused)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setPaused(input: SetPausedRequest): SetPausedResponse =
      session.request("Animation.setPaused", input)

  /**
   * Sets the paused state of a set of animations.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setPaused)
   *
   * @param animations Animations to set the pause state of.
   * @param paused Paused state to set to.
   */
  public suspend fun setPaused(animations: List<String>, paused: Boolean): SetPausedResponse {
    val input = SetPausedRequest(animations, paused)
    return setPaused(input)
  }

  /**
   * Sets the playback rate of the document timeline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setPlaybackRate)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setPlaybackRate(input: SetPlaybackRateRequest): SetPlaybackRateResponse =
      session.request("Animation.setPlaybackRate", input)

  /**
   * Sets the playback rate of the document timeline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setPlaybackRate)
   *
   * @param playbackRate Playback rate for animations on page
   */
  public suspend fun setPlaybackRate(playbackRate: Double): SetPlaybackRateResponse {
    val input = SetPlaybackRateRequest(playbackRate)
    return setPlaybackRate(input)
  }

  /**
   * Sets the timing of an animation node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setTiming)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setTiming(input: SetTimingRequest): SetTimingResponse =
      session.request("Animation.setTiming", input)

  /**
   * Sets the timing of an animation node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setTiming)
   *
   * @param animationId Animation id.
   * @param duration Duration of the animation.
   * @param delay Delay of the animation.
   */
  public suspend fun setTiming(
    animationId: String,
    duration: Double,
    delay: Double,
  ): SetTimingResponse {
    val input = SetTimingRequest(animationId, duration, delay)
    return setTiming(input)
  }
}
