@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.audits

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.audits.events.AuditsEvent
import org.hildan.chrome.devtools.domains.network.RequestId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [AuditsDomain.getEncodedResponse] command.
 */
@Serializable
public data class GetEncodedResponseRequest(
  /**
   * Identifier of the network request to get content for.
   */
  public val requestId: RequestId,
  /**
   * The encoding to use.
   */
  public val encoding: EncodedResponseEncoding,
  /**
   * The quality of the encoding (0-1). (defaults to 1)
   */
  public val quality: Double? = null,
  /**
   * Whether to only return the size information (defaults to false).
   */
  public val sizeOnly: Boolean? = null,
) {
  public class Builder(
    /**
     * Identifier of the network request to get content for.
     */
    public val requestId: RequestId,
    /**
     * The encoding to use.
     */
    public val encoding: EncodedResponseEncoding,
  ) {
    /**
     * The quality of the encoding (0-1). (defaults to 1)
     */
    public var quality: Double? = null

    /**
     * Whether to only return the size information (defaults to false).
     */
    public var sizeOnly: Boolean? = null

    public fun build(): GetEncodedResponseRequest = GetEncodedResponseRequest(requestId, encoding,
        quality, sizeOnly)
  }
}

/**
 * Response type for the [AuditsDomain.getEncodedResponse] command.
 */
@Serializable
public data class GetEncodedResponseResponse(
  /**
   * The encoded body as a base64 string. Omitted if sizeOnly is true. (Encoded as a base64 string
   * when passed over JSON)
   */
  public val body: String? = null,
  /**
   * Size before re-encoding.
   */
  public val originalSize: Int,
  /**
   * Size after re-encoding.
   */
  public val encodedSize: Int,
)

/**
 * A dummy response object for the [AuditsDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [AuditsDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [AuditsDomain.checkContrast] command.
 */
@Serializable
public data class CheckContrastRequest(
  /**
   * Whether to report WCAG AAA level issues. Default is false.
   */
  public val reportAAA: Boolean? = null,
) {
  public class Builder() {
    /**
     * Whether to report WCAG AAA level issues. Default is false.
     */
    public var reportAAA: Boolean? = null

    public fun build(): CheckContrastRequest = CheckContrastRequest(reportAAA)
  }
}

/**
 * A dummy response object for the [AuditsDomain.checkContrast] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object CheckContrastResponse

/**
 * Audits domain allows investigation of page violations and possible improvements.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits)
 */
@ExperimentalChromeApi
public class AuditsDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out AuditsEvent>> =
      mapOf(
      "Audits.issueAdded" to serializer<AuditsEvent.IssueAdded>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<AuditsEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#event-issueAdded)
   */
  public fun issueAddedEvents(): Flow<AuditsEvent.IssueAdded> =
      session.typedEvents("Audits.issueAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("issueAddedEvents()"),
  )
  public fun issueAdded(): Flow<AuditsEvent.IssueAdded> = issueAddedEvents()

  /**
   * Returns the response body and size if it were re-encoded with the specified settings. Only
   * applies to images.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-getEncodedResponse)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getEncodedResponse(input: GetEncodedResponseRequest):
      GetEncodedResponseResponse = session.request("Audits.getEncodedResponse", input)

  /**
   * Returns the response body and size if it were re-encoded with the specified settings. Only
   * applies to images.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-getEncodedResponse)
   *
   * @param requestId Identifier of the network request to get content for.
   * @param encoding The encoding to use.
   */
  @JvmOverloads
  public suspend inline fun getEncodedResponse(
    requestId: RequestId,
    encoding: EncodedResponseEncoding,
    optionalArgs: GetEncodedResponseRequest.Builder.() -> Unit = {},
  ): GetEncodedResponseResponse {
    val builder = GetEncodedResponseRequest.Builder(requestId, encoding)
    val input = builder.apply(optionalArgs).build()
    return getEncodedResponse(input)
  }

  /**
   * Disables issues domain, prevents further issues from being reported to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Audits.disable", Unit)

  /**
   * Enables issues domain, sends the issues collected so far to the client by means of the
   * `issueAdded` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Audits.enable", Unit)

  /**
   * Runs the contrast check for the target page. Found issues are reported
   * using Audits.issueAdded event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-checkContrast)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun checkContrast(input: CheckContrastRequest): CheckContrastResponse =
      session.request("Audits.checkContrast", input)

  /**
   * Runs the contrast check for the target page. Found issues are reported
   * using Audits.issueAdded event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-checkContrast)
   */
  @JvmOverloads
  public suspend inline fun checkContrast(optionalArgs: CheckContrastRequest.Builder.() -> Unit =
      {}): CheckContrastResponse {
    val builder = CheckContrastRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return checkContrast(input)
  }
}
