@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.audits

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.network.ClientSecurityState
import org.hildan.chrome.devtools.domains.network.CorsErrorStatus
import org.hildan.chrome.devtools.domains.network.IPAddressSpace
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.RequestId
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.runtime.ScriptId

/**
 * Information about a cookie that is affected by an inspector issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AffectedCookie)
 */
@Serializable
public data class AffectedCookie(
  /**
   * The following three properties uniquely identify a cookie
   */
  public val name: String,
  public val path: String,
  public val domain: String,
)

/**
 * Information about a request that is affected by an inspector issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AffectedRequest)
 */
@Serializable
public data class AffectedRequest(
  /**
   * The unique request id.
   */
  public val requestId: RequestId,
  public val url: String? = null,
)

/**
 * Information about the frame affected by an inspector issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AffectedFrame)
 */
@Serializable
public data class AffectedFrame(
  public val frameId: FrameId,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-CookieExclusionReason)
 */
@Serializable
public enum class CookieExclusionReason {
  ExcludeSameSiteUnspecifiedTreatedAsLax,
  ExcludeSameSiteNoneInsecure,
  ExcludeSameSiteLax,
  ExcludeSameSiteStrict,
  ExcludeInvalidSameParty,
  ExcludeSamePartyCrossPartyContext,
  ExcludeDomainNonASCII,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-CookieWarningReason)
 */
@Serializable
public enum class CookieWarningReason {
  WarnSameSiteUnspecifiedCrossSiteContext,
  WarnSameSiteNoneInsecure,
  WarnSameSiteUnspecifiedLaxAllowUnsafe,
  WarnSameSiteStrictLaxDowngradeStrict,
  WarnSameSiteStrictCrossDowngradeStrict,
  WarnSameSiteStrictCrossDowngradeLax,
  WarnSameSiteLaxCrossDowngradeStrict,
  WarnSameSiteLaxCrossDowngradeLax,
  WarnAttributeValueExceedsMaxSize,
  WarnDomainNonASCII,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-CookieOperation)
 */
@Serializable
public enum class CookieOperation {
  SetCookie,
  ReadCookie,
}

/**
 * This information is currently necessary, as the front-end has a difficult
 * time finding a specific cookie. With this, we can convey specific error
 * information without the cookie.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-CookieIssueDetails)
 */
@Serializable
public data class CookieIssueDetails(
  /**
   * If AffectedCookie is not set then rawCookieLine contains the raw
   * Set-Cookie header string. This hints at a problem where the
   * cookie line is syntactically or semantically malformed in a way
   * that no valid cookie could be created.
   */
  public val cookie: AffectedCookie? = null,
  public val rawCookieLine: String? = null,
  public val cookieWarningReasons: List<CookieWarningReason>,
  public val cookieExclusionReasons: List<CookieExclusionReason>,
  /**
   * Optionally identifies the site-for-cookies and the cookie url, which
   * may be used by the front-end as additional context.
   */
  public val operation: CookieOperation,
  public val siteForCookies: String? = null,
  public val cookieUrl: String? = null,
  public val request: AffectedRequest? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-MixedContentResolutionStatus)
 */
@Serializable
public enum class MixedContentResolutionStatus {
  MixedContentBlocked,
  MixedContentAutomaticallyUpgraded,
  MixedContentWarning,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-MixedContentResourceType)
 */
@Serializable
public enum class MixedContentResourceType {
  AttributionSrc,
  Audio,
  Beacon,
  CSPReport,
  Download,
  EventSource,
  Favicon,
  Font,
  Form,
  Frame,
  Image,
  Import,
  Manifest,
  Ping,
  PluginData,
  PluginResource,
  Prefetch,
  Resource,
  Script,
  ServiceWorker,
  SharedWorker,
  Stylesheet,
  Track,
  Video,
  Worker,
  XMLHttpRequest,
  XSLT,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-MixedContentIssueDetails)
 */
@Serializable
public data class MixedContentIssueDetails(
  /**
   * The type of resource causing the mixed content issue (css, js, iframe,
   * form,...). Marked as optional because it is mapped to from
   * blink::mojom::RequestContextType, which will be replaced
   * by network::mojom::RequestDestination
   */
  public val resourceType: MixedContentResourceType? = null,
  /**
   * The way the mixed content issue is being resolved.
   */
  public val resolutionStatus: MixedContentResolutionStatus,
  /**
   * The unsafe http url causing the mixed content issue.
   */
  public val insecureURL: String,
  /**
   * The url responsible for the call to an unsafe url.
   */
  public val mainResourceURL: String,
  /**
   * The mixed content request.
   * Does not always exist (e.g. for unsafe form submission urls).
   */
  public val request: AffectedRequest? = null,
  /**
   * Optional because not every mixed content issue is necessarily linked to a frame.
   */
  public val frame: AffectedFrame? = null,
)

/**
 * Enum indicating the reason a response has been blocked. These reasons are
 * refinements of the net error BLOCKED_BY_RESPONSE.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-BlockedByResponseReason)
 */
@Serializable
public enum class BlockedByResponseReason {
  CoepFrameResourceNeedsCoepHeader,
  CoopSandboxedIFrameCannotNavigateToCoopPage,
  CorpNotSameOrigin,
  CorpNotSameOriginAfterDefaultedToSameOriginByCoep,
  CorpNotSameSite,
}

/**
 * Details for a request that has been blocked with the BLOCKED_BY_RESPONSE
 * code. Currently only used for COEP/COOP, but may be extended to include
 * some CSP errors in the future.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-BlockedByResponseIssueDetails)
 */
@Serializable
public data class BlockedByResponseIssueDetails(
  public val request: AffectedRequest,
  public val parentFrame: AffectedFrame? = null,
  public val blockedFrame: AffectedFrame? = null,
  public val reason: BlockedByResponseReason,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-HeavyAdResolutionStatus)
 */
@Serializable
public enum class HeavyAdResolutionStatus {
  HeavyAdBlocked,
  HeavyAdWarning,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-HeavyAdReason)
 */
@Serializable
public enum class HeavyAdReason {
  NetworkTotalLimit,
  CpuTotalLimit,
  CpuPeakLimit,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-HeavyAdIssueDetails)
 */
@Serializable
public data class HeavyAdIssueDetails(
  /**
   * The resolution status, either blocking the content or warning.
   */
  public val resolution: HeavyAdResolutionStatus,
  /**
   * The reason the ad was blocked, total network or cpu or peak cpu.
   */
  public val reason: HeavyAdReason,
  /**
   * The frame that was blocked.
   */
  public val frame: AffectedFrame,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-ContentSecurityPolicyViolationType)
 */
@Serializable
public enum class ContentSecurityPolicyViolationType {
  kInlineViolation,
  kEvalViolation,
  kURLViolation,
  kTrustedTypesSinkViolation,
  kTrustedTypesPolicyViolation,
  kWasmEvalViolation,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SourceCodeLocation)
 */
@Serializable
public data class SourceCodeLocation(
  public val scriptId: ScriptId? = null,
  public val url: String,
  public val lineNumber: Int,
  public val columnNumber: Int,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-ContentSecurityPolicyIssueDetails)
 */
@Serializable
public data class ContentSecurityPolicyIssueDetails(
  /**
   * The url not included in allowed sources.
   */
  public val blockedURL: String? = null,
  /**
   * Specific directive that is violated, causing the CSP issue.
   */
  public val violatedDirective: String,
  public val isReportOnly: Boolean,
  public val contentSecurityPolicyViolationType: ContentSecurityPolicyViolationType,
  public val frameAncestor: AffectedFrame? = null,
  public val sourceCodeLocation: SourceCodeLocation? = null,
  public val violatingNodeId: BackendNodeId? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SharedArrayBufferIssueType)
 */
@Serializable
public enum class SharedArrayBufferIssueType {
  TransferIssue,
  CreationIssue,
}

/**
 * Details for a issue arising from an SAB being instantiated in, or
 * transferred to a context that is not cross-origin isolated.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SharedArrayBufferIssueDetails)
 */
@Serializable
public data class SharedArrayBufferIssueDetails(
  public val sourceCodeLocation: SourceCodeLocation,
  public val isWarning: Boolean,
  public val type: SharedArrayBufferIssueType,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-TwaQualityEnforcementViolationType)
 */
@Serializable
public enum class TwaQualityEnforcementViolationType {
  kHttpError,
  kUnavailableOffline,
  kDigitalAssetLinks,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-TrustedWebActivityIssueDetails)
 */
@Serializable
public data class TrustedWebActivityIssueDetails(
  /**
   * The url that triggers the violation.
   */
  public val url: String,
  public val violationType: TwaQualityEnforcementViolationType,
  public val httpStatusCode: Int? = null,
  /**
   * The package name of the Trusted Web Activity client app. This field is
   * only used when violation type is kDigitalAssetLinks.
   */
  public val packageName: String? = null,
  /**
   * The signature of the Trusted Web Activity client app. This field is only
   * used when violation type is kDigitalAssetLinks.
   */
  public val signature: String? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-LowTextContrastIssueDetails)
 */
@Serializable
public data class LowTextContrastIssueDetails(
  public val violatingNodeId: BackendNodeId,
  public val violatingNodeSelector: String,
  public val contrastRatio: Double,
  public val thresholdAA: Double,
  public val thresholdAAA: Double,
  public val fontSize: String,
  public val fontWeight: String,
)

/**
 * Details for a CORS related issue, e.g. a warning or error related to
 * CORS RFC1918 enforcement.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-CorsIssueDetails)
 */
@Serializable
public data class CorsIssueDetails(
  public val corsErrorStatus: CorsErrorStatus,
  public val isWarning: Boolean,
  public val request: AffectedRequest,
  public val location: SourceCodeLocation? = null,
  public val initiatorOrigin: String? = null,
  public val resourceIPAddressSpace: IPAddressSpace? = null,
  public val clientSecurityState: ClientSecurityState? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AttributionReportingIssueType)
 */
@Serializable
public enum class AttributionReportingIssueType {
  PermissionPolicyDisabled,
  PermissionPolicyNotDelegated,
  UntrustworthyReportingOrigin,
  InsecureContext,
  InvalidHeader,
  InvalidRegisterTriggerHeader,
  InvalidEligibleHeader,
  TooManyConcurrentRequests,
  SourceAndTriggerHeaders,
  SourceIgnored,
  TriggerIgnored,
}

/**
 * Details for issues around "Attribution Reporting API" usage.
 * Explainer: https://github.com/WICG/attribution-reporting-api
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AttributionReportingIssueDetails)
 */
@Serializable
public data class AttributionReportingIssueDetails(
  public val violationType: AttributionReportingIssueType,
  public val request: AffectedRequest? = null,
  public val violatingNodeId: BackendNodeId? = null,
  public val invalidParameter: String? = null,
)

/**
 * Details for issues about documents in Quirks Mode
 * or Limited Quirks Mode that affects page layouting.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-QuirksModeIssueDetails)
 */
@Serializable
public data class QuirksModeIssueDetails(
  /**
   * If false, it means the document's mode is "quirks"
   * instead of "limited-quirks".
   */
  public val isLimitedQuirksMode: Boolean,
  public val documentNodeId: BackendNodeId,
  public val url: String,
  public val frameId: FrameId,
  public val loaderId: LoaderId,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-NavigatorUserAgentIssueDetails)
 */
@Serializable
public data class NavigatorUserAgentIssueDetails(
  public val url: String,
  public val location: SourceCodeLocation? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-GenericIssueErrorType)
 */
@Serializable
public enum class GenericIssueErrorType {
  CrossOriginPortalPostMessageError,
}

/**
 * Depending on the concrete errorType, different properties are set.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-GenericIssueDetails)
 */
@Serializable
public data class GenericIssueDetails(
  /**
   * Issues with the same errorType are aggregated in the frontend.
   */
  public val errorType: GenericIssueErrorType,
  public val frameId: FrameId? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-DeprecationIssueType)
 */
@Serializable
public enum class DeprecationIssueType {
  AuthorizationCoveredByWildcard,
  CanRequestURLHTTPContainingNewline,
  ChromeLoadTimesConnectionInfo,
  ChromeLoadTimesFirstPaintAfterLoadTime,
  ChromeLoadTimesWasAlternateProtocolAvailable,
  CookieWithTruncatingChar,
  CrossOriginAccessBasedOnDocumentDomain,
  CrossOriginWindowAlert,
  CrossOriginWindowConfirm,
  CSSSelectorInternalMediaControlsOverlayCastButton,
  DeprecationExample,
  DocumentDomainSettingWithoutOriginAgentClusterHeader,
  EventPath,
  ExpectCTHeader,
  GeolocationInsecureOrigin,
  GeolocationInsecureOriginDeprecatedNotRemoved,
  GetUserMediaInsecureOrigin,
  HostCandidateAttributeGetter,
  IdentityInCanMakePaymentEvent,
  InsecurePrivateNetworkSubresourceRequest,
  LocalCSSFileExtensionRejected,
  MediaSourceAbortRemove,
  MediaSourceDurationTruncatingBuffered,
  NoSysexWebMIDIWithoutPermission,
  NotificationInsecureOrigin,
  NotificationPermissionRequestedIframe,
  ObsoleteWebRtcCipherSuite,
  OpenWebDatabaseInsecureContext,
  OverflowVisibleOnReplacedElement,
  PaymentInstruments,
  PaymentRequestCSPViolation,
  PersistentQuotaType,
  PictureSourceSrc,
  PrefixedCancelAnimationFrame,
  PrefixedRequestAnimationFrame,
  PrefixedStorageInfo,
  PrefixedVideoDisplayingFullscreen,
  PrefixedVideoEnterFullscreen,
  PrefixedVideoEnterFullScreen,
  PrefixedVideoExitFullscreen,
  PrefixedVideoExitFullScreen,
  PrefixedVideoSupportsFullscreen,
  RangeExpand,
  RequestedSubresourceWithEmbeddedCredentials,
  RTCConstraintEnableDtlsSrtpFalse,
  RTCConstraintEnableDtlsSrtpTrue,
  RTCPeerConnectionComplexPlanBSdpUsingDefaultSdpSemantics,
  RTCPeerConnectionSdpSemanticsPlanB,
  RtcpMuxPolicyNegotiate,
  SharedArrayBufferConstructedWithoutIsolation,
  TextToSpeech_DisallowedByAutoplay,
  V8SharedArrayBufferConstructedInExtensionWithoutIsolation,
  XHRJSONEncodingDetection,
  XMLHttpRequestSynchronousInNonWorkerOutsideBeforeUnload,
  XRSupportsSession,
}

/**
 * This issue tracks information needed to print a deprecation message.
 * https://source.chromium.org/chromium/chromium/src/+/main:third_party/blink/renderer/core/frame/third_party/blink/renderer/core/frame/deprecation/README.md
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-DeprecationIssueDetails)
 */
@Serializable
public data class DeprecationIssueDetails(
  public val affectedFrame: AffectedFrame? = null,
  public val sourceCodeLocation: SourceCodeLocation,
  public val type: DeprecationIssueType,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-ClientHintIssueReason)
 */
@Serializable
public enum class ClientHintIssueReason {
  MetaTagAllowListInvalidOrigin,
  MetaTagModifiedHTML,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-FederatedAuthRequestIssueDetails)
 */
@Serializable
public data class FederatedAuthRequestIssueDetails(
  public val federatedAuthRequestIssueReason: FederatedAuthRequestIssueReason,
)

/**
 * Represents the failure reason when a federated authentication reason fails.
 * Should be updated alongside RequestIdTokenStatus in
 * third_party/blink/public/mojom/devtools/inspector_issue.mojom to include
 * all cases except for success.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-FederatedAuthRequestIssueReason)
 */
@Serializable
public enum class FederatedAuthRequestIssueReason {
  ShouldEmbargo,
  TooManyRequests,
  ManifestListHttpNotFound,
  ManifestListNoResponse,
  ManifestListInvalidResponse,
  ManifestNotInManifestList,
  ManifestListTooBig,
  ManifestHttpNotFound,
  ManifestNoResponse,
  ManifestInvalidResponse,
  ClientMetadataHttpNotFound,
  ClientMetadataNoResponse,
  ClientMetadataInvalidResponse,
  DisabledInSettings,
  ErrorFetchingSignin,
  InvalidSigninResponse,
  AccountsHttpNotFound,
  AccountsNoResponse,
  AccountsInvalidResponse,
  IdTokenHttpNotFound,
  IdTokenNoResponse,
  IdTokenInvalidResponse,
  IdTokenInvalidRequest,
  ErrorIdToken,
  Canceled,
  RpPageNotVisible,
}

/**
 * This issue tracks client hints related issues. It's used to deprecate old
 * features, encourage the use of new ones, and provide general guidance.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-ClientHintIssueDetails)
 */
@Serializable
public data class ClientHintIssueDetails(
  public val sourceCodeLocation: SourceCodeLocation,
  public val clientHintIssueReason: ClientHintIssueReason,
)

/**
 * A unique identifier for the type of issue. Each type may use one of the
 * optional fields in InspectorIssueDetails to convey more specific
 * information about the kind of issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-InspectorIssueCode)
 */
@Serializable
public enum class InspectorIssueCode {
  CookieIssue,
  MixedContentIssue,
  BlockedByResponseIssue,
  HeavyAdIssue,
  ContentSecurityPolicyIssue,
  SharedArrayBufferIssue,
  TrustedWebActivityIssue,
  LowTextContrastIssue,
  CorsIssue,
  AttributionReportingIssue,
  QuirksModeIssue,
  NavigatorUserAgentIssue,
  GenericIssue,
  DeprecationIssue,
  ClientHintIssue,
  FederatedAuthRequestIssue,
}

/**
 * This struct holds a list of optional fields with additional information
 * specific to the kind of issue. When adding a new issue code, please also
 * add a new optional field to this type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-InspectorIssueDetails)
 */
@Serializable
public data class InspectorIssueDetails(
  public val cookieIssueDetails: CookieIssueDetails? = null,
  public val mixedContentIssueDetails: MixedContentIssueDetails? = null,
  public val blockedByResponseIssueDetails: BlockedByResponseIssueDetails? = null,
  public val heavyAdIssueDetails: HeavyAdIssueDetails? = null,
  public val contentSecurityPolicyIssueDetails: ContentSecurityPolicyIssueDetails? = null,
  public val sharedArrayBufferIssueDetails: SharedArrayBufferIssueDetails? = null,
  public val twaQualityEnforcementDetails: TrustedWebActivityIssueDetails? = null,
  public val lowTextContrastIssueDetails: LowTextContrastIssueDetails? = null,
  public val corsIssueDetails: CorsIssueDetails? = null,
  public val attributionReportingIssueDetails: AttributionReportingIssueDetails? = null,
  public val quirksModeIssueDetails: QuirksModeIssueDetails? = null,
  public val navigatorUserAgentIssueDetails: NavigatorUserAgentIssueDetails? = null,
  public val genericIssueDetails: GenericIssueDetails? = null,
  public val deprecationIssueDetails: DeprecationIssueDetails? = null,
  public val clientHintIssueDetails: ClientHintIssueDetails? = null,
  public val federatedAuthRequestIssueDetails: FederatedAuthRequestIssueDetails? = null,
)

/**
 * A unique id for a DevTools inspector issue. Allows other entities (e.g.
 * exceptions, CDP message, console messages, etc.) to reference an issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-IssueId)
 */
public typealias IssueId = String

/**
 * An inspector issue reported from the back-end.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-InspectorIssue)
 */
@Serializable
public data class InspectorIssue(
  public val code: InspectorIssueCode,
  public val details: InspectorIssueDetails,
  /**
   * A unique id for this issue. May be omitted if no other entity (e.g.
   * exception, CDP message, etc.) is referencing this issue.
   */
  public val issueId: IssueId? = null,
)

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'encoding' of the command 'Audits.getEncodedResponse'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-EncodedResponseEncoding)
 */
@Serializable
public enum class EncodedResponseEncoding {
  webp,
  jpeg,
  png,
}
