@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.backgroundservice

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.backgroundservice.events.BackgroundServiceEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.startObserving]
 * command.
 */
@Serializable
public data class StartObservingRequest(
  public val service: ServiceName,
)

/**
 * A dummy response object for the [BackgroundServiceDomain.startObserving] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object StartObservingResponse

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.stopObserving]
 * command.
 */
@Serializable
public data class StopObservingRequest(
  public val service: ServiceName,
)

/**
 * A dummy response object for the [BackgroundServiceDomain.stopObserving] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object StopObservingResponse

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.setRecording]
 * command.
 */
@Serializable
public data class SetRecordingRequest(
  public val shouldRecord: Boolean,
  public val service: ServiceName,
)

/**
 * A dummy response object for the [BackgroundServiceDomain.setRecording] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetRecordingResponse

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.clearEvents] command.
 */
@Serializable
public data class ClearEventsRequest(
  public val service: ServiceName,
)

/**
 * A dummy response object for the [BackgroundServiceDomain.clearEvents] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object ClearEventsResponse

/**
 * Defines events for background web platform features.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService)
 */
@ExperimentalChromeApi
public class BackgroundServiceDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName:
      Map<String, DeserializationStrategy<out BackgroundServiceEvent>> = mapOf(
      "BackgroundService.recordingStateChanged" to
          serializer<BackgroundServiceEvent.RecordingStateChanged>(),
      "BackgroundService.backgroundServiceEventReceived" to
          serializer<BackgroundServiceEvent.BackgroundServiceEventReceived>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<BackgroundServiceEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Called when the recording state for the service has been updated.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#event-recordingStateChanged)
   */
  public fun recordingStateChangedEvents(): Flow<BackgroundServiceEvent.RecordingStateChanged> =
      session.typedEvents("BackgroundService.recordingStateChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("recordingStateChangedEvents()"),
  )
  public fun recordingStateChanged(): Flow<BackgroundServiceEvent.RecordingStateChanged> =
      recordingStateChangedEvents()

  /**
   * Called with all existing backgroundServiceEvents when enabled, and all new
   * events afterwards if enabled and recording.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#event-backgroundServiceEventReceived)
   */
  public fun backgroundServiceEventReceivedEvents():
      Flow<BackgroundServiceEvent.BackgroundServiceEventReceived> =
      session.typedEvents("BackgroundService.backgroundServiceEventReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("backgroundServiceEventReceivedEvents()"),
  )
  public fun backgroundServiceEventReceived():
      Flow<BackgroundServiceEvent.BackgroundServiceEventReceived> =
      backgroundServiceEventReceivedEvents()

  /**
   * Enables event updates for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-startObserving)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startObserving(input: StartObservingRequest): StartObservingResponse =
      session.request("BackgroundService.startObserving", input)

  /**
   * Enables event updates for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-startObserving)
   */
  public suspend fun startObserving(service: ServiceName): StartObservingResponse {
    val input = StartObservingRequest(service)
    return startObserving(input)
  }

  /**
   * Disables event updates for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-stopObserving)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun stopObserving(input: StopObservingRequest): StopObservingResponse =
      session.request("BackgroundService.stopObserving", input)

  /**
   * Disables event updates for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-stopObserving)
   */
  public suspend fun stopObserving(service: ServiceName): StopObservingResponse {
    val input = StopObservingRequest(service)
    return stopObserving(input)
  }

  /**
   * Set the recording state for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-setRecording)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setRecording(input: SetRecordingRequest): SetRecordingResponse =
      session.request("BackgroundService.setRecording", input)

  /**
   * Set the recording state for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-setRecording)
   */
  public suspend fun setRecording(shouldRecord: Boolean, service: ServiceName):
      SetRecordingResponse {
    val input = SetRecordingRequest(shouldRecord, service)
    return setRecording(input)
  }

  /**
   * Clears all stored data for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-clearEvents)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clearEvents(input: ClearEventsRequest): ClearEventsResponse =
      session.request("BackgroundService.clearEvents", input)

  /**
   * Clears all stored data for the service.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/BackgroundService/#method-clearEvents)
   */
  public suspend fun clearEvents(service: ServiceName): ClearEventsResponse {
    val input = ClearEventsRequest(service)
    return clearEvents(input)
  }
}
