@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.browser

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.browser.events.BrowserEvent
import org.hildan.chrome.devtools.domains.target.TargetID
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [BrowserDomain.setPermission] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetPermissionRequest(
  /**
   * Descriptor of permission to override.
   */
  public val permission: PermissionDescriptor,
  /**
   * Setting of the permission.
   */
  public val setting: PermissionSetting,
  /**
   * Origin the permission applies to, all origins if not specified.
   */
  public val origin: String? = null,
  /**
   * Context to override. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder(
    /**
     * Descriptor of permission to override.
     */
    public val permission: PermissionDescriptor,
    /**
     * Setting of the permission.
     */
    public val setting: PermissionSetting,
  ) {
    /**
     * Origin the permission applies to, all origins if not specified.
     */
    public var origin: String? = null

    /**
     * Context to override. When omitted, default browser context is used.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): SetPermissionRequest = SetPermissionRequest(permission, setting, origin,
        browserContextId)
  }
}

/**
 * A dummy response object for the [BrowserDomain.setPermission] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetPermissionResponse

/**
 * Request object containing input parameters for the [BrowserDomain.grantPermissions] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GrantPermissionsRequest(
  public val permissions: List<PermissionType>,
  /**
   * Origin the permission applies to, all origins if not specified.
   */
  public val origin: String? = null,
  /**
   * BrowserContext to override permissions. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder(
    public val permissions: List<PermissionType>,
  ) {
    /**
     * Origin the permission applies to, all origins if not specified.
     */
    public var origin: String? = null

    /**
     * BrowserContext to override permissions. When omitted, default browser context is used.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): GrantPermissionsRequest = GrantPermissionsRequest(permissions, origin,
        browserContextId)
  }
}

/**
 * A dummy response object for the [BrowserDomain.grantPermissions] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object GrantPermissionsResponse

/**
 * Request object containing input parameters for the [BrowserDomain.resetPermissions] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ResetPermissionsRequest(
  /**
   * BrowserContext to reset permissions. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder() {
    /**
     * BrowserContext to reset permissions. When omitted, default browser context is used.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): ResetPermissionsRequest = ResetPermissionsRequest(browserContextId)
  }
}

/**
 * A dummy response object for the [BrowserDomain.resetPermissions] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ResetPermissionsResponse

/**
 * Request object containing input parameters for the [BrowserDomain.setDownloadBehavior] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDownloadBehaviorRequest(
  /**
   * Whether to allow all or deny all download requests, or use default Chrome behavior if
   * available (otherwise deny). |allowAndName| allows download and names files according to
   * their dowmload guids.
   */
  public val behavior: DownloadBehavior,
  /**
   * BrowserContext to set download behavior. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null,
  /**
   * The default path to save downloaded files to. This is required if behavior is set to 'allow'
   * or 'allowAndName'.
   */
  public val downloadPath: String? = null,
  /**
   * Whether to emit download events (defaults to false).
   */
  public val eventsEnabled: Boolean? = null,
) {
  public class Builder(
    /**
     * Whether to allow all or deny all download requests, or use default Chrome behavior if
     * available (otherwise deny). |allowAndName| allows download and names files according to
     * their dowmload guids.
     */
    public val behavior: DownloadBehavior,
  ) {
    /**
     * BrowserContext to set download behavior. When omitted, default browser context is used.
     */
    public var browserContextId: BrowserContextID? = null

    /**
     * The default path to save downloaded files to. This is required if behavior is set to 'allow'
     * or 'allowAndName'.
     */
    public var downloadPath: String? = null

    /**
     * Whether to emit download events (defaults to false).
     */
    public var eventsEnabled: Boolean? = null

    public fun build(): SetDownloadBehaviorRequest = SetDownloadBehaviorRequest(behavior,
        browserContextId, downloadPath, eventsEnabled)
  }
}

/**
 * A dummy response object for the [BrowserDomain.setDownloadBehavior] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetDownloadBehaviorResponse

/**
 * Request object containing input parameters for the [BrowserDomain.cancelDownload] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CancelDownloadRequest(
  /**
   * Global unique identifier of the download.
   */
  public val guid: String,
  /**
   * BrowserContext to perform the action in. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder(
    /**
     * Global unique identifier of the download.
     */
    public val guid: String,
  ) {
    /**
     * BrowserContext to perform the action in. When omitted, default browser context is used.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): CancelDownloadRequest = CancelDownloadRequest(guid, browserContextId)
  }
}

/**
 * A dummy response object for the [BrowserDomain.cancelDownload] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object CancelDownloadResponse

/**
 * A dummy response object for the [BrowserDomain.close] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object CloseResponse

/**
 * A dummy response object for the [BrowserDomain.crash] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
@ExperimentalChromeApi
public object CrashResponse

/**
 * A dummy response object for the [BrowserDomain.crashGpuProcess] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object CrashGpuProcessResponse

/**
 * Response type for the [BrowserDomain.getVersion] command.
 */
@Serializable
public data class GetVersionResponse(
  /**
   * Protocol version.
   */
  public val protocolVersion: String,
  /**
   * Product name.
   */
  public val product: String,
  /**
   * Product revision.
   */
  public val revision: String,
  /**
   * User-Agent.
   */
  public val userAgent: String,
  /**
   * V8 version.
   */
  public val jsVersion: String,
)

/**
 * Response type for the [BrowserDomain.getBrowserCommandLine] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetBrowserCommandLineResponse(
  /**
   * Commandline parameters
   */
  public val arguments: List<String>,
)

/**
 * Request object containing input parameters for the [BrowserDomain.getHistograms] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramsRequest(
  /**
   * Requested substring in name. Only histograms which have query as a
   * substring in their name are extracted. An empty or absent query returns
   * all histograms.
   */
  public val query: String? = null,
  /**
   * If true, retrieve delta since last call.
   */
  public val delta: Boolean? = null,
) {
  public class Builder() {
    /**
     * Requested substring in name. Only histograms which have query as a
     * substring in their name are extracted. An empty or absent query returns
     * all histograms.
     */
    public var query: String? = null

    /**
     * If true, retrieve delta since last call.
     */
    public var delta: Boolean? = null

    public fun build(): GetHistogramsRequest = GetHistogramsRequest(query, delta)
  }
}

/**
 * Response type for the [BrowserDomain.getHistograms] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramsResponse(
  /**
   * Histograms.
   */
  public val histograms: List<Histogram>,
)

/**
 * Request object containing input parameters for the [BrowserDomain.getHistogram] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramRequest(
  /**
   * Requested histogram name.
   */
  public val name: String,
  /**
   * If true, retrieve delta since last call.
   */
  public val delta: Boolean? = null,
) {
  public class Builder(
    /**
     * Requested histogram name.
     */
    public val name: String,
  ) {
    /**
     * If true, retrieve delta since last call.
     */
    public var delta: Boolean? = null

    public fun build(): GetHistogramRequest = GetHistogramRequest(name, delta)
  }
}

/**
 * Response type for the [BrowserDomain.getHistogram] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramResponse(
  /**
   * Histogram.
   */
  public val histogram: Histogram,
)

/**
 * Request object containing input parameters for the [BrowserDomain.getWindowBounds] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowBoundsRequest(
  /**
   * Browser window id.
   */
  public val windowId: WindowID,
)

/**
 * Response type for the [BrowserDomain.getWindowBounds] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowBoundsResponse(
  /**
   * Bounds information of the window. When window state is 'minimized', the restored window
   * position and size are returned.
   */
  public val bounds: Bounds,
)

/**
 * Request object containing input parameters for the [BrowserDomain.getWindowForTarget] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowForTargetRequest(
  /**
   * Devtools agent host id. If called as a part of the session, associated targetId is used.
   */
  public val targetId: TargetID? = null,
) {
  public class Builder() {
    /**
     * Devtools agent host id. If called as a part of the session, associated targetId is used.
     */
    public var targetId: TargetID? = null

    public fun build(): GetWindowForTargetRequest = GetWindowForTargetRequest(targetId)
  }
}

/**
 * Response type for the [BrowserDomain.getWindowForTarget] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowForTargetResponse(
  /**
   * Browser window id.
   */
  public val windowId: WindowID,
  /**
   * Bounds information of the window. When window state is 'minimized', the restored window
   * position and size are returned.
   */
  public val bounds: Bounds,
)

/**
 * Request object containing input parameters for the [BrowserDomain.setWindowBounds] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetWindowBoundsRequest(
  /**
   * Browser window id.
   */
  public val windowId: WindowID,
  /**
   * New window bounds. The 'minimized', 'maximized' and 'fullscreen' states cannot be combined
   * with 'left', 'top', 'width' or 'height'. Leaves unspecified fields unchanged.
   */
  public val bounds: Bounds,
)

/**
 * A dummy response object for the [BrowserDomain.setWindowBounds] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetWindowBoundsResponse

/**
 * Request object containing input parameters for the [BrowserDomain.setDockTile] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDockTileRequest(
  public val badgeLabel: String? = null,
  /**
   * Png encoded image. (Encoded as a base64 string when passed over JSON)
   */
  public val image: String? = null,
) {
  public class Builder() {
    public var badgeLabel: String? = null

    /**
     * Png encoded image. (Encoded as a base64 string when passed over JSON)
     */
    public var image: String? = null

    public fun build(): SetDockTileRequest = SetDockTileRequest(badgeLabel, image)
  }
}

/**
 * A dummy response object for the [BrowserDomain.setDockTile] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetDockTileResponse

/**
 * Request object containing input parameters for the [BrowserDomain.executeBrowserCommand] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ExecuteBrowserCommandRequest(
  public val commandId: BrowserCommandId,
)

/**
 * A dummy response object for the [BrowserDomain.executeBrowserCommand] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ExecuteBrowserCommandResponse

/**
 * The Browser domain defines methods and events for browser managing.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser)
 */
public class BrowserDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out BrowserEvent>> =
      mapOf(
      "Browser.downloadWillBegin" to serializer<BrowserEvent.DownloadWillBegin>(),
      "Browser.downloadProgress" to serializer<BrowserEvent.DownloadProgress>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<BrowserEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fired when page is about to start a download.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#event-downloadWillBegin)
   */
  @ExperimentalChromeApi
  public fun downloadWillBeginEvents(): Flow<BrowserEvent.DownloadWillBegin> =
      session.typedEvents("Browser.downloadWillBegin")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("downloadWillBeginEvents()"),
  )
  public fun downloadWillBegin(): Flow<BrowserEvent.DownloadWillBegin> = downloadWillBeginEvents()

  /**
   * Fired when download makes progress. Last call has |done| == true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#event-downloadProgress)
   */
  @ExperimentalChromeApi
  public fun downloadProgressEvents(): Flow<BrowserEvent.DownloadProgress> =
      session.typedEvents("Browser.downloadProgress")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("downloadProgressEvents()"),
  )
  public fun downloadProgress(): Flow<BrowserEvent.DownloadProgress> = downloadProgressEvents()

  /**
   * Set permission settings for given origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setPermission)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setPermission(input: SetPermissionRequest): SetPermissionResponse =
      session.request("Browser.setPermission", input)

  /**
   * Set permission settings for given origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setPermission)
   *
   * @param permission Descriptor of permission to override.
   * @param setting Setting of the permission.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setPermission(
    permission: PermissionDescriptor,
    setting: PermissionSetting,
    optionalArgs: SetPermissionRequest.Builder.() -> Unit = {},
  ): SetPermissionResponse {
    val builder = SetPermissionRequest.Builder(permission, setting)
    val input = builder.apply(optionalArgs).build()
    return setPermission(input)
  }

  /**
   * Grant specific permissions to the given origin and reject all others.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-grantPermissions)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun grantPermissions(input: GrantPermissionsRequest): GrantPermissionsResponse =
      session.request("Browser.grantPermissions", input)

  /**
   * Grant specific permissions to the given origin and reject all others.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-grantPermissions)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun grantPermissions(permissions: List<PermissionType>,
      optionalArgs: GrantPermissionsRequest.Builder.() -> Unit = {}): GrantPermissionsResponse {
    val builder = GrantPermissionsRequest.Builder(permissions)
    val input = builder.apply(optionalArgs).build()
    return grantPermissions(input)
  }

  /**
   * Reset all permission management for all origins.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-resetPermissions)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun resetPermissions(input: ResetPermissionsRequest): ResetPermissionsResponse =
      session.request("Browser.resetPermissions", input)

  /**
   * Reset all permission management for all origins.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-resetPermissions)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun resetPermissions(optionalArgs: ResetPermissionsRequest.Builder.() -> Unit = {}):
      ResetPermissionsResponse {
    val builder = ResetPermissionsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return resetPermissions(input)
  }

  /**
   * Set the behavior when downloading a file.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setDownloadBehavior)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setDownloadBehavior(input: SetDownloadBehaviorRequest):
      SetDownloadBehaviorResponse = session.request("Browser.setDownloadBehavior", input)

  /**
   * Set the behavior when downloading a file.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setDownloadBehavior)
   *
   * @param behavior Whether to allow all or deny all download requests, or use default Chrome
   * behavior if
   * available (otherwise deny). |allowAndName| allows download and names files according to
   * their dowmload guids.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setDownloadBehavior(behavior: DownloadBehavior,
      optionalArgs: SetDownloadBehaviorRequest.Builder.() -> Unit = {}):
      SetDownloadBehaviorResponse {
    val builder = SetDownloadBehaviorRequest.Builder(behavior)
    val input = builder.apply(optionalArgs).build()
    return setDownloadBehavior(input)
  }

  /**
   * Cancel a download if in progress
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-cancelDownload)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun cancelDownload(input: CancelDownloadRequest): CancelDownloadResponse =
      session.request("Browser.cancelDownload", input)

  /**
   * Cancel a download if in progress
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-cancelDownload)
   *
   * @param guid Global unique identifier of the download.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun cancelDownload(guid: String,
      optionalArgs: CancelDownloadRequest.Builder.() -> Unit = {}): CancelDownloadResponse {
    val builder = CancelDownloadRequest.Builder(guid)
    val input = builder.apply(optionalArgs).build()
    return cancelDownload(input)
  }

  /**
   * Close browser gracefully.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-close)
   */
  public suspend fun close(): CloseResponse = session.request("Browser.close", Unit)

  /**
   * Crashes browser on the main thread.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-crash)
   */
  @ExperimentalChromeApi
  public suspend fun crash(): CrashResponse = session.request("Browser.crash", Unit)

  /**
   * Crashes GPU process.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-crashGpuProcess)
   */
  @ExperimentalChromeApi
  public suspend fun crashGpuProcess(): CrashGpuProcessResponse =
      session.request("Browser.crashGpuProcess", Unit)

  /**
   * Returns version information.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getVersion)
   */
  public suspend fun getVersion(): GetVersionResponse = session.request("Browser.getVersion", Unit)

  /**
   * Returns the command line switches for the browser process if, and only if
   * --enable-automation is on the commandline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getBrowserCommandLine)
   */
  @ExperimentalChromeApi
  public suspend fun getBrowserCommandLine(): GetBrowserCommandLineResponse =
      session.request("Browser.getBrowserCommandLine", Unit)

  /**
   * Get Chrome histograms.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getHistograms)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getHistograms(input: GetHistogramsRequest): GetHistogramsResponse =
      session.request("Browser.getHistograms", input)

  /**
   * Get Chrome histograms.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getHistograms)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getHistograms(optionalArgs: GetHistogramsRequest.Builder.() -> Unit =
      {}): GetHistogramsResponse {
    val builder = GetHistogramsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getHistograms(input)
  }

  /**
   * Get a Chrome histogram by name.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getHistogram)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getHistogram(input: GetHistogramRequest): GetHistogramResponse =
      session.request("Browser.getHistogram", input)

  /**
   * Get a Chrome histogram by name.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getHistogram)
   *
   * @param name Requested histogram name.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getHistogram(name: String,
      optionalArgs: GetHistogramRequest.Builder.() -> Unit = {}): GetHistogramResponse {
    val builder = GetHistogramRequest.Builder(name)
    val input = builder.apply(optionalArgs).build()
    return getHistogram(input)
  }

  /**
   * Get position and size of the browser window.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getWindowBounds)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getWindowBounds(input: GetWindowBoundsRequest): GetWindowBoundsResponse =
      session.request("Browser.getWindowBounds", input)

  /**
   * Get position and size of the browser window.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getWindowBounds)
   *
   * @param windowId Browser window id.
   */
  @ExperimentalChromeApi
  public suspend fun getWindowBounds(windowId: WindowID): GetWindowBoundsResponse {
    val input = GetWindowBoundsRequest(windowId)
    return getWindowBounds(input)
  }

  /**
   * Get the browser window that contains the devtools target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getWindowForTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getWindowForTarget(input: GetWindowForTargetRequest):
      GetWindowForTargetResponse = session.request("Browser.getWindowForTarget", input)

  /**
   * Get the browser window that contains the devtools target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getWindowForTarget)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun getWindowForTarget(optionalArgs: GetWindowForTargetRequest.Builder.() -> Unit = {}):
      GetWindowForTargetResponse {
    val builder = GetWindowForTargetRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getWindowForTarget(input)
  }

  /**
   * Set position and/or size of the browser window.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setWindowBounds)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setWindowBounds(input: SetWindowBoundsRequest): SetWindowBoundsResponse =
      session.request("Browser.setWindowBounds", input)

  /**
   * Set position and/or size of the browser window.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setWindowBounds)
   *
   * @param windowId Browser window id.
   * @param bounds New window bounds. The 'minimized', 'maximized' and 'fullscreen' states cannot be
   * combined
   * with 'left', 'top', 'width' or 'height'. Leaves unspecified fields unchanged.
   */
  @ExperimentalChromeApi
  public suspend fun setWindowBounds(windowId: WindowID, bounds: Bounds): SetWindowBoundsResponse {
    val input = SetWindowBoundsRequest(windowId, bounds)
    return setWindowBounds(input)
  }

  /**
   * Set dock tile details, platform-specific.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setDockTile)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setDockTile(input: SetDockTileRequest): SetDockTileResponse =
      session.request("Browser.setDockTile", input)

  /**
   * Set dock tile details, platform-specific.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setDockTile)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setDockTile(optionalArgs: SetDockTileRequest.Builder.() -> Unit = {}):
      SetDockTileResponse {
    val builder = SetDockTileRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setDockTile(input)
  }

  /**
   * Invoke custom browser commands used by telemetry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-executeBrowserCommand)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun executeBrowserCommand(input: ExecuteBrowserCommandRequest):
      ExecuteBrowserCommandResponse = session.request("Browser.executeBrowserCommand", input)

  /**
   * Invoke custom browser commands used by telemetry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-executeBrowserCommand)
   */
  @ExperimentalChromeApi
  public suspend fun executeBrowserCommand(commandId: BrowserCommandId):
      ExecuteBrowserCommandResponse {
    val input = ExecuteBrowserCommandRequest(commandId)
    return executeBrowserCommand(input)
  }
}
