@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.cachestorage

import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the [CacheStorageDomain.deleteCache] command.
 */
@Serializable
public data class DeleteCacheRequest(
  /**
   * Id of cache for deletion.
   */
  public val cacheId: CacheId,
)

/**
 * A dummy response object for the [CacheStorageDomain.deleteCache] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DeleteCacheResponse

/**
 * Request object containing input parameters for the [CacheStorageDomain.deleteEntry] command.
 */
@Serializable
public data class DeleteEntryRequest(
  /**
   * Id of cache where the entry will be deleted.
   */
  public val cacheId: CacheId,
  /**
   * URL spec of the request.
   */
  public val request: String,
)

/**
 * A dummy response object for the [CacheStorageDomain.deleteEntry] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DeleteEntryResponse

/**
 * Request object containing input parameters for the [CacheStorageDomain.requestCacheNames]
 * command.
 */
@Serializable
public data class RequestCacheNamesRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String,
)

/**
 * Response type for the [CacheStorageDomain.requestCacheNames] command.
 */
@Serializable
public data class RequestCacheNamesResponse(
  /**
   * Caches for the security origin.
   */
  public val caches: List<Cache>,
)

/**
 * Request object containing input parameters for the [CacheStorageDomain.requestCachedResponse]
 * command.
 */
@Serializable
public data class RequestCachedResponseRequest(
  /**
   * Id of cache that contains the entry.
   */
  public val cacheId: CacheId,
  /**
   * URL spec of the request.
   */
  public val requestURL: String,
  /**
   * headers of the request.
   */
  public val requestHeaders: List<Header>,
)

/**
 * Response type for the [CacheStorageDomain.requestCachedResponse] command.
 */
@Serializable
public data class RequestCachedResponseResponse(
  /**
   * Response read from the cache.
   */
  public val response: CachedResponse,
)

/**
 * Request object containing input parameters for the [CacheStorageDomain.requestEntries] command.
 */
@Serializable
public data class RequestEntriesRequest(
  /**
   * ID of cache to get entries from.
   */
  public val cacheId: CacheId,
  /**
   * Number of records to skip.
   */
  public val skipCount: Int? = null,
  /**
   * Number of records to fetch.
   */
  public val pageSize: Int? = null,
  /**
   * If present, only return the entries containing this substring in the path
   */
  public val pathFilter: String? = null,
) {
  public class Builder(
    /**
     * ID of cache to get entries from.
     */
    public val cacheId: CacheId,
  ) {
    /**
     * Number of records to skip.
     */
    public var skipCount: Int? = null

    /**
     * Number of records to fetch.
     */
    public var pageSize: Int? = null

    /**
     * If present, only return the entries containing this substring in the path
     */
    public var pathFilter: String? = null

    public fun build(): RequestEntriesRequest = RequestEntriesRequest(cacheId, skipCount, pageSize,
        pathFilter)
  }
}

/**
 * Response type for the [CacheStorageDomain.requestEntries] command.
 */
@Serializable
public data class RequestEntriesResponse(
  /**
   * Array of object store data entries.
   */
  public val cacheDataEntries: List<DataEntry>,
  /**
   * Count of returned entries from this storage. If pathFilter is empty, it
   * is the count of all entries from this storage.
   */
  public val returnCount: Double,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage)
 */
@ExperimentalChromeApi
public class CacheStorageDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Deletes a cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-deleteCache)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun deleteCache(input: DeleteCacheRequest): DeleteCacheResponse =
      session.request("CacheStorage.deleteCache", input)

  /**
   * Deletes a cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-deleteCache)
   *
   * @param cacheId Id of cache for deletion.
   */
  public suspend fun deleteCache(cacheId: CacheId): DeleteCacheResponse {
    val input = DeleteCacheRequest(cacheId)
    return deleteCache(input)
  }

  /**
   * Deletes a cache entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-deleteEntry)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun deleteEntry(input: DeleteEntryRequest): DeleteEntryResponse =
      session.request("CacheStorage.deleteEntry", input)

  /**
   * Deletes a cache entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-deleteEntry)
   *
   * @param cacheId Id of cache where the entry will be deleted.
   * @param request URL spec of the request.
   */
  public suspend fun deleteEntry(cacheId: CacheId, request: String): DeleteEntryResponse {
    val input = DeleteEntryRequest(cacheId, request)
    return deleteEntry(input)
  }

  /**
   * Requests cache names.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestCacheNames)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestCacheNames(input: RequestCacheNamesRequest): RequestCacheNamesResponse =
      session.request("CacheStorage.requestCacheNames", input)

  /**
   * Requests cache names.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestCacheNames)
   *
   * @param securityOrigin Security origin.
   */
  public suspend fun requestCacheNames(securityOrigin: String): RequestCacheNamesResponse {
    val input = RequestCacheNamesRequest(securityOrigin)
    return requestCacheNames(input)
  }

  /**
   * Fetches cache entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestCachedResponse)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestCachedResponse(input: RequestCachedResponseRequest):
      RequestCachedResponseResponse = session.request("CacheStorage.requestCachedResponse", input)

  /**
   * Fetches cache entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestCachedResponse)
   *
   * @param cacheId Id of cache that contains the entry.
   * @param requestURL URL spec of the request.
   * @param requestHeaders headers of the request.
   */
  public suspend fun requestCachedResponse(
    cacheId: CacheId,
    requestURL: String,
    requestHeaders: List<Header>,
  ): RequestCachedResponseResponse {
    val input = RequestCachedResponseRequest(cacheId, requestURL, requestHeaders)
    return requestCachedResponse(input)
  }

  /**
   * Requests data from cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestEntries)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestEntries(input: RequestEntriesRequest): RequestEntriesResponse =
      session.request("CacheStorage.requestEntries", input)

  /**
   * Requests data from cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestEntries)
   *
   * @param cacheId ID of cache to get entries from.
   */
  @JvmOverloads
  public suspend inline fun requestEntries(cacheId: CacheId,
      optionalArgs: RequestEntriesRequest.Builder.() -> Unit = {}): RequestEntriesResponse {
    val builder = RequestEntriesRequest.Builder(cacheId)
    val input = builder.apply(optionalArgs).build()
    return requestEntries(input)
  }
}
