@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.cast

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.cast.events.CastEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [CastDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  public val presentationUrl: String? = null,
) {
  public class Builder() {
    public var presentationUrl: String? = null

    public fun build(): EnableRequest = EnableRequest(presentationUrl)
  }
}

/**
 * A dummy response object for the [CastDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * A dummy response object for the [CastDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [CastDomain.setSinkToUse] command.
 */
@Serializable
public data class SetSinkToUseRequest(
  public val sinkName: String,
)

/**
 * A dummy response object for the [CastDomain.setSinkToUse] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object SetSinkToUseResponse

/**
 * Request object containing input parameters for the [CastDomain.startDesktopMirroring] command.
 */
@Serializable
public data class StartDesktopMirroringRequest(
  public val sinkName: String,
)

/**
 * A dummy response object for the [CastDomain.startDesktopMirroring] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object StartDesktopMirroringResponse

/**
 * Request object containing input parameters for the [CastDomain.startTabMirroring] command.
 */
@Serializable
public data class StartTabMirroringRequest(
  public val sinkName: String,
)

/**
 * A dummy response object for the [CastDomain.startTabMirroring] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object StartTabMirroringResponse

/**
 * Request object containing input parameters for the [CastDomain.stopCasting] command.
 */
@Serializable
public data class StopCastingRequest(
  public val sinkName: String,
)

/**
 * A dummy response object for the [CastDomain.stopCasting] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object StopCastingResponse

/**
 * A domain for interacting with Cast, Presentation API, and Remote Playback API
 * functionalities.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast)
 */
@ExperimentalChromeApi
public class CastDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out CastEvent>> = mapOf(
      "Cast.sinksUpdated" to serializer<CastEvent.SinksUpdated>(),
      "Cast.issueUpdated" to serializer<CastEvent.IssueUpdated>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<CastEvent> = session.typedEvents(deserializersByEventName)

  /**
   * This is fired whenever the list of available sinks changes. A sink is a
   * device or a software surface that you can cast to.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#event-sinksUpdated)
   */
  public fun sinksUpdatedEvents(): Flow<CastEvent.SinksUpdated> =
      session.typedEvents("Cast.sinksUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("sinksUpdatedEvents()"),
  )
  public fun sinksUpdated(): Flow<CastEvent.SinksUpdated> = sinksUpdatedEvents()

  /**
   * This is fired whenever the outstanding issue/error message changes.
   * |issueMessage| is empty if there is no issue.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#event-issueUpdated)
   */
  public fun issueUpdatedEvents(): Flow<CastEvent.IssueUpdated> =
      session.typedEvents("Cast.issueUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("issueUpdatedEvents()"),
  )
  public fun issueUpdated(): Flow<CastEvent.IssueUpdated> = issueUpdatedEvents()

  /**
   * Starts observing for sinks that can be used for tab mirroring, and if set,
   * sinks compatible with |presentationUrl| as well. When sinks are found, a
   * |sinksUpdated| event is fired.
   * Also starts observing for issue messages. When an issue is added or removed,
   * an |issueUpdated| event is fired.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse = session.request("Cast.enable",
      input)

  /**
   * Starts observing for sinks that can be used for tab mirroring, and if set,
   * sinks compatible with |presentationUrl| as well. When sinks are found, a
   * |sinksUpdated| event is fired.
   * Also starts observing for issue messages. When an issue is added or removed,
   * an |issueUpdated| event is fired.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Stops observing for sinks and issues.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Cast.disable", Unit)

  /**
   * Sets a sink to be used when the web page requests the browser to choose a
   * sink via Presentation API, Remote Playback API, or Cast SDK.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-setSinkToUse)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setSinkToUse(input: SetSinkToUseRequest): SetSinkToUseResponse =
      session.request("Cast.setSinkToUse", input)

  /**
   * Sets a sink to be used when the web page requests the browser to choose a
   * sink via Presentation API, Remote Playback API, or Cast SDK.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-setSinkToUse)
   */
  public suspend fun setSinkToUse(sinkName: String): SetSinkToUseResponse {
    val input = SetSinkToUseRequest(sinkName)
    return setSinkToUse(input)
  }

  /**
   * Starts mirroring the desktop to the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-startDesktopMirroring)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startDesktopMirroring(input: StartDesktopMirroringRequest):
      StartDesktopMirroringResponse = session.request("Cast.startDesktopMirroring", input)

  /**
   * Starts mirroring the desktop to the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-startDesktopMirroring)
   */
  public suspend fun startDesktopMirroring(sinkName: String): StartDesktopMirroringResponse {
    val input = StartDesktopMirroringRequest(sinkName)
    return startDesktopMirroring(input)
  }

  /**
   * Starts mirroring the tab to the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-startTabMirroring)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startTabMirroring(input: StartTabMirroringRequest): StartTabMirroringResponse =
      session.request("Cast.startTabMirroring", input)

  /**
   * Starts mirroring the tab to the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-startTabMirroring)
   */
  public suspend fun startTabMirroring(sinkName: String): StartTabMirroringResponse {
    val input = StartTabMirroringRequest(sinkName)
    return startTabMirroring(input)
  }

  /**
   * Stops the active Cast session on the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-stopCasting)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun stopCasting(input: StopCastingRequest): StopCastingResponse =
      session.request("Cast.stopCasting", input)

  /**
   * Stops the active Cast session on the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-stopCasting)
   */
  public suspend fun stopCasting(sinkName: String): StopCastingResponse {
    val input = StopCastingRequest(sinkName)
    return stopCasting(input)
  }
}
