@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.css

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.css.events.CSSEvent
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [CSSDomain.addRule] command.
 */
@Serializable
public data class AddRuleRequest(
  /**
   * The css style sheet identifier where a new rule should be inserted.
   */
  public val styleSheetId: StyleSheetId,
  /**
   * The text of a new rule.
   */
  public val ruleText: String,
  /**
   * Text position of a new rule in the target style sheet.
   */
  public val location: SourceRange,
)

/**
 * Response type for the [CSSDomain.addRule] command.
 */
@Serializable
public data class AddRuleResponse(
  /**
   * The newly created rule.
   */
  public val rule: CSSRule,
)

/**
 * Request object containing input parameters for the [CSSDomain.collectClassNames] command.
 */
@Serializable
public data class CollectClassNamesRequest(
  public val styleSheetId: StyleSheetId,
)

/**
 * Response type for the [CSSDomain.collectClassNames] command.
 */
@Serializable
public data class CollectClassNamesResponse(
  /**
   * Class name list.
   */
  public val classNames: List<String>,
)

/**
 * Request object containing input parameters for the [CSSDomain.createStyleSheet] command.
 */
@Serializable
public data class CreateStyleSheetRequest(
  /**
   * Identifier of the frame where "via-inspector" stylesheet should be created.
   */
  public val frameId: FrameId,
)

/**
 * Response type for the [CSSDomain.createStyleSheet] command.
 */
@Serializable
public data class CreateStyleSheetResponse(
  /**
   * Identifier of the created "via-inspector" stylesheet.
   */
  public val styleSheetId: StyleSheetId,
)

/**
 * A dummy response object for the [CSSDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [CSSDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [CSSDomain.forcePseudoState] command.
 */
@Serializable
public data class ForcePseudoStateRequest(
  /**
   * The element id for which to force the pseudo state.
   */
  public val nodeId: NodeId,
  /**
   * Element pseudo classes to force when computing the element's style.
   */
  public val forcedPseudoClasses: List<String>,
)

/**
 * A dummy response object for the [CSSDomain.forcePseudoState] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object ForcePseudoStateResponse

/**
 * Request object containing input parameters for the [CSSDomain.getBackgroundColors] command.
 */
@Serializable
public data class GetBackgroundColorsRequest(
  /**
   * Id of the node to get background colors for.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [CSSDomain.getBackgroundColors] command.
 */
@Serializable
public data class GetBackgroundColorsResponse(
  /**
   * The range of background colors behind this element, if it contains any visible text. If no
   * visible text is present, this will be undefined. In the case of a flat background color,
   * this will consist of simply that color. In the case of a gradient, this will consist of each
   * of the color stops. For anything more complicated, this will be an empty array. Images will
   * be ignored (as if the image had failed to load).
   */
  public val backgroundColors: List<String>? = null,
  /**
   * The computed font size for this node, as a CSS computed value string (e.g. '12px').
   */
  public val computedFontSize: String? = null,
  /**
   * The computed font weight for this node, as a CSS computed value string (e.g. 'normal' or
   * '100').
   */
  public val computedFontWeight: String? = null,
)

/**
 * Request object containing input parameters for the [CSSDomain.getComputedStyleForNode] command.
 */
@Serializable
public data class GetComputedStyleForNodeRequest(
  public val nodeId: NodeId,
)

/**
 * Response type for the [CSSDomain.getComputedStyleForNode] command.
 */
@Serializable
public data class GetComputedStyleForNodeResponse(
  /**
   * Computed style for the specified DOM node.
   */
  public val computedStyle: List<CSSComputedStyleProperty>,
)

/**
 * Request object containing input parameters for the [CSSDomain.getInlineStylesForNode] command.
 */
@Serializable
public data class GetInlineStylesForNodeRequest(
  public val nodeId: NodeId,
)

/**
 * Response type for the [CSSDomain.getInlineStylesForNode] command.
 */
@Serializable
public data class GetInlineStylesForNodeResponse(
  /**
   * Inline style for the specified DOM node.
   */
  public val inlineStyle: CSSStyle? = null,
  /**
   * Attribute-defined element style (e.g. resulting from "width=20 height=100%").
   */
  public val attributesStyle: CSSStyle? = null,
)

/**
 * Request object containing input parameters for the [CSSDomain.getMatchedStylesForNode] command.
 */
@Serializable
public data class GetMatchedStylesForNodeRequest(
  public val nodeId: NodeId,
)

/**
 * Response type for the [CSSDomain.getMatchedStylesForNode] command.
 */
@Serializable
public data class GetMatchedStylesForNodeResponse(
  /**
   * Inline style for the specified DOM node.
   */
  public val inlineStyle: CSSStyle? = null,
  /**
   * Attribute-defined element style (e.g. resulting from "width=20 height=100%").
   */
  public val attributesStyle: CSSStyle? = null,
  /**
   * CSS rules matching this node, from all applicable stylesheets.
   */
  public val matchedCSSRules: List<RuleMatch>? = null,
  /**
   * Pseudo style matches for this node.
   */
  public val pseudoElements: List<PseudoElementMatches>? = null,
  /**
   * A chain of inherited styles (from the immediate node parent up to the DOM tree root).
   */
  public val inherited: List<InheritedStyleEntry>? = null,
  /**
   * A chain of inherited pseudo element styles (from the immediate node parent up to the DOM tree
   * root).
   */
  public val inheritedPseudoElements: List<InheritedPseudoElementMatches>? = null,
  /**
   * A list of CSS keyframed animations matching this node.
   */
  public val cssKeyframesRules: List<CSSKeyframesRule>? = null,
  /**
   * Id of the first parent element that does not have display: contents.
   */
  @ExperimentalChromeApi
  public val parentLayoutNodeId: NodeId? = null,
)

/**
 * Response type for the [CSSDomain.getMediaQueries] command.
 */
@Serializable
public data class GetMediaQueriesResponse(
  public val medias: List<CSSMedia>,
)

/**
 * Request object containing input parameters for the [CSSDomain.getPlatformFontsForNode] command.
 */
@Serializable
public data class GetPlatformFontsForNodeRequest(
  public val nodeId: NodeId,
)

/**
 * Response type for the [CSSDomain.getPlatformFontsForNode] command.
 */
@Serializable
public data class GetPlatformFontsForNodeResponse(
  /**
   * Usage statistics for every employed platform font.
   */
  public val fonts: List<PlatformFontUsage>,
)

/**
 * Request object containing input parameters for the [CSSDomain.getStyleSheetText] command.
 */
@Serializable
public data class GetStyleSheetTextRequest(
  public val styleSheetId: StyleSheetId,
)

/**
 * Response type for the [CSSDomain.getStyleSheetText] command.
 */
@Serializable
public data class GetStyleSheetTextResponse(
  /**
   * The stylesheet text.
   */
  public val text: String,
)

/**
 * Request object containing input parameters for the [CSSDomain.getLayersForNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetLayersForNodeRequest(
  public val nodeId: NodeId,
)

/**
 * Response type for the [CSSDomain.getLayersForNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetLayersForNodeResponse(
  public val rootLayer: CSSLayerData,
)

/**
 * Request object containing input parameters for the [CSSDomain.trackComputedStyleUpdates] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TrackComputedStyleUpdatesRequest(
  public val propertiesToTrack: List<CSSComputedStyleProperty>,
)

/**
 * A dummy response object for the [CSSDomain.trackComputedStyleUpdates] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object TrackComputedStyleUpdatesResponse

/**
 * Response type for the [CSSDomain.takeComputedStyleUpdates] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeComputedStyleUpdatesResponse(
  /**
   * The list of node Ids that have their tracked computed styles updated
   */
  public val nodeIds: List<NodeId>,
)

/**
 * Request object containing input parameters for the [CSSDomain.setEffectivePropertyValueForNode]
 * command.
 */
@Serializable
public data class SetEffectivePropertyValueForNodeRequest(
  /**
   * The element id for which to set property.
   */
  public val nodeId: NodeId,
  public val propertyName: String,
  public val `value`: String,
)

/**
 * A dummy response object for the [CSSDomain.setEffectivePropertyValueForNode] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetEffectivePropertyValueForNodeResponse

/**
 * Request object containing input parameters for the [CSSDomain.setKeyframeKey] command.
 */
@Serializable
public data class SetKeyframeKeyRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val keyText: String,
)

/**
 * Response type for the [CSSDomain.setKeyframeKey] command.
 */
@Serializable
public data class SetKeyframeKeyResponse(
  /**
   * The resulting key text after modification.
   */
  public val keyText: Value,
)

/**
 * Request object containing input parameters for the [CSSDomain.setMediaText] command.
 */
@Serializable
public data class SetMediaTextRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val text: String,
)

/**
 * Response type for the [CSSDomain.setMediaText] command.
 */
@Serializable
public data class SetMediaTextResponse(
  /**
   * The resulting CSS media rule after modification.
   */
  public val media: CSSMedia,
)

/**
 * Request object containing input parameters for the [CSSDomain.setContainerQueryText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetContainerQueryTextRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val text: String,
)

/**
 * Response type for the [CSSDomain.setContainerQueryText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetContainerQueryTextResponse(
  /**
   * The resulting CSS container query rule after modification.
   */
  public val containerQuery: CSSContainerQuery,
)

/**
 * Request object containing input parameters for the [CSSDomain.setSupportsText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetSupportsTextRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val text: String,
)

/**
 * Response type for the [CSSDomain.setSupportsText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetSupportsTextResponse(
  /**
   * The resulting CSS Supports rule after modification.
   */
  public val supports: CSSSupports,
)

/**
 * Request object containing input parameters for the [CSSDomain.setScopeText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetScopeTextRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val text: String,
)

/**
 * Response type for the [CSSDomain.setScopeText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetScopeTextResponse(
  /**
   * The resulting CSS Scope rule after modification.
   */
  public val scope: CSSScope,
)

/**
 * Request object containing input parameters for the [CSSDomain.setRuleSelector] command.
 */
@Serializable
public data class SetRuleSelectorRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val selector: String,
)

/**
 * Response type for the [CSSDomain.setRuleSelector] command.
 */
@Serializable
public data class SetRuleSelectorResponse(
  /**
   * The resulting selector list after modification.
   */
  public val selectorList: SelectorList,
)

/**
 * Request object containing input parameters for the [CSSDomain.setStyleSheetText] command.
 */
@Serializable
public data class SetStyleSheetTextRequest(
  public val styleSheetId: StyleSheetId,
  public val text: String,
)

/**
 * Response type for the [CSSDomain.setStyleSheetText] command.
 */
@Serializable
public data class SetStyleSheetTextResponse(
  /**
   * URL of source map associated with script (if any).
   */
  public val sourceMapURL: String? = null,
)

/**
 * Request object containing input parameters for the [CSSDomain.setStyleTexts] command.
 */
@Serializable
public data class SetStyleTextsRequest(
  public val edits: List<StyleDeclarationEdit>,
)

/**
 * Response type for the [CSSDomain.setStyleTexts] command.
 */
@Serializable
public data class SetStyleTextsResponse(
  /**
   * The resulting styles after modification.
   */
  public val styles: List<CSSStyle>,
)

/**
 * A dummy response object for the [CSSDomain.startRuleUsageTracking] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object StartRuleUsageTrackingResponse

/**
 * Response type for the [CSSDomain.stopRuleUsageTracking] command.
 */
@Serializable
public data class StopRuleUsageTrackingResponse(
  public val ruleUsage: List<RuleUsage>,
)

/**
 * Response type for the [CSSDomain.takeCoverageDelta] command.
 */
@Serializable
public data class TakeCoverageDeltaResponse(
  public val coverage: List<RuleUsage>,
  /**
   * Monotonically increasing time, in seconds.
   */
  public val timestamp: Double,
)

/**
 * Request object containing input parameters for the [CSSDomain.setLocalFontsEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetLocalFontsEnabledRequest(
  /**
   * Whether rendering of local fonts is enabled.
   */
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [CSSDomain.setLocalFontsEnabled] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetLocalFontsEnabledResponse

/**
 * This domain exposes CSS read/write operations. All CSS objects (stylesheets, rules, and styles)
 * have an associated `id` used in subsequent operations on the related object. Each object type has
 * a specific `id` structure, and those are not interchangeable between objects of different kinds.
 * CSS objects can be loaded using the `get*ForNode()` calls (which accept a DOM node id). A client
 * can also keep track of stylesheets via the `styleSheetAdded`/`styleSheetRemoved` events and
 * subsequently load the required stylesheet contents using the `getStyleSheet[Text]()` methods.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS)
 */
@ExperimentalChromeApi
public class CSSDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out CSSEvent>> = mapOf(
      "CSS.fontsUpdated" to serializer<CSSEvent.FontsUpdated>(),
      "CSS.mediaQueryResultChanged" to serializer<CSSEvent.MediaQueryResultChanged>(),
      "CSS.styleSheetAdded" to serializer<CSSEvent.StyleSheetAdded>(),
      "CSS.styleSheetChanged" to serializer<CSSEvent.StyleSheetChanged>(),
      "CSS.styleSheetRemoved" to serializer<CSSEvent.StyleSheetRemoved>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<CSSEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fires whenever a web font is updated.  A non-empty font parameter indicates a successfully
   * loaded
   * web font
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#event-fontsUpdated)
   */
  public fun fontsUpdatedEvents(): Flow<CSSEvent.FontsUpdated> =
      session.typedEvents("CSS.fontsUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("fontsUpdatedEvents()"),
  )
  public fun fontsUpdated(): Flow<CSSEvent.FontsUpdated> = fontsUpdatedEvents()

  /**
   * Fires whenever a MediaQuery result changes (for example, after a browser window has been
   * resized.) The current implementation considers only viewport-dependent media features.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#event-mediaQueryResultChanged)
   */
  public fun mediaQueryResultChangedEvents(): Flow<CSSEvent.MediaQueryResultChanged> =
      session.typedEvents("CSS.mediaQueryResultChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("mediaQueryResultChangedEvents()"),
  )
  public fun mediaQueryResultChanged(): Flow<CSSEvent.MediaQueryResultChanged> =
      mediaQueryResultChangedEvents()

  /**
   * Fired whenever an active document stylesheet is added.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#event-styleSheetAdded)
   */
  public fun styleSheetAddedEvents(): Flow<CSSEvent.StyleSheetAdded> =
      session.typedEvents("CSS.styleSheetAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("styleSheetAddedEvents()"),
  )
  public fun styleSheetAdded(): Flow<CSSEvent.StyleSheetAdded> = styleSheetAddedEvents()

  /**
   * Fired whenever a stylesheet is changed as a result of the client operation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#event-styleSheetChanged)
   */
  public fun styleSheetChangedEvents(): Flow<CSSEvent.StyleSheetChanged> =
      session.typedEvents("CSS.styleSheetChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("styleSheetChangedEvents()"),
  )
  public fun styleSheetChanged(): Flow<CSSEvent.StyleSheetChanged> = styleSheetChangedEvents()

  /**
   * Fired whenever an active document stylesheet is removed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#event-styleSheetRemoved)
   */
  public fun styleSheetRemovedEvents(): Flow<CSSEvent.StyleSheetRemoved> =
      session.typedEvents("CSS.styleSheetRemoved")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("styleSheetRemovedEvents()"),
  )
  public fun styleSheetRemoved(): Flow<CSSEvent.StyleSheetRemoved> = styleSheetRemovedEvents()

  /**
   * Inserts a new rule with the given `ruleText` in a stylesheet with given `styleSheetId`, at the
   * position specified by `location`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-addRule)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun addRule(input: AddRuleRequest): AddRuleResponse =
      session.request("CSS.addRule", input)

  /**
   * Inserts a new rule with the given `ruleText` in a stylesheet with given `styleSheetId`, at the
   * position specified by `location`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-addRule)
   *
   * @param styleSheetId The css style sheet identifier where a new rule should be inserted.
   * @param ruleText The text of a new rule.
   * @param location Text position of a new rule in the target style sheet.
   */
  public suspend fun addRule(
    styleSheetId: StyleSheetId,
    ruleText: String,
    location: SourceRange,
  ): AddRuleResponse {
    val input = AddRuleRequest(styleSheetId, ruleText, location)
    return addRule(input)
  }

  /**
   * Returns all class names from specified stylesheet.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-collectClassNames)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun collectClassNames(input: CollectClassNamesRequest): CollectClassNamesResponse =
      session.request("CSS.collectClassNames", input)

  /**
   * Returns all class names from specified stylesheet.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-collectClassNames)
   */
  public suspend fun collectClassNames(styleSheetId: StyleSheetId): CollectClassNamesResponse {
    val input = CollectClassNamesRequest(styleSheetId)
    return collectClassNames(input)
  }

  /**
   * Creates a new special "via-inspector" stylesheet in the frame with given `frameId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-createStyleSheet)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun createStyleSheet(input: CreateStyleSheetRequest): CreateStyleSheetResponse =
      session.request("CSS.createStyleSheet", input)

  /**
   * Creates a new special "via-inspector" stylesheet in the frame with given `frameId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-createStyleSheet)
   *
   * @param frameId Identifier of the frame where "via-inspector" stylesheet should be created.
   */
  public suspend fun createStyleSheet(frameId: FrameId): CreateStyleSheetResponse {
    val input = CreateStyleSheetRequest(frameId)
    return createStyleSheet(input)
  }

  /**
   * Disables the CSS agent for the given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("CSS.disable", Unit)

  /**
   * Enables the CSS agent for the given page. Clients should not assume that the CSS agent has been
   * enabled until the result of this command is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("CSS.enable", Unit)

  /**
   * Ensures that the given node will have specified pseudo-classes whenever its style is computed
   * by
   * the browser.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-forcePseudoState)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun forcePseudoState(input: ForcePseudoStateRequest): ForcePseudoStateResponse =
      session.request("CSS.forcePseudoState", input)

  /**
   * Ensures that the given node will have specified pseudo-classes whenever its style is computed
   * by
   * the browser.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-forcePseudoState)
   *
   * @param nodeId The element id for which to force the pseudo state.
   * @param forcedPseudoClasses Element pseudo classes to force when computing the element's style.
   */
  public suspend fun forcePseudoState(nodeId: NodeId, forcedPseudoClasses: List<String>):
      ForcePseudoStateResponse {
    val input = ForcePseudoStateRequest(nodeId, forcedPseudoClasses)
    return forcePseudoState(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getBackgroundColors)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getBackgroundColors(input: GetBackgroundColorsRequest):
      GetBackgroundColorsResponse = session.request("CSS.getBackgroundColors", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getBackgroundColors)
   *
   * @param nodeId Id of the node to get background colors for.
   */
  public suspend fun getBackgroundColors(nodeId: NodeId): GetBackgroundColorsResponse {
    val input = GetBackgroundColorsRequest(nodeId)
    return getBackgroundColors(input)
  }

  /**
   * Returns the computed style for a DOM node identified by `nodeId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getComputedStyleForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getComputedStyleForNode(input: GetComputedStyleForNodeRequest):
      GetComputedStyleForNodeResponse = session.request("CSS.getComputedStyleForNode", input)

  /**
   * Returns the computed style for a DOM node identified by `nodeId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getComputedStyleForNode)
   */
  public suspend fun getComputedStyleForNode(nodeId: NodeId): GetComputedStyleForNodeResponse {
    val input = GetComputedStyleForNodeRequest(nodeId)
    return getComputedStyleForNode(input)
  }

  /**
   * Returns the styles defined inline (explicitly in the "style" attribute and implicitly, using
   * DOM
   * attributes) for a DOM node identified by `nodeId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getInlineStylesForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getInlineStylesForNode(input: GetInlineStylesForNodeRequest):
      GetInlineStylesForNodeResponse = session.request("CSS.getInlineStylesForNode", input)

  /**
   * Returns the styles defined inline (explicitly in the "style" attribute and implicitly, using
   * DOM
   * attributes) for a DOM node identified by `nodeId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getInlineStylesForNode)
   */
  public suspend fun getInlineStylesForNode(nodeId: NodeId): GetInlineStylesForNodeResponse {
    val input = GetInlineStylesForNodeRequest(nodeId)
    return getInlineStylesForNode(input)
  }

  /**
   * Returns requested styles for a DOM node identified by `nodeId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getMatchedStylesForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getMatchedStylesForNode(input: GetMatchedStylesForNodeRequest):
      GetMatchedStylesForNodeResponse = session.request("CSS.getMatchedStylesForNode", input)

  /**
   * Returns requested styles for a DOM node identified by `nodeId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getMatchedStylesForNode)
   */
  public suspend fun getMatchedStylesForNode(nodeId: NodeId): GetMatchedStylesForNodeResponse {
    val input = GetMatchedStylesForNodeRequest(nodeId)
    return getMatchedStylesForNode(input)
  }

  /**
   * Returns all media queries parsed by the rendering engine.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getMediaQueries)
   */
  public suspend fun getMediaQueries(): GetMediaQueriesResponse =
      session.request("CSS.getMediaQueries", Unit)

  /**
   * Requests information about platform fonts which we used to render child TextNodes in the given
   * node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getPlatformFontsForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getPlatformFontsForNode(input: GetPlatformFontsForNodeRequest):
      GetPlatformFontsForNodeResponse = session.request("CSS.getPlatformFontsForNode", input)

  /**
   * Requests information about platform fonts which we used to render child TextNodes in the given
   * node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getPlatformFontsForNode)
   */
  public suspend fun getPlatformFontsForNode(nodeId: NodeId): GetPlatformFontsForNodeResponse {
    val input = GetPlatformFontsForNodeRequest(nodeId)
    return getPlatformFontsForNode(input)
  }

  /**
   * Returns the current textual content for a stylesheet.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getStyleSheetText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getStyleSheetText(input: GetStyleSheetTextRequest): GetStyleSheetTextResponse =
      session.request("CSS.getStyleSheetText", input)

  /**
   * Returns the current textual content for a stylesheet.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getStyleSheetText)
   */
  public suspend fun getStyleSheetText(styleSheetId: StyleSheetId): GetStyleSheetTextResponse {
    val input = GetStyleSheetTextRequest(styleSheetId)
    return getStyleSheetText(input)
  }

  /**
   * Returns all layers parsed by the rendering engine for the tree scope of a node.
   * Given a DOM element identified by nodeId, getLayersForNode returns the root
   * layer for the nearest ancestor document or shadow root. The layer root contains
   * the full layer tree for the tree scope and their ordering.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getLayersForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getLayersForNode(input: GetLayersForNodeRequest): GetLayersForNodeResponse =
      session.request("CSS.getLayersForNode", input)

  /**
   * Returns all layers parsed by the rendering engine for the tree scope of a node.
   * Given a DOM element identified by nodeId, getLayersForNode returns the root
   * layer for the nearest ancestor document or shadow root. The layer root contains
   * the full layer tree for the tree scope and their ordering.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-getLayersForNode)
   */
  @ExperimentalChromeApi
  public suspend fun getLayersForNode(nodeId: NodeId): GetLayersForNodeResponse {
    val input = GetLayersForNodeRequest(nodeId)
    return getLayersForNode(input)
  }

  /**
   * Starts tracking the given computed styles for updates. The specified array of properties
   * replaces the one previously specified. Pass empty array to disable tracking.
   * Use takeComputedStyleUpdates to retrieve the list of nodes that had properties modified.
   * The changes to computed style properties are only tracked for nodes pushed to the front-end
   * by the DOM agent. If no changes to the tracked properties occur after the node has been pushed
   * to the front-end, no updates will be issued for the node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-trackComputedStyleUpdates)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun trackComputedStyleUpdates(input: TrackComputedStyleUpdatesRequest):
      TrackComputedStyleUpdatesResponse = session.request("CSS.trackComputedStyleUpdates", input)

  /**
   * Starts tracking the given computed styles for updates. The specified array of properties
   * replaces the one previously specified. Pass empty array to disable tracking.
   * Use takeComputedStyleUpdates to retrieve the list of nodes that had properties modified.
   * The changes to computed style properties are only tracked for nodes pushed to the front-end
   * by the DOM agent. If no changes to the tracked properties occur after the node has been pushed
   * to the front-end, no updates will be issued for the node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-trackComputedStyleUpdates)
   */
  @ExperimentalChromeApi
  public suspend fun trackComputedStyleUpdates(propertiesToTrack: List<CSSComputedStyleProperty>):
      TrackComputedStyleUpdatesResponse {
    val input = TrackComputedStyleUpdatesRequest(propertiesToTrack)
    return trackComputedStyleUpdates(input)
  }

  /**
   * Polls the next batch of computed style updates.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-takeComputedStyleUpdates)
   */
  @ExperimentalChromeApi
  public suspend fun takeComputedStyleUpdates(): TakeComputedStyleUpdatesResponse =
      session.request("CSS.takeComputedStyleUpdates", Unit)

  /**
   * Find a rule with the given active property for the given node and set the new value for this
   * property
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setEffectivePropertyValueForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend
      fun setEffectivePropertyValueForNode(input: SetEffectivePropertyValueForNodeRequest):
      SetEffectivePropertyValueForNodeResponse =
      session.request("CSS.setEffectivePropertyValueForNode", input)

  /**
   * Find a rule with the given active property for the given node and set the new value for this
   * property
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setEffectivePropertyValueForNode)
   *
   * @param nodeId The element id for which to set property.
   */
  public suspend fun setEffectivePropertyValueForNode(
    nodeId: NodeId,
    propertyName: String,
    `value`: String,
  ): SetEffectivePropertyValueForNodeResponse {
    val input = SetEffectivePropertyValueForNodeRequest(nodeId, propertyName, value)
    return setEffectivePropertyValueForNode(input)
  }

  /**
   * Modifies the keyframe rule key text.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setKeyframeKey)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setKeyframeKey(input: SetKeyframeKeyRequest): SetKeyframeKeyResponse =
      session.request("CSS.setKeyframeKey", input)

  /**
   * Modifies the keyframe rule key text.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setKeyframeKey)
   */
  public suspend fun setKeyframeKey(
    styleSheetId: StyleSheetId,
    range: SourceRange,
    keyText: String,
  ): SetKeyframeKeyResponse {
    val input = SetKeyframeKeyRequest(styleSheetId, range, keyText)
    return setKeyframeKey(input)
  }

  /**
   * Modifies the rule selector.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setMediaText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setMediaText(input: SetMediaTextRequest): SetMediaTextResponse =
      session.request("CSS.setMediaText", input)

  /**
   * Modifies the rule selector.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setMediaText)
   */
  public suspend fun setMediaText(
    styleSheetId: StyleSheetId,
    range: SourceRange,
    text: String,
  ): SetMediaTextResponse {
    val input = SetMediaTextRequest(styleSheetId, range, text)
    return setMediaText(input)
  }

  /**
   * Modifies the expression of a container query.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setContainerQueryText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setContainerQueryText(input: SetContainerQueryTextRequest):
      SetContainerQueryTextResponse = session.request("CSS.setContainerQueryText", input)

  /**
   * Modifies the expression of a container query.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setContainerQueryText)
   */
  @ExperimentalChromeApi
  public suspend fun setContainerQueryText(
    styleSheetId: StyleSheetId,
    range: SourceRange,
    text: String,
  ): SetContainerQueryTextResponse {
    val input = SetContainerQueryTextRequest(styleSheetId, range, text)
    return setContainerQueryText(input)
  }

  /**
   * Modifies the expression of a supports at-rule.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setSupportsText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setSupportsText(input: SetSupportsTextRequest): SetSupportsTextResponse =
      session.request("CSS.setSupportsText", input)

  /**
   * Modifies the expression of a supports at-rule.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setSupportsText)
   */
  @ExperimentalChromeApi
  public suspend fun setSupportsText(
    styleSheetId: StyleSheetId,
    range: SourceRange,
    text: String,
  ): SetSupportsTextResponse {
    val input = SetSupportsTextRequest(styleSheetId, range, text)
    return setSupportsText(input)
  }

  /**
   * Modifies the expression of a scope at-rule.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setScopeText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setScopeText(input: SetScopeTextRequest): SetScopeTextResponse =
      session.request("CSS.setScopeText", input)

  /**
   * Modifies the expression of a scope at-rule.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setScopeText)
   */
  @ExperimentalChromeApi
  public suspend fun setScopeText(
    styleSheetId: StyleSheetId,
    range: SourceRange,
    text: String,
  ): SetScopeTextResponse {
    val input = SetScopeTextRequest(styleSheetId, range, text)
    return setScopeText(input)
  }

  /**
   * Modifies the rule selector.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setRuleSelector)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setRuleSelector(input: SetRuleSelectorRequest): SetRuleSelectorResponse =
      session.request("CSS.setRuleSelector", input)

  /**
   * Modifies the rule selector.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setRuleSelector)
   */
  public suspend fun setRuleSelector(
    styleSheetId: StyleSheetId,
    range: SourceRange,
    selector: String,
  ): SetRuleSelectorResponse {
    val input = SetRuleSelectorRequest(styleSheetId, range, selector)
    return setRuleSelector(input)
  }

  /**
   * Sets the new stylesheet text.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setStyleSheetText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setStyleSheetText(input: SetStyleSheetTextRequest): SetStyleSheetTextResponse =
      session.request("CSS.setStyleSheetText", input)

  /**
   * Sets the new stylesheet text.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setStyleSheetText)
   */
  public suspend fun setStyleSheetText(styleSheetId: StyleSheetId, text: String):
      SetStyleSheetTextResponse {
    val input = SetStyleSheetTextRequest(styleSheetId, text)
    return setStyleSheetText(input)
  }

  /**
   * Applies specified style edits one after another in the given order.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setStyleTexts)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setStyleTexts(input: SetStyleTextsRequest): SetStyleTextsResponse =
      session.request("CSS.setStyleTexts", input)

  /**
   * Applies specified style edits one after another in the given order.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setStyleTexts)
   */
  public suspend fun setStyleTexts(edits: List<StyleDeclarationEdit>): SetStyleTextsResponse {
    val input = SetStyleTextsRequest(edits)
    return setStyleTexts(input)
  }

  /**
   * Enables the selector recording.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-startRuleUsageTracking)
   */
  public suspend fun startRuleUsageTracking(): StartRuleUsageTrackingResponse =
      session.request("CSS.startRuleUsageTracking", Unit)

  /**
   * Stop tracking rule usage and return the list of rules that were used since last call to
   * `takeCoverageDelta` (or since start of coverage instrumentation)
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-stopRuleUsageTracking)
   */
  public suspend fun stopRuleUsageTracking(): StopRuleUsageTrackingResponse =
      session.request("CSS.stopRuleUsageTracking", Unit)

  /**
   * Obtain list of rules that became used since last call to this method (or since start of
   * coverage
   * instrumentation)
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-takeCoverageDelta)
   */
  public suspend fun takeCoverageDelta(): TakeCoverageDeltaResponse =
      session.request("CSS.takeCoverageDelta", Unit)

  /**
   * Enables/disables rendering of local CSS fonts (enabled by default).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setLocalFontsEnabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setLocalFontsEnabled(input: SetLocalFontsEnabledRequest):
      SetLocalFontsEnabledResponse = session.request("CSS.setLocalFontsEnabled", input)

  /**
   * Enables/disables rendering of local CSS fonts (enabled by default).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CSS/#method-setLocalFontsEnabled)
   *
   * @param enabled Whether rendering of local fonts is enabled.
   */
  @ExperimentalChromeApi
  public suspend fun setLocalFontsEnabled(enabled: Boolean): SetLocalFontsEnabledResponse {
    val input = SetLocalFontsEnabledRequest(enabled)
    return setLocalFontsEnabled(input)
  }
}
