@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.database

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.database.events.DatabaseEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [DatabaseDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [DatabaseDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [DatabaseDomain.executeSQL] command.
 */
@Serializable
public data class ExecuteSQLRequest(
  public val databaseId: DatabaseId,
  public val query: String,
)

/**
 * Response type for the [DatabaseDomain.executeSQL] command.
 */
@Serializable
public data class ExecuteSQLResponse(
  public val columnNames: List<String>? = null,
  public val values: List<JsonElement>? = null,
  public val sqlError: Error? = null,
)

/**
 * Request object containing input parameters for the [DatabaseDomain.getDatabaseTableNames]
 * command.
 */
@Serializable
public data class GetDatabaseTableNamesRequest(
  public val databaseId: DatabaseId,
)

/**
 * Response type for the [DatabaseDomain.getDatabaseTableNames] command.
 */
@Serializable
public data class GetDatabaseTableNamesResponse(
  public val tableNames: List<String>,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database)
 */
@ExperimentalChromeApi
public class DatabaseDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DatabaseEvent>> =
      mapOf(
      "Database.addDatabase" to serializer<DatabaseEvent.AddDatabase>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DatabaseEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#event-addDatabase)
   */
  public fun addDatabaseEvents(): Flow<DatabaseEvent.AddDatabase> =
      session.typedEvents("Database.addDatabase")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("addDatabaseEvents()"),
  )
  public fun addDatabase(): Flow<DatabaseEvent.AddDatabase> = addDatabaseEvents()

  /**
   * Disables database tracking, prevents database events from being sent to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Database.disable", Unit)

  /**
   * Enables database tracking, database events will now be delivered to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Database.enable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-executeSQL)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun executeSQL(input: ExecuteSQLRequest): ExecuteSQLResponse =
      session.request("Database.executeSQL", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-executeSQL)
   */
  public suspend fun executeSQL(databaseId: DatabaseId, query: String): ExecuteSQLResponse {
    val input = ExecuteSQLRequest(databaseId, query)
    return executeSQL(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-getDatabaseTableNames)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getDatabaseTableNames(input: GetDatabaseTableNamesRequest):
      GetDatabaseTableNamesResponse = session.request("Database.getDatabaseTableNames", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-getDatabaseTableNames)
   */
  public suspend fun getDatabaseTableNames(databaseId: DatabaseId): GetDatabaseTableNamesResponse {
    val input = GetDatabaseTableNamesRequest(databaseId)
    return getDatabaseTableNames(input)
  }
}
