@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.debugger

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.debugger.events.DebuggerEvent
import org.hildan.chrome.devtools.domains.runtime.CallArgument
import org.hildan.chrome.devtools.domains.runtime.ExceptionDetails
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.domains.runtime.ScriptId
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.domains.runtime.StackTraceId
import org.hildan.chrome.devtools.domains.runtime.TimeDelta
import org.hildan.chrome.devtools.domains.runtime.UniqueDebuggerId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [DebuggerDomain.continueToLocation] command.
 */
@Serializable
public data class ContinueToLocationRequest(
  /**
   * Location to continue to.
   */
  public val location: Location,
  public val targetCallFrames: TargetCallFrames? = null,
) {
  public class Builder(
    /**
     * Location to continue to.
     */
    public val location: Location,
  ) {
    public var targetCallFrames: TargetCallFrames? = null

    public fun build(): ContinueToLocationRequest = ContinueToLocationRequest(location,
        targetCallFrames)
  }
}

/**
 * A dummy response object for the [DebuggerDomain.continueToLocation] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ContinueToLocationResponse

/**
 * A dummy response object for the [DebuggerDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * The maximum size in bytes of collected scripts (not referenced by other heap objects)
   * the debugger can hold. Puts no limit if parameter is omitted.
   */
  @ExperimentalChromeApi
  public val maxScriptsCacheSize: Double? = null,
) {
  public class Builder() {
    /**
     * The maximum size in bytes of collected scripts (not referenced by other heap objects)
     * the debugger can hold. Puts no limit if parameter is omitted.
     */
    @ExperimentalChromeApi
    public var maxScriptsCacheSize: Double? = null

    public fun build(): EnableRequest = EnableRequest(maxScriptsCacheSize)
  }
}

/**
 * Response type for the [DebuggerDomain.enable] command.
 */
@Serializable
public data class EnableResponse(
  /**
   * Unique identifier of the debugger.
   */
  @ExperimentalChromeApi
  public val debuggerId: UniqueDebuggerId? = null,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.evaluateOnCallFrame] command.
 */
@Serializable
public data class EvaluateOnCallFrameRequest(
  /**
   * Call frame identifier to evaluate on.
   */
  public val callFrameId: CallFrameId,
  /**
   * Expression to evaluate.
   */
  public val expression: String,
  /**
   * String object group name to put result into (allows rapid releasing resulting object handles
   * using `releaseObjectGroup`).
   */
  public val objectGroup: String? = null,
  /**
   * Specifies whether command line API should be available to the evaluated expression, defaults
   * to false.
   */
  public val includeCommandLineAPI: Boolean? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Whether the result is expected to be a JSON object that should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * Whether to throw an exception if side effect cannot be ruled out during evaluation.
   */
  public val throwOnSideEffect: Boolean? = null,
  /**
   * Terminate execution after timing out (number of milliseconds).
   */
  @ExperimentalChromeApi
  public val timeout: TimeDelta? = null,
) {
  public class Builder(
    /**
     * Call frame identifier to evaluate on.
     */
    public val callFrameId: CallFrameId,
    /**
     * Expression to evaluate.
     */
    public val expression: String,
  ) {
    /**
     * String object group name to put result into (allows rapid releasing resulting object handles
     * using `releaseObjectGroup`).
     */
    public var objectGroup: String? = null

    /**
     * Specifies whether command line API should be available to the evaluated expression, defaults
     * to false.
     */
    public var includeCommandLineAPI: Boolean? = null

    /**
     * In silent mode exceptions thrown during evaluation are not reported and do not pause
     * execution. Overrides `setPauseOnException` state.
     */
    public var silent: Boolean? = null

    /**
     * Whether the result is expected to be a JSON object that should be sent by value.
     */
    public var returnByValue: Boolean? = null

    /**
     * Whether preview should be generated for the result.
     */
    @ExperimentalChromeApi
    public var generatePreview: Boolean? = null

    /**
     * Whether to throw an exception if side effect cannot be ruled out during evaluation.
     */
    public var throwOnSideEffect: Boolean? = null

    /**
     * Terminate execution after timing out (number of milliseconds).
     */
    @ExperimentalChromeApi
    public var timeout: TimeDelta? = null

    public fun build(): EvaluateOnCallFrameRequest = EvaluateOnCallFrameRequest(callFrameId,
        expression, objectGroup, includeCommandLineAPI, silent, returnByValue, generatePreview,
        throwOnSideEffect, timeout)
  }
}

/**
 * Response type for the [DebuggerDomain.evaluateOnCallFrame] command.
 */
@Serializable
public data class EvaluateOnCallFrameResponse(
  /**
   * Object wrapper for the evaluation result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getPossibleBreakpoints]
 * command.
 */
@Serializable
public data class GetPossibleBreakpointsRequest(
  /**
   * Start of range to search possible breakpoint locations in.
   */
  public val start: Location,
  /**
   * End of range to search possible breakpoint locations in (excluding). When not specified, end
   * of scripts is used as end of range.
   */
  public val end: Location? = null,
  /**
   * Only consider locations which are in the same (non-nested) function as start.
   */
  public val restrictToFunction: Boolean? = null,
) {
  public class Builder(
    /**
     * Start of range to search possible breakpoint locations in.
     */
    public val start: Location,
  ) {
    /**
     * End of range to search possible breakpoint locations in (excluding). When not specified, end
     * of scripts is used as end of range.
     */
    public var end: Location? = null

    /**
     * Only consider locations which are in the same (non-nested) function as start.
     */
    public var restrictToFunction: Boolean? = null

    public fun build(): GetPossibleBreakpointsRequest = GetPossibleBreakpointsRequest(start, end,
        restrictToFunction)
  }
}

/**
 * Response type for the [DebuggerDomain.getPossibleBreakpoints] command.
 */
@Serializable
public data class GetPossibleBreakpointsResponse(
  /**
   * List of the possible breakpoint locations.
   */
  public val locations: List<BreakLocation>,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getScriptSource] command.
 */
@Serializable
public data class GetScriptSourceRequest(
  /**
   * Id of the script to get source for.
   */
  public val scriptId: ScriptId,
)

/**
 * Response type for the [DebuggerDomain.getScriptSource] command.
 */
@Serializable
public data class GetScriptSourceResponse(
  /**
   * Script source (empty in case of Wasm bytecode).
   */
  public val scriptSource: String,
  /**
   * Wasm bytecode. (Encoded as a base64 string when passed over JSON)
   */
  public val bytecode: String? = null,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.disassembleWasmModule]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class DisassembleWasmModuleRequest(
  /**
   * Id of the script to disassemble
   */
  public val scriptId: ScriptId,
)

/**
 * Response type for the [DebuggerDomain.disassembleWasmModule] command.
 */
@Serializable
@ExperimentalChromeApi
public data class DisassembleWasmModuleResponse(
  /**
   * For large modules, return a stream from which additional chunks of
   * disassembly can be read successively.
   */
  public val streamId: String? = null,
  /**
   * The total number of lines in the disassembly text.
   */
  public val totalNumberOfLines: Int,
  /**
   * The offsets of all function bodies, in the format [start1, end1,
   * start2, end2, ...] where all ends are exclusive.
   */
  public val functionBodyOffsets: List<Int>,
  /**
   * The first chunk of disassembly.
   */
  public val chunk: WasmDisassemblyChunk,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.nextWasmDisassemblyChunk]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class NextWasmDisassemblyChunkRequest(
  public val streamId: String,
)

/**
 * Response type for the [DebuggerDomain.nextWasmDisassemblyChunk] command.
 */
@Serializable
@ExperimentalChromeApi
public data class NextWasmDisassemblyChunkResponse(
  /**
   * The next chunk of disassembly.
   */
  public val chunk: WasmDisassemblyChunk,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getWasmBytecode] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetWasmBytecodeRequest(
  /**
   * Id of the Wasm script to get source for.
   */
  public val scriptId: ScriptId,
)

/**
 * Response type for the [DebuggerDomain.getWasmBytecode] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetWasmBytecodeResponse(
  /**
   * Script source. (Encoded as a base64 string when passed over JSON)
   */
  public val bytecode: String,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getStackTrace] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetStackTraceRequest(
  public val stackTraceId: StackTraceId,
)

/**
 * Response type for the [DebuggerDomain.getStackTrace] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetStackTraceResponse(
  public val stackTrace: StackTrace,
)

/**
 * A dummy response object for the [DebuggerDomain.pause] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object PauseResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.pauseOnAsyncCall] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class PauseOnAsyncCallRequest(
  /**
   * Debugger will pause when async call with given stack trace is started.
   */
  public val parentStackTraceId: StackTraceId,
)

/**
 * A dummy response object for the [DebuggerDomain.pauseOnAsyncCall] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public object PauseOnAsyncCallResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.removeBreakpoint] command.
 */
@Serializable
public data class RemoveBreakpointRequest(
  public val breakpointId: BreakpointId,
)

/**
 * A dummy response object for the [DebuggerDomain.removeBreakpoint] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object RemoveBreakpointResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.restartFrame] command.
 */
@Serializable
public data class RestartFrameRequest(
  /**
   * Call frame identifier to evaluate on.
   */
  public val callFrameId: CallFrameId,
  /**
   * The `mode` parameter must be present and set to 'StepInto', otherwise
   * `restartFrame` will error out.
   */
  @ExperimentalChromeApi
  public val mode: FrameMode? = null,
) {
  public class Builder(
    /**
     * Call frame identifier to evaluate on.
     */
    public val callFrameId: CallFrameId,
  ) {
    /**
     * The `mode` parameter must be present and set to 'StepInto', otherwise
     * `restartFrame` will error out.
     */
    @ExperimentalChromeApi
    public var mode: FrameMode? = null

    public fun build(): RestartFrameRequest = RestartFrameRequest(callFrameId, mode)
  }
}

/**
 * Response type for the [DebuggerDomain.restartFrame] command.
 */
@Serializable
public data class RestartFrameResponse(
  /**
   * New stack trace.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val callFrames: List<CallFrame>,
  /**
   * Async stack trace, if any.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val asyncStackTrace: StackTrace? = null,
  /**
   * Async stack trace, if any.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val asyncStackTraceId: StackTraceId? = null,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.resume] command.
 */
@Serializable
public data class ResumeRequest(
  /**
   * Set to true to terminate execution upon resuming execution. In contrast
   * to Runtime.terminateExecution, this will allows to execute further
   * JavaScript (i.e. via evaluation) until execution of the paused code
   * is actually resumed, at which point termination is triggered.
   * If execution is currently not paused, this parameter has no effect.
   */
  public val terminateOnResume: Boolean? = null,
) {
  public class Builder() {
    /**
     * Set to true to terminate execution upon resuming execution. In contrast
     * to Runtime.terminateExecution, this will allows to execute further
     * JavaScript (i.e. via evaluation) until execution of the paused code
     * is actually resumed, at which point termination is triggered.
     * If execution is currently not paused, this parameter has no effect.
     */
    public var terminateOnResume: Boolean? = null

    public fun build(): ResumeRequest = ResumeRequest(terminateOnResume)
  }
}

/**
 * A dummy response object for the [DebuggerDomain.resume] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object ResumeResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.searchInContent] command.
 */
@Serializable
public data class SearchInContentRequest(
  /**
   * Id of the script to search in.
   */
  public val scriptId: ScriptId,
  /**
   * String to search for.
   */
  public val query: String,
  /**
   * If true, search is case sensitive.
   */
  public val caseSensitive: Boolean? = null,
  /**
   * If true, treats string parameter as regex.
   */
  public val isRegex: Boolean? = null,
) {
  public class Builder(
    /**
     * Id of the script to search in.
     */
    public val scriptId: ScriptId,
    /**
     * String to search for.
     */
    public val query: String,
  ) {
    /**
     * If true, search is case sensitive.
     */
    public var caseSensitive: Boolean? = null

    /**
     * If true, treats string parameter as regex.
     */
    public var isRegex: Boolean? = null

    public fun build(): SearchInContentRequest = SearchInContentRequest(scriptId, query,
        caseSensitive, isRegex)
  }
}

/**
 * Response type for the [DebuggerDomain.searchInContent] command.
 */
@Serializable
public data class SearchInContentResponse(
  /**
   * List of search matches.
   */
  public val result: List<SearchMatch>,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setAsyncCallStackDepth]
 * command.
 */
@Serializable
public data class SetAsyncCallStackDepthRequest(
  /**
   * Maximum depth of async call stacks. Setting to `0` will effectively disable collecting async
   * call stacks (default).
   */
  public val maxDepth: Int,
)

/**
 * A dummy response object for the [DebuggerDomain.setAsyncCallStackDepth] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetAsyncCallStackDepthResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setBlackboxPatterns] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBlackboxPatternsRequest(
  /**
   * Array of regexps that will be used to check script url for blackbox state.
   */
  public val patterns: List<String>,
)

/**
 * A dummy response object for the [DebuggerDomain.setBlackboxPatterns] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetBlackboxPatternsResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setBlackboxedRanges] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBlackboxedRangesRequest(
  /**
   * Id of the script.
   */
  public val scriptId: ScriptId,
  public val positions: List<ScriptPosition>,
)

/**
 * A dummy response object for the [DebuggerDomain.setBlackboxedRanges] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetBlackboxedRangesResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpoint] command.
 */
@Serializable
public data class SetBreakpointRequest(
  /**
   * Location to set breakpoint in.
   */
  public val location: Location,
  /**
   * Expression to use as a breakpoint condition. When specified, debugger will only stop on the
   * breakpoint if this expression evaluates to true.
   */
  public val condition: String? = null,
) {
  public class Builder(
    /**
     * Location to set breakpoint in.
     */
    public val location: Location,
  ) {
    /**
     * Expression to use as a breakpoint condition. When specified, debugger will only stop on the
     * breakpoint if this expression evaluates to true.
     */
    public var condition: String? = null

    public fun build(): SetBreakpointRequest = SetBreakpointRequest(location, condition)
  }
}

/**
 * Response type for the [DebuggerDomain.setBreakpoint] command.
 */
@Serializable
public data class SetBreakpointResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId,
  /**
   * Location this breakpoint resolved into.
   */
  public val actualLocation: Location,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setInstrumentationBreakpoint]
 * command.
 */
@Serializable
public data class SetInstrumentationBreakpointRequest(
  /**
   * Instrumentation name.
   */
  public val instrumentation: BreakpointInstrumentation,
)

/**
 * Response type for the [DebuggerDomain.setInstrumentationBreakpoint] command.
 */
@Serializable
public data class SetInstrumentationBreakpointResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpointByUrl] command.
 */
@Serializable
public data class SetBreakpointByUrlRequest(
  /**
   * Line number to set breakpoint at.
   */
  public val lineNumber: Int,
  /**
   * URL of the resources to set breakpoint on.
   */
  public val url: String? = null,
  /**
   * Regex pattern for the URLs of the resources to set breakpoints on. Either `url` or
   * `urlRegex` must be specified.
   */
  public val urlRegex: String? = null,
  /**
   * Script hash of the resources to set breakpoint on.
   */
  public val scriptHash: String? = null,
  /**
   * Offset in the line to set breakpoint at.
   */
  public val columnNumber: Int? = null,
  /**
   * Expression to use as a breakpoint condition. When specified, debugger will only stop on the
   * breakpoint if this expression evaluates to true.
   */
  public val condition: String? = null,
) {
  public class Builder(
    /**
     * Line number to set breakpoint at.
     */
    public val lineNumber: Int,
  ) {
    /**
     * URL of the resources to set breakpoint on.
     */
    public var url: String? = null

    /**
     * Regex pattern for the URLs of the resources to set breakpoints on. Either `url` or
     * `urlRegex` must be specified.
     */
    public var urlRegex: String? = null

    /**
     * Script hash of the resources to set breakpoint on.
     */
    public var scriptHash: String? = null

    /**
     * Offset in the line to set breakpoint at.
     */
    public var columnNumber: Int? = null

    /**
     * Expression to use as a breakpoint condition. When specified, debugger will only stop on the
     * breakpoint if this expression evaluates to true.
     */
    public var condition: String? = null

    public fun build(): SetBreakpointByUrlRequest = SetBreakpointByUrlRequest(lineNumber, url,
        urlRegex, scriptHash, columnNumber, condition)
  }
}

/**
 * Response type for the [DebuggerDomain.setBreakpointByUrl] command.
 */
@Serializable
public data class SetBreakpointByUrlResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId,
  /**
   * List of the locations this breakpoint resolved into upon addition.
   */
  public val locations: List<Location>,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpointOnFunctionCall]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBreakpointOnFunctionCallRequest(
  /**
   * Function object id.
   */
  public val objectId: RemoteObjectId,
  /**
   * Expression to use as a breakpoint condition. When specified, debugger will
   * stop on the breakpoint if this expression evaluates to true.
   */
  public val condition: String? = null,
) {
  public class Builder(
    /**
     * Function object id.
     */
    public val objectId: RemoteObjectId,
  ) {
    /**
     * Expression to use as a breakpoint condition. When specified, debugger will
     * stop on the breakpoint if this expression evaluates to true.
     */
    public var condition: String? = null

    public fun build(): SetBreakpointOnFunctionCallRequest =
        SetBreakpointOnFunctionCallRequest(objectId, condition)
  }
}

/**
 * Response type for the [DebuggerDomain.setBreakpointOnFunctionCall] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBreakpointOnFunctionCallResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpointsActive] command.
 */
@Serializable
public data class SetBreakpointsActiveRequest(
  /**
   * New value for breakpoints active state.
   */
  public val active: Boolean,
)

/**
 * A dummy response object for the [DebuggerDomain.setBreakpointsActive] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetBreakpointsActiveResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setPauseOnExceptions] command.
 */
@Serializable
public data class SetPauseOnExceptionsRequest(
  /**
   * Pause on exceptions mode.
   */
  public val state: PauseOnExceptionsState,
)

/**
 * A dummy response object for the [DebuggerDomain.setPauseOnExceptions] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetPauseOnExceptionsResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setReturnValue] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetReturnValueRequest(
  /**
   * New return value.
   */
  public val newValue: CallArgument,
)

/**
 * A dummy response object for the [DebuggerDomain.setReturnValue] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetReturnValueResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setScriptSource] command.
 */
@Serializable
public data class SetScriptSourceRequest(
  /**
   * Id of the script to edit.
   */
  public val scriptId: ScriptId,
  /**
   * New content of the script.
   */
  public val scriptSource: String,
  /**
   * If true the change will not actually be applied. Dry run may be used to get result
   * description without actually modifying the code.
   */
  public val dryRun: Boolean? = null,
  /**
   * If true, then `scriptSource` is allowed to change the function on top of the stack
   * as long as the top-most stack frame is the only activation of that function.
   */
  @ExperimentalChromeApi
  public val allowTopFrameEditing: Boolean? = null,
) {
  public class Builder(
    /**
     * Id of the script to edit.
     */
    public val scriptId: ScriptId,
    /**
     * New content of the script.
     */
    public val scriptSource: String,
  ) {
    /**
     * If true the change will not actually be applied. Dry run may be used to get result
     * description without actually modifying the code.
     */
    public var dryRun: Boolean? = null

    /**
     * If true, then `scriptSource` is allowed to change the function on top of the stack
     * as long as the top-most stack frame is the only activation of that function.
     */
    @ExperimentalChromeApi
    public var allowTopFrameEditing: Boolean? = null

    public fun build(): SetScriptSourceRequest = SetScriptSourceRequest(scriptId, scriptSource,
        dryRun, allowTopFrameEditing)
  }
}

/**
 * Response type for the [DebuggerDomain.setScriptSource] command.
 */
@Serializable
public data class SetScriptSourceResponse(
  /**
   * New stack trace in case editing has happened while VM was stopped.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val callFrames: List<CallFrame>? = null,
  /**
   * Whether current call stack  was modified after applying the changes.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val stackChanged: Boolean? = null,
  /**
   * Async stack trace, if any.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val asyncStackTrace: StackTrace? = null,
  /**
   * Async stack trace, if any.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val asyncStackTraceId: StackTraceId? = null,
  /**
   * Whether the operation was successful or not. Only `Ok` denotes a
   * successful live edit while the other enum variants denote why
   * the live edit failed.
   */
  @ExperimentalChromeApi
  public val status: ScriptSourceStatus? = null,
  /**
   * Exception details if any. Only present when `status` is `CompileError`.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setSkipAllPauses] command.
 */
@Serializable
public data class SetSkipAllPausesRequest(
  /**
   * New value for skip pauses state.
   */
  public val skip: Boolean,
)

/**
 * A dummy response object for the [DebuggerDomain.setSkipAllPauses] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetSkipAllPausesResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.setVariableValue] command.
 */
@Serializable
public data class SetVariableValueRequest(
  /**
   * 0-based number of scope as was listed in scope chain. Only 'local', 'closure' and 'catch'
   * scope types are allowed. Other scopes could be manipulated manually.
   */
  public val scopeNumber: Int,
  /**
   * Variable name.
   */
  public val variableName: String,
  /**
   * New variable value.
   */
  public val newValue: CallArgument,
  /**
   * Id of callframe that holds variable.
   */
  public val callFrameId: CallFrameId,
)

/**
 * A dummy response object for the [DebuggerDomain.setVariableValue] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetVariableValueResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.stepInto] command.
 */
@Serializable
public data class StepIntoRequest(
  /**
   * Debugger will pause on the execution of the first async task which was scheduled
   * before next pause.
   */
  @ExperimentalChromeApi
  public val breakOnAsyncCall: Boolean? = null,
  /**
   * The skipList specifies location ranges that should be skipped on step into.
   */
  @ExperimentalChromeApi
  public val skipList: List<LocationRange>? = null,
) {
  public class Builder() {
    /**
     * Debugger will pause on the execution of the first async task which was scheduled
     * before next pause.
     */
    @ExperimentalChromeApi
    public var breakOnAsyncCall: Boolean? = null

    /**
     * The skipList specifies location ranges that should be skipped on step into.
     */
    @ExperimentalChromeApi
    public var skipList: List<LocationRange>? = null

    public fun build(): StepIntoRequest = StepIntoRequest(breakOnAsyncCall, skipList)
  }
}

/**
 * A dummy response object for the [DebuggerDomain.stepInto] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object StepIntoResponse

/**
 * A dummy response object for the [DebuggerDomain.stepOut] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object StepOutResponse

/**
 * Request object containing input parameters for the [DebuggerDomain.stepOver] command.
 */
@Serializable
public data class StepOverRequest(
  /**
   * The skipList specifies location ranges that should be skipped on step over.
   */
  @ExperimentalChromeApi
  public val skipList: List<LocationRange>? = null,
) {
  public class Builder() {
    /**
     * The skipList specifies location ranges that should be skipped on step over.
     */
    @ExperimentalChromeApi
    public var skipList: List<LocationRange>? = null

    public fun build(): StepOverRequest = StepOverRequest(skipList)
  }
}

/**
 * A dummy response object for the [DebuggerDomain.stepOver] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object StepOverResponse

/**
 * Debugger domain exposes JavaScript debugging capabilities. It allows setting and removing
 * breakpoints, stepping through execution, exploring stack traces, etc.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger)
 */
public class DebuggerDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DebuggerEvent>> =
      mapOf(
      "Debugger.breakpointResolved" to serializer<DebuggerEvent.BreakpointResolved>(),
      "Debugger.paused" to serializer<DebuggerEvent.Paused>(),
      "Debugger.resumed" to serializer<DebuggerEvent.Resumed>(),
      "Debugger.scriptFailedToParse" to serializer<DebuggerEvent.ScriptFailedToParse>(),
      "Debugger.scriptParsed" to serializer<DebuggerEvent.ScriptParsed>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DebuggerEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fired when breakpoint is resolved to an actual script and location.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#event-breakpointResolved)
   */
  public fun breakpointResolvedEvents(): Flow<DebuggerEvent.BreakpointResolved> =
      session.typedEvents("Debugger.breakpointResolved")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("breakpointResolvedEvents()"),
  )
  public fun breakpointResolved(): Flow<DebuggerEvent.BreakpointResolved> =
      breakpointResolvedEvents()

  /**
   * Fired when the virtual machine stopped on breakpoint or exception or any other stop criteria.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#event-paused)
   */
  public fun pausedEvents(): Flow<DebuggerEvent.Paused> = session.typedEvents("Debugger.paused")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("pausedEvents()"),
  )
  public fun paused(): Flow<DebuggerEvent.Paused> = pausedEvents()

  /**
   * Fired when the virtual machine resumed execution.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#event-resumed)
   */
  public fun resumedEvents(): Flow<DebuggerEvent.Resumed> = session.typedEvents("Debugger.resumed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("resumedEvents()"),
  )
  public fun resumed(): Flow<DebuggerEvent.Resumed> = resumedEvents()

  /**
   * Fired when virtual machine fails to parse the script.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#event-scriptFailedToParse)
   */
  public fun scriptFailedToParseEvents(): Flow<DebuggerEvent.ScriptFailedToParse> =
      session.typedEvents("Debugger.scriptFailedToParse")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("scriptFailedToParseEvents()"),
  )
  public fun scriptFailedToParse(): Flow<DebuggerEvent.ScriptFailedToParse> =
      scriptFailedToParseEvents()

  /**
   * Fired when virtual machine parses script. This event is also fired for all known and
   * uncollected
   * scripts upon enabling debugger.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#event-scriptParsed)
   */
  public fun scriptParsedEvents(): Flow<DebuggerEvent.ScriptParsed> =
      session.typedEvents("Debugger.scriptParsed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("scriptParsedEvents()"),
  )
  public fun scriptParsed(): Flow<DebuggerEvent.ScriptParsed> = scriptParsedEvents()

  /**
   * Continues execution until specific location is reached.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-continueToLocation)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun continueToLocation(input: ContinueToLocationRequest):
      ContinueToLocationResponse = session.request("Debugger.continueToLocation", input)

  /**
   * Continues execution until specific location is reached.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-continueToLocation)
   *
   * @param location Location to continue to.
   */
  @JvmOverloads
  public suspend inline fun continueToLocation(location: Location,
      optionalArgs: ContinueToLocationRequest.Builder.() -> Unit = {}): ContinueToLocationResponse {
    val builder = ContinueToLocationRequest.Builder(location)
    val input = builder.apply(optionalArgs).build()
    return continueToLocation(input)
  }

  /**
   * Disables debugger for given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Debugger.disable", Unit)

  /**
   * Enables debugger for the given page. Clients should not assume that the debugging has been
   * enabled until the result for this command is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse =
      session.request("Debugger.enable", input)

  /**
   * Enables debugger for the given page. Clients should not assume that the debugging has been
   * enabled until the result for this command is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Evaluates expression on a given call frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-evaluateOnCallFrame)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun evaluateOnCallFrame(input: EvaluateOnCallFrameRequest):
      EvaluateOnCallFrameResponse = session.request("Debugger.evaluateOnCallFrame", input)

  /**
   * Evaluates expression on a given call frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-evaluateOnCallFrame)
   *
   * @param callFrameId Call frame identifier to evaluate on.
   * @param expression Expression to evaluate.
   */
  @JvmOverloads
  public suspend inline fun evaluateOnCallFrame(
    callFrameId: CallFrameId,
    expression: String,
    optionalArgs: EvaluateOnCallFrameRequest.Builder.() -> Unit = {},
  ): EvaluateOnCallFrameResponse {
    val builder = EvaluateOnCallFrameRequest.Builder(callFrameId, expression)
    val input = builder.apply(optionalArgs).build()
    return evaluateOnCallFrame(input)
  }

  /**
   * Returns possible locations for breakpoint. scriptId in start and end range locations should be
   * the same.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getPossibleBreakpoints)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getPossibleBreakpoints(input: GetPossibleBreakpointsRequest):
      GetPossibleBreakpointsResponse = session.request("Debugger.getPossibleBreakpoints", input)

  /**
   * Returns possible locations for breakpoint. scriptId in start and end range locations should be
   * the same.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getPossibleBreakpoints)
   *
   * @param start Start of range to search possible breakpoint locations in.
   */
  @JvmOverloads
  public suspend inline fun getPossibleBreakpoints(start: Location,
      optionalArgs: GetPossibleBreakpointsRequest.Builder.() -> Unit = {}):
      GetPossibleBreakpointsResponse {
    val builder = GetPossibleBreakpointsRequest.Builder(start)
    val input = builder.apply(optionalArgs).build()
    return getPossibleBreakpoints(input)
  }

  /**
   * Returns source for the script with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getScriptSource)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getScriptSource(input: GetScriptSourceRequest): GetScriptSourceResponse =
      session.request("Debugger.getScriptSource", input)

  /**
   * Returns source for the script with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getScriptSource)
   *
   * @param scriptId Id of the script to get source for.
   */
  public suspend fun getScriptSource(scriptId: ScriptId): GetScriptSourceResponse {
    val input = GetScriptSourceRequest(scriptId)
    return getScriptSource(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-disassembleWasmModule)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun disassembleWasmModule(input: DisassembleWasmModuleRequest):
      DisassembleWasmModuleResponse = session.request("Debugger.disassembleWasmModule", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-disassembleWasmModule)
   *
   * @param scriptId Id of the script to disassemble
   */
  @ExperimentalChromeApi
  public suspend fun disassembleWasmModule(scriptId: ScriptId): DisassembleWasmModuleResponse {
    val input = DisassembleWasmModuleRequest(scriptId)
    return disassembleWasmModule(input)
  }

  /**
   * Disassemble the next chunk of lines for the module corresponding to the
   * stream. If disassembly is complete, this API will invalidate the streamId
   * and return an empty chunk. Any subsequent calls for the now invalid stream
   * will return errors.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-nextWasmDisassemblyChunk)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun nextWasmDisassemblyChunk(input: NextWasmDisassemblyChunkRequest):
      NextWasmDisassemblyChunkResponse = session.request("Debugger.nextWasmDisassemblyChunk", input)

  /**
   * Disassemble the next chunk of lines for the module corresponding to the
   * stream. If disassembly is complete, this API will invalidate the streamId
   * and return an empty chunk. Any subsequent calls for the now invalid stream
   * will return errors.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-nextWasmDisassemblyChunk)
   */
  @ExperimentalChromeApi
  public suspend fun nextWasmDisassemblyChunk(streamId: String): NextWasmDisassemblyChunkResponse {
    val input = NextWasmDisassemblyChunkRequest(streamId)
    return nextWasmDisassemblyChunk(input)
  }

  /**
   * This command is deprecated. Use getScriptSource instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getWasmBytecode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun getWasmBytecode(input: GetWasmBytecodeRequest): GetWasmBytecodeResponse =
      session.request("Debugger.getWasmBytecode", input)

  /**
   * This command is deprecated. Use getScriptSource instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getWasmBytecode)
   *
   * @param scriptId Id of the Wasm script to get source for.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun getWasmBytecode(scriptId: ScriptId): GetWasmBytecodeResponse {
    val input = GetWasmBytecodeRequest(scriptId)
    return getWasmBytecode(input)
  }

  /**
   * Returns stack trace with given `stackTraceId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getStackTrace)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getStackTrace(input: GetStackTraceRequest): GetStackTraceResponse =
      session.request("Debugger.getStackTrace", input)

  /**
   * Returns stack trace with given `stackTraceId`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-getStackTrace)
   */
  @ExperimentalChromeApi
  public suspend fun getStackTrace(stackTraceId: StackTraceId): GetStackTraceResponse {
    val input = GetStackTraceRequest(stackTraceId)
    return getStackTrace(input)
  }

  /**
   * Stops on the next JavaScript statement.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-pause)
   */
  public suspend fun pause(): PauseResponse = session.request("Debugger.pause", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-pauseOnAsyncCall)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun pauseOnAsyncCall(input: PauseOnAsyncCallRequest): PauseOnAsyncCallResponse =
      session.request("Debugger.pauseOnAsyncCall", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-pauseOnAsyncCall)
   *
   * @param parentStackTraceId Debugger will pause when async call with given stack trace is
   * started.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun pauseOnAsyncCall(parentStackTraceId: StackTraceId): PauseOnAsyncCallResponse {
    val input = PauseOnAsyncCallRequest(parentStackTraceId)
    return pauseOnAsyncCall(input)
  }

  /**
   * Removes JavaScript breakpoint.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-removeBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeBreakpoint(input: RemoveBreakpointRequest): RemoveBreakpointResponse =
      session.request("Debugger.removeBreakpoint", input)

  /**
   * Removes JavaScript breakpoint.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-removeBreakpoint)
   */
  public suspend fun removeBreakpoint(breakpointId: BreakpointId): RemoveBreakpointResponse {
    val input = RemoveBreakpointRequest(breakpointId)
    return removeBreakpoint(input)
  }

  /**
   * Restarts particular call frame from the beginning. The old, deprecated
   * behavior of `restartFrame` is to stay paused and allow further CDP commands
   * after a restart was scheduled. This can cause problems with restarting, so
   * we now continue execution immediatly after it has been scheduled until we
   * reach the beginning of the restarted frame.
   *
   * To stay back-wards compatible, `restartFrame` now expects a `mode`
   * parameter to be present. If the `mode` parameter is missing, `restartFrame`
   * errors out.
   *
   * The various return values are deprecated and `callFrames` is always empty.
   * Use the call frames from the `Debugger#paused` events instead, that fires
   * once V8 pauses at the beginning of the restarted function.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-restartFrame)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun restartFrame(input: RestartFrameRequest): RestartFrameResponse =
      session.request("Debugger.restartFrame", input)

  /**
   * Restarts particular call frame from the beginning. The old, deprecated
   * behavior of `restartFrame` is to stay paused and allow further CDP commands
   * after a restart was scheduled. This can cause problems with restarting, so
   * we now continue execution immediatly after it has been scheduled until we
   * reach the beginning of the restarted frame.
   *
   * To stay back-wards compatible, `restartFrame` now expects a `mode`
   * parameter to be present. If the `mode` parameter is missing, `restartFrame`
   * errors out.
   *
   * The various return values are deprecated and `callFrames` is always empty.
   * Use the call frames from the `Debugger#paused` events instead, that fires
   * once V8 pauses at the beginning of the restarted function.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-restartFrame)
   *
   * @param callFrameId Call frame identifier to evaluate on.
   */
  @JvmOverloads
  public suspend inline fun restartFrame(callFrameId: CallFrameId,
      optionalArgs: RestartFrameRequest.Builder.() -> Unit = {}): RestartFrameResponse {
    val builder = RestartFrameRequest.Builder(callFrameId)
    val input = builder.apply(optionalArgs).build()
    return restartFrame(input)
  }

  /**
   * Resumes JavaScript execution.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-resume)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun resume(input: ResumeRequest): ResumeResponse =
      session.request("Debugger.resume", input)

  /**
   * Resumes JavaScript execution.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-resume)
   */
  @JvmOverloads
  public suspend inline fun resume(optionalArgs: ResumeRequest.Builder.() -> Unit = {}):
      ResumeResponse {
    val builder = ResumeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return resume(input)
  }

  /**
   * Searches for given string in script content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-searchInContent)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun searchInContent(input: SearchInContentRequest): SearchInContentResponse =
      session.request("Debugger.searchInContent", input)

  /**
   * Searches for given string in script content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-searchInContent)
   *
   * @param scriptId Id of the script to search in.
   * @param query String to search for.
   */
  @JvmOverloads
  public suspend inline fun searchInContent(
    scriptId: ScriptId,
    query: String,
    optionalArgs: SearchInContentRequest.Builder.() -> Unit = {},
  ): SearchInContentResponse {
    val builder = SearchInContentRequest.Builder(scriptId, query)
    val input = builder.apply(optionalArgs).build()
    return searchInContent(input)
  }

  /**
   * Enables or disables async call stacks tracking.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setAsyncCallStackDepth)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setAsyncCallStackDepth(input: SetAsyncCallStackDepthRequest):
      SetAsyncCallStackDepthResponse = session.request("Debugger.setAsyncCallStackDepth", input)

  /**
   * Enables or disables async call stacks tracking.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setAsyncCallStackDepth)
   *
   * @param maxDepth Maximum depth of async call stacks. Setting to `0` will effectively disable
   * collecting async
   * call stacks (default).
   */
  public suspend fun setAsyncCallStackDepth(maxDepth: Int): SetAsyncCallStackDepthResponse {
    val input = SetAsyncCallStackDepthRequest(maxDepth)
    return setAsyncCallStackDepth(input)
  }

  /**
   * Replace previous blackbox patterns with passed ones. Forces backend to skip stepping/pausing in
   * scripts with url matching one of the patterns. VM will try to leave blackboxed script by
   * performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBlackboxPatterns)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setBlackboxPatterns(input: SetBlackboxPatternsRequest):
      SetBlackboxPatternsResponse = session.request("Debugger.setBlackboxPatterns", input)

  /**
   * Replace previous blackbox patterns with passed ones. Forces backend to skip stepping/pausing in
   * scripts with url matching one of the patterns. VM will try to leave blackboxed script by
   * performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBlackboxPatterns)
   *
   * @param patterns Array of regexps that will be used to check script url for blackbox state.
   */
  @ExperimentalChromeApi
  public suspend fun setBlackboxPatterns(patterns: List<String>): SetBlackboxPatternsResponse {
    val input = SetBlackboxPatternsRequest(patterns)
    return setBlackboxPatterns(input)
  }

  /**
   * Makes backend skip steps in the script in blackboxed ranges. VM will try leave blacklisted
   * scripts by performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
   * Positions array contains positions where blackbox state is changed. First interval isn't
   * blackboxed. Array should be sorted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBlackboxedRanges)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setBlackboxedRanges(input: SetBlackboxedRangesRequest):
      SetBlackboxedRangesResponse = session.request("Debugger.setBlackboxedRanges", input)

  /**
   * Makes backend skip steps in the script in blackboxed ranges. VM will try leave blacklisted
   * scripts by performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
   * Positions array contains positions where blackbox state is changed. First interval isn't
   * blackboxed. Array should be sorted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBlackboxedRanges)
   *
   * @param scriptId Id of the script.
   */
  @ExperimentalChromeApi
  public suspend fun setBlackboxedRanges(scriptId: ScriptId, positions: List<ScriptPosition>):
      SetBlackboxedRangesResponse {
    val input = SetBlackboxedRangesRequest(scriptId, positions)
    return setBlackboxedRanges(input)
  }

  /**
   * Sets JavaScript breakpoint at a given location.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setBreakpoint(input: SetBreakpointRequest): SetBreakpointResponse =
      session.request("Debugger.setBreakpoint", input)

  /**
   * Sets JavaScript breakpoint at a given location.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpoint)
   *
   * @param location Location to set breakpoint in.
   */
  @JvmOverloads
  public suspend inline fun setBreakpoint(location: Location,
      optionalArgs: SetBreakpointRequest.Builder.() -> Unit = {}): SetBreakpointResponse {
    val builder = SetBreakpointRequest.Builder(location)
    val input = builder.apply(optionalArgs).build()
    return setBreakpoint(input)
  }

  /**
   * Sets instrumentation breakpoint.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setInstrumentationBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setInstrumentationBreakpoint(input: SetInstrumentationBreakpointRequest):
      SetInstrumentationBreakpointResponse =
      session.request("Debugger.setInstrumentationBreakpoint", input)

  /**
   * Sets instrumentation breakpoint.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setInstrumentationBreakpoint)
   *
   * @param instrumentation Instrumentation name.
   */
  public suspend fun setInstrumentationBreakpoint(instrumentation: BreakpointInstrumentation):
      SetInstrumentationBreakpointResponse {
    val input = SetInstrumentationBreakpointRequest(instrumentation)
    return setInstrumentationBreakpoint(input)
  }

  /**
   * Sets JavaScript breakpoint at given location specified either by URL or URL regex. Once this
   * command is issued, all existing parsed scripts will have breakpoints resolved and returned in
   * `locations` property. Further matching script parsing will result in subsequent
   * `breakpointResolved` events issued. This logical breakpoint will survive page reloads.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpointByUrl)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setBreakpointByUrl(input: SetBreakpointByUrlRequest):
      SetBreakpointByUrlResponse = session.request("Debugger.setBreakpointByUrl", input)

  /**
   * Sets JavaScript breakpoint at given location specified either by URL or URL regex. Once this
   * command is issued, all existing parsed scripts will have breakpoints resolved and returned in
   * `locations` property. Further matching script parsing will result in subsequent
   * `breakpointResolved` events issued. This logical breakpoint will survive page reloads.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpointByUrl)
   *
   * @param lineNumber Line number to set breakpoint at.
   */
  @JvmOverloads
  public suspend inline fun setBreakpointByUrl(lineNumber: Int,
      optionalArgs: SetBreakpointByUrlRequest.Builder.() -> Unit = {}): SetBreakpointByUrlResponse {
    val builder = SetBreakpointByUrlRequest.Builder(lineNumber)
    val input = builder.apply(optionalArgs).build()
    return setBreakpointByUrl(input)
  }

  /**
   * Sets JavaScript breakpoint before each call to the given function.
   * If another function was created from the same source as a given one,
   * calling it will also trigger the breakpoint.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpointOnFunctionCall)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setBreakpointOnFunctionCall(input: SetBreakpointOnFunctionCallRequest):
      SetBreakpointOnFunctionCallResponse = session.request("Debugger.setBreakpointOnFunctionCall",
      input)

  /**
   * Sets JavaScript breakpoint before each call to the given function.
   * If another function was created from the same source as a given one,
   * calling it will also trigger the breakpoint.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpointOnFunctionCall)
   *
   * @param objectId Function object id.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setBreakpointOnFunctionCall(objectId: RemoteObjectId,
      optionalArgs: SetBreakpointOnFunctionCallRequest.Builder.() -> Unit = {}):
      SetBreakpointOnFunctionCallResponse {
    val builder = SetBreakpointOnFunctionCallRequest.Builder(objectId)
    val input = builder.apply(optionalArgs).build()
    return setBreakpointOnFunctionCall(input)
  }

  /**
   * Activates / deactivates all breakpoints on the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpointsActive)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setBreakpointsActive(input: SetBreakpointsActiveRequest):
      SetBreakpointsActiveResponse = session.request("Debugger.setBreakpointsActive", input)

  /**
   * Activates / deactivates all breakpoints on the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setBreakpointsActive)
   *
   * @param active New value for breakpoints active state.
   */
  public suspend fun setBreakpointsActive(active: Boolean): SetBreakpointsActiveResponse {
    val input = SetBreakpointsActiveRequest(active)
    return setBreakpointsActive(input)
  }

  /**
   * Defines pause on exceptions state. Can be set to stop on all exceptions, uncaught exceptions or
   * no exceptions. Initial pause on exceptions state is `none`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setPauseOnExceptions)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setPauseOnExceptions(input: SetPauseOnExceptionsRequest):
      SetPauseOnExceptionsResponse = session.request("Debugger.setPauseOnExceptions", input)

  /**
   * Defines pause on exceptions state. Can be set to stop on all exceptions, uncaught exceptions or
   * no exceptions. Initial pause on exceptions state is `none`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setPauseOnExceptions)
   *
   * @param state Pause on exceptions mode.
   */
  public suspend fun setPauseOnExceptions(state: PauseOnExceptionsState):
      SetPauseOnExceptionsResponse {
    val input = SetPauseOnExceptionsRequest(state)
    return setPauseOnExceptions(input)
  }

  /**
   * Changes return value in top frame. Available only at return break position.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setReturnValue)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setReturnValue(input: SetReturnValueRequest): SetReturnValueResponse =
      session.request("Debugger.setReturnValue", input)

  /**
   * Changes return value in top frame. Available only at return break position.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setReturnValue)
   *
   * @param newValue New return value.
   */
  @ExperimentalChromeApi
  public suspend fun setReturnValue(newValue: CallArgument): SetReturnValueResponse {
    val input = SetReturnValueRequest(newValue)
    return setReturnValue(input)
  }

  /**
   * Edits JavaScript source live.
   *
   * In general, functions that are currently on the stack can not be edited with
   * a single exception: If the edited function is the top-most stack frame and
   * that is the only activation of that function on the stack. In this case
   * the live edit will be successful and a `Debugger.restartFrame` for the
   * top-most function is automatically triggered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setScriptSource)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setScriptSource(input: SetScriptSourceRequest): SetScriptSourceResponse =
      session.request("Debugger.setScriptSource", input)

  /**
   * Edits JavaScript source live.
   *
   * In general, functions that are currently on the stack can not be edited with
   * a single exception: If the edited function is the top-most stack frame and
   * that is the only activation of that function on the stack. In this case
   * the live edit will be successful and a `Debugger.restartFrame` for the
   * top-most function is automatically triggered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setScriptSource)
   *
   * @param scriptId Id of the script to edit.
   * @param scriptSource New content of the script.
   */
  @JvmOverloads
  public suspend inline fun setScriptSource(
    scriptId: ScriptId,
    scriptSource: String,
    optionalArgs: SetScriptSourceRequest.Builder.() -> Unit = {},
  ): SetScriptSourceResponse {
    val builder = SetScriptSourceRequest.Builder(scriptId, scriptSource)
    val input = builder.apply(optionalArgs).build()
    return setScriptSource(input)
  }

  /**
   * Makes page not interrupt on any pauses (breakpoint, exception, dom exception etc).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setSkipAllPauses)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setSkipAllPauses(input: SetSkipAllPausesRequest): SetSkipAllPausesResponse =
      session.request("Debugger.setSkipAllPauses", input)

  /**
   * Makes page not interrupt on any pauses (breakpoint, exception, dom exception etc).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setSkipAllPauses)
   *
   * @param skip New value for skip pauses state.
   */
  public suspend fun setSkipAllPauses(skip: Boolean): SetSkipAllPausesResponse {
    val input = SetSkipAllPausesRequest(skip)
    return setSkipAllPauses(input)
  }

  /**
   * Changes value of variable in a callframe. Object-based scopes are not supported and must be
   * mutated manually.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setVariableValue)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setVariableValue(input: SetVariableValueRequest): SetVariableValueResponse =
      session.request("Debugger.setVariableValue", input)

  /**
   * Changes value of variable in a callframe. Object-based scopes are not supported and must be
   * mutated manually.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-setVariableValue)
   *
   * @param scopeNumber 0-based number of scope as was listed in scope chain. Only 'local',
   * 'closure' and 'catch'
   * scope types are allowed. Other scopes could be manipulated manually.
   * @param variableName Variable name.
   * @param newValue New variable value.
   * @param callFrameId Id of callframe that holds variable.
   */
  public suspend fun setVariableValue(
    scopeNumber: Int,
    variableName: String,
    newValue: CallArgument,
    callFrameId: CallFrameId,
  ): SetVariableValueResponse {
    val input = SetVariableValueRequest(scopeNumber, variableName, newValue, callFrameId)
    return setVariableValue(input)
  }

  /**
   * Steps into the function call.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-stepInto)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun stepInto(input: StepIntoRequest): StepIntoResponse =
      session.request("Debugger.stepInto", input)

  /**
   * Steps into the function call.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-stepInto)
   */
  @JvmOverloads
  public suspend inline fun stepInto(optionalArgs: StepIntoRequest.Builder.() -> Unit = {}):
      StepIntoResponse {
    val builder = StepIntoRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return stepInto(input)
  }

  /**
   * Steps out of the function call.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-stepOut)
   */
  public suspend fun stepOut(): StepOutResponse = session.request("Debugger.stepOut", Unit)

  /**
   * Steps over the statement.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-stepOver)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun stepOver(input: StepOverRequest): StepOverResponse =
      session.request("Debugger.stepOver", input)

  /**
   * Steps over the statement.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#method-stepOver)
   */
  @JvmOverloads
  public suspend inline fun stepOver(optionalArgs: StepOverRequest.Builder.() -> Unit = {}):
      StepOverResponse {
    val builder = StepOverRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return stepOver(input)
  }
}
