@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.debugger

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.ScriptId
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Breakpoint identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-BreakpointId)
 */
public typealias BreakpointId = String

/**
 * Call frame identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-CallFrameId)
 */
public typealias CallFrameId = String

/**
 * Location in the source code.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-Location)
 */
@Serializable
public data class Location(
  /**
   * Script identifier as reported in the `Debugger.scriptParsed`.
   */
  public val scriptId: ScriptId,
  /**
   * Line number in the script (0-based).
   */
  public val lineNumber: Int,
  /**
   * Column number in the script (0-based).
   */
  public val columnNumber: Int? = null,
)

/**
 * Location in the source code.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-ScriptPosition)
 */
@ExperimentalChromeApi
@Serializable
public data class ScriptPosition(
  public val lineNumber: Int,
  public val columnNumber: Int,
)

/**
 * Location range within one script.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-LocationRange)
 */
@ExperimentalChromeApi
@Serializable
public data class LocationRange(
  public val scriptId: ScriptId,
  public val start: ScriptPosition,
  public val end: ScriptPosition,
)

/**
 * JavaScript call frame. Array of call frames form the call stack.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-CallFrame)
 */
@Serializable
public data class CallFrame(
  /**
   * Call frame identifier. This identifier is only valid while the virtual machine is paused.
   */
  public val callFrameId: CallFrameId,
  /**
   * Name of the JavaScript function called on this call frame.
   */
  public val functionName: String,
  /**
   * Location in the source code.
   */
  public val functionLocation: Location? = null,
  /**
   * Location in the source code.
   */
  public val location: Location,
  /**
   * JavaScript script name or url.
   * Deprecated in favor of using the `location.scriptId` to resolve the URL via a previously
   * sent `Debugger.scriptParsed` event.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val url: String,
  /**
   * Scope chain for this call frame.
   */
  public val scopeChain: List<Scope>,
  /**
   * `this` object for this call frame.
   */
  public val `this`: RemoteObject,
  /**
   * The value being returned, if the function is at return point.
   */
  public val returnValue: RemoteObject? = null,
  /**
   * Valid only while the VM is paused and indicates whether this frame
   * can be restarted or not. Note that a `true` value here does not
   * guarantee that Debugger#restartFrame with this CallFrameId will be
   * successful, but it is very likely.
   */
  @ExperimentalChromeApi
  public val canBeRestarted: Boolean? = null,
)

/**
 * Scope description.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-Scope)
 */
@Serializable
public data class Scope(
  /**
   * Scope type.
   */
  public val type: ScopeType,
  /**
   * Object representing the scope. For `global` and `with` scopes it represents the actual
   * object; for the rest of the scopes, it is artificial transient object enumerating scope
   * variables as its properties.
   */
  public val `object`: RemoteObject,
  public val name: String? = null,
  /**
   * Location in the source code where scope starts
   */
  public val startLocation: Location? = null,
  /**
   * Location in the source code where scope ends
   */
  public val endLocation: Location? = null,
)

/**
 * Search match for resource.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-SearchMatch)
 */
@Serializable
public data class SearchMatch(
  /**
   * Line number in resource content.
   */
  public val lineNumber: Double,
  /**
   * Line with match content.
   */
  public val lineContent: String,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-BreakLocation)
 */
@Serializable
public data class BreakLocation(
  /**
   * Script identifier as reported in the `Debugger.scriptParsed`.
   */
  public val scriptId: ScriptId,
  /**
   * Line number in the script (0-based).
   */
  public val lineNumber: Int,
  /**
   * Column number in the script (0-based).
   */
  public val columnNumber: Int? = null,
  public val type: BreakLocationType? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-WasmDisassemblyChunk)
 */
@ExperimentalChromeApi
@Serializable
public data class WasmDisassemblyChunk(
  /**
   * The next chunk of disassembled lines.
   */
  public val lines: List<String>,
  /**
   * The bytecode offsets describing the start of each line.
   */
  public val bytecodeOffsets: List<Int>,
)

/**
 * Enum of possible script languages.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-ScriptLanguage)
 */
@Serializable
public enum class ScriptLanguage {
  JavaScript,
  WebAssembly,
}

/**
 * Debug symbols available for a wasm script.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-DebugSymbols)
 */
@Serializable
public data class DebugSymbols(
  /**
   * Type of the debug symbols.
   */
  public val type: DebugSymbolsType,
  /**
   * URL of the external symbol source.
   */
  public val externalURL: String? = null,
)

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the property 'type' of the type 'Scope'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-ScopeType)
 */
@Serializable
public enum class ScopeType {
  global,
  local,
  with,
  closure,
  `catch`,
  block,
  script,
  eval,
  module,
  `wasm-expression-stack`,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the property 'type' of the type 'BreakLocation'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-BreakLocationType)
 */
@Serializable
public enum class BreakLocationType {
  debuggerStatement,
  call,
  `return`,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the property 'type' of the type 'DebugSymbols'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-DebugSymbolsType)
 */
@Serializable
public enum class DebugSymbolsType {
  None,
  SourceMap,
  EmbeddedDWARF,
  ExternalDWARF,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'targetCallFrames' of the command 'Debugger.continueToLocation'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-TargetCallFrames)
 */
@Serializable
public enum class TargetCallFrames {
  any,
  current,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'mode' of the command 'Debugger.restartFrame'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-FrameMode)
 */
@Serializable
public enum class FrameMode {
  StepInto,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'instrumentation' of the command 'Debugger.setInstrumentationBreakpoint'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-BreakpointInstrumentation)
 */
@Serializable
public enum class BreakpointInstrumentation {
  beforeScriptExecution,
  beforeScriptWithSourceMapExecution,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'state' of the command 'Debugger.setPauseOnExceptions'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-PauseOnExceptionsState)
 */
@Serializable
public enum class PauseOnExceptionsState {
  none,
  uncaught,
  all,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'status' of the command 'Debugger.setScriptSource'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-ScriptSourceStatus)
 */
@Serializable
public enum class ScriptSourceStatus {
  Ok,
  CompileError,
  BlockedByActiveGenerator,
  BlockedByActiveFunction,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'reason' of the event 'Debugger.paused'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Debugger/#type-PausedReason)
 */
@Serializable
public enum class PausedReason {
  ambiguous,
  assert,
  CSPViolation,
  debugCommand,
  DOM,
  EventListener,
  exception,
  instrumentation,
  OOM,
  other,
  promiseRejection,
  XHR,
}
