@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.dom

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.events.DOMEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.runtime.ExecutionContextId
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [DOMDomain.collectClassNamesFromSubtree]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class CollectClassNamesFromSubtreeRequest(
  /**
   * Id of the node to collect class names.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [DOMDomain.collectClassNamesFromSubtree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CollectClassNamesFromSubtreeResponse(
  /**
   * Class name list.
   */
  public val classNames: List<String>,
)

/**
 * Request object containing input parameters for the [DOMDomain.copyTo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CopyToRequest(
  /**
   * Id of the node to copy.
   */
  public val nodeId: NodeId,
  /**
   * Id of the element to drop the copy into.
   */
  public val targetNodeId: NodeId,
  /**
   * Drop the copy before this node (if absent, the copy becomes the last child of
   * `targetNodeId`).
   */
  public val insertBeforeNodeId: NodeId? = null,
) {
  public class Builder(
    /**
     * Id of the node to copy.
     */
    public val nodeId: NodeId,
    /**
     * Id of the element to drop the copy into.
     */
    public val targetNodeId: NodeId,
  ) {
    /**
     * Drop the copy before this node (if absent, the copy becomes the last child of
     * `targetNodeId`).
     */
    public var insertBeforeNodeId: NodeId? = null

    public fun build(): CopyToRequest = CopyToRequest(nodeId, targetNodeId, insertBeforeNodeId)
  }
}

/**
 * Response type for the [DOMDomain.copyTo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CopyToResponse(
  /**
   * Id of the node clone.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.describeNode] command.
 */
@Serializable
public data class DescribeNodeRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false).
   */
  public val pierce: Boolean? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    /**
     * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
     * entire subtree or provide an integer larger than 0.
     */
    public var depth: Int? = null

    /**
     * Whether or not iframes and shadow roots should be traversed when returning the subtree
     * (default is false).
     */
    public var pierce: Boolean? = null

    public fun build(): DescribeNodeRequest = DescribeNodeRequest(nodeId, backendNodeId, objectId,
        depth, pierce)
  }
}

/**
 * Response type for the [DOMDomain.describeNode] command.
 */
@Serializable
public data class DescribeNodeResponse(
  /**
   * Node description.
   */
  public val node: Node,
)

/**
 * Request object containing input parameters for the [DOMDomain.scrollIntoViewIfNeeded] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ScrollIntoViewIfNeededRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * The rect to be scrolled into view, relative to the node's border box, in CSS pixels.
   * When omitted, center of the node will be used, similar to Element.scrollIntoView.
   */
  public val rect: Rect? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    /**
     * The rect to be scrolled into view, relative to the node's border box, in CSS pixels.
     * When omitted, center of the node will be used, similar to Element.scrollIntoView.
     */
    public var rect: Rect? = null

    public fun build(): ScrollIntoViewIfNeededRequest = ScrollIntoViewIfNeededRequest(nodeId,
        backendNodeId, objectId, rect)
  }
}

/**
 * A dummy response object for the [DOMDomain.scrollIntoViewIfNeeded] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ScrollIntoViewIfNeededResponse

/**
 * A dummy response object for the [DOMDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [DOMDomain.discardSearchResults] command.
 */
@Serializable
@ExperimentalChromeApi
public data class DiscardSearchResultsRequest(
  /**
   * Unique search session identifier.
   */
  public val searchId: String,
)

/**
 * A dummy response object for the [DOMDomain.discardSearchResults] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object DiscardSearchResultsResponse

/**
 * Request object containing input parameters for the [DOMDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * Whether to include whitespaces in the children array of returned Nodes.
   */
  @ExperimentalChromeApi
  public val includeWhitespace: IncludeWhitespace? = null,
) {
  public class Builder() {
    /**
     * Whether to include whitespaces in the children array of returned Nodes.
     */
    @ExperimentalChromeApi
    public var includeWhitespace: IncludeWhitespace? = null

    public fun build(): EnableRequest = EnableRequest(includeWhitespace)
  }
}

/**
 * A dummy response object for the [DOMDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [DOMDomain.focus] command.
 */
@Serializable
public data class FocusRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): FocusRequest = FocusRequest(nodeId, backendNodeId, objectId)
  }
}

/**
 * A dummy response object for the [DOMDomain.focus] command. This command doesn't return any result
 * at the moment, but this could happen in the future, or could have happened in the past. For forwards
 * and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object FocusResponse

/**
 * Request object containing input parameters for the [DOMDomain.getAttributes] command.
 */
@Serializable
public data class GetAttributesRequest(
  /**
   * Id of the node to retrieve attibutes for.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [DOMDomain.getAttributes] command.
 */
@Serializable
public data class GetAttributesResponse(
  /**
   * An interleaved array of node attribute names and values.
   */
  public val attributes: List<String>,
)

/**
 * Request object containing input parameters for the [DOMDomain.getBoxModel] command.
 */
@Serializable
public data class GetBoxModelRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): GetBoxModelRequest = GetBoxModelRequest(nodeId, backendNodeId, objectId)
  }
}

/**
 * Response type for the [DOMDomain.getBoxModel] command.
 */
@Serializable
public data class GetBoxModelResponse(
  /**
   * Box model for the node.
   */
  public val model: BoxModel,
)

/**
 * Request object containing input parameters for the [DOMDomain.getContentQuads] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetContentQuadsRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): GetContentQuadsRequest = GetContentQuadsRequest(nodeId, backendNodeId,
        objectId)
  }
}

/**
 * Response type for the [DOMDomain.getContentQuads] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetContentQuadsResponse(
  /**
   * Quads that describe node layout relative to viewport.
   */
  public val quads: List<Quad>,
)

/**
 * Request object containing input parameters for the [DOMDomain.getDocument] command.
 */
@Serializable
public data class GetDocumentRequest(
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false).
   */
  public val pierce: Boolean? = null,
) {
  public class Builder() {
    /**
     * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
     * entire subtree or provide an integer larger than 0.
     */
    public var depth: Int? = null

    /**
     * Whether or not iframes and shadow roots should be traversed when returning the subtree
     * (default is false).
     */
    public var pierce: Boolean? = null

    public fun build(): GetDocumentRequest = GetDocumentRequest(depth, pierce)
  }
}

/**
 * Response type for the [DOMDomain.getDocument] command.
 */
@Serializable
public data class GetDocumentResponse(
  /**
   * Resulting node.
   */
  public val root: Node,
)

/**
 * Request object containing input parameters for the [DOMDomain.getFlattenedDocument] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetFlattenedDocumentRequest(
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false).
   */
  public val pierce: Boolean? = null,
) {
  public class Builder() {
    /**
     * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
     * entire subtree or provide an integer larger than 0.
     */
    public var depth: Int? = null

    /**
     * Whether or not iframes and shadow roots should be traversed when returning the subtree
     * (default is false).
     */
    public var pierce: Boolean? = null

    public fun build(): GetFlattenedDocumentRequest = GetFlattenedDocumentRequest(depth, pierce)
  }
}

/**
 * Response type for the [DOMDomain.getFlattenedDocument] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetFlattenedDocumentResponse(
  /**
   * Resulting node.
   */
  public val nodes: List<Node>,
)

/**
 * Request object containing input parameters for the [DOMDomain.getNodesForSubtreeByStyle] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodesForSubtreeByStyleRequest(
  /**
   * Node ID pointing to the root of a subtree.
   */
  public val nodeId: NodeId,
  /**
   * The style to filter nodes by (includes nodes if any of properties matches).
   */
  public val computedStyles: List<CSSComputedStyleProperty>,
  /**
   * Whether or not iframes and shadow roots in the same target should be traversed when returning
   * the
   * results (default is false).
   */
  public val pierce: Boolean? = null,
) {
  public class Builder(
    /**
     * Node ID pointing to the root of a subtree.
     */
    public val nodeId: NodeId,
    /**
     * The style to filter nodes by (includes nodes if any of properties matches).
     */
    public val computedStyles: List<CSSComputedStyleProperty>,
  ) {
    /**
     * Whether or not iframes and shadow roots in the same target should be traversed when returning
     * the
     * results (default is false).
     */
    public var pierce: Boolean? = null

    public fun build(): GetNodesForSubtreeByStyleRequest = GetNodesForSubtreeByStyleRequest(nodeId,
        computedStyles, pierce)
  }
}

/**
 * Response type for the [DOMDomain.getNodesForSubtreeByStyle] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodesForSubtreeByStyleResponse(
  /**
   * Resulting nodes.
   */
  public val nodeIds: List<NodeId>,
)

/**
 * Request object containing input parameters for the [DOMDomain.getNodeForLocation] command.
 */
@Serializable
public data class GetNodeForLocationRequest(
  /**
   * X coordinate.
   */
  public val x: Int,
  /**
   * Y coordinate.
   */
  public val y: Int,
  /**
   * False to skip to the nearest non-UA shadow root ancestor (default: false).
   */
  public val includeUserAgentShadowDOM: Boolean? = null,
  /**
   * Whether to ignore pointer-events: none on elements and hit test them.
   */
  public val ignorePointerEventsNone: Boolean? = null,
) {
  public class Builder(
    /**
     * X coordinate.
     */
    public val x: Int,
    /**
     * Y coordinate.
     */
    public val y: Int,
  ) {
    /**
     * False to skip to the nearest non-UA shadow root ancestor (default: false).
     */
    public var includeUserAgentShadowDOM: Boolean? = null

    /**
     * Whether to ignore pointer-events: none on elements and hit test them.
     */
    public var ignorePointerEventsNone: Boolean? = null

    public fun build(): GetNodeForLocationRequest = GetNodeForLocationRequest(x, y,
        includeUserAgentShadowDOM, ignorePointerEventsNone)
  }
}

/**
 * Response type for the [DOMDomain.getNodeForLocation] command.
 */
@Serializable
public data class GetNodeForLocationResponse(
  /**
   * Resulting node.
   */
  public val backendNodeId: BackendNodeId,
  /**
   * Frame this node belongs to.
   */
  public val frameId: FrameId,
  /**
   * Id of the node at given coordinates, only when enabled and requested document.
   */
  public val nodeId: NodeId? = null,
)

/**
 * Request object containing input parameters for the [DOMDomain.getOuterHTML] command.
 */
@Serializable
public data class GetOuterHTMLRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder() {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): GetOuterHTMLRequest = GetOuterHTMLRequest(nodeId, backendNodeId, objectId)
  }
}

/**
 * Response type for the [DOMDomain.getOuterHTML] command.
 */
@Serializable
public data class GetOuterHTMLResponse(
  /**
   * Outer HTML markup.
   */
  public val outerHTML: String,
)

/**
 * Request object containing input parameters for the [DOMDomain.getRelayoutBoundary] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRelayoutBoundaryRequest(
  /**
   * Id of the node.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [DOMDomain.getRelayoutBoundary] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRelayoutBoundaryResponse(
  /**
   * Relayout boundary node id for the given node.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.getSearchResults] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSearchResultsRequest(
  /**
   * Unique search session identifier.
   */
  public val searchId: String,
  /**
   * Start index of the search result to be returned.
   */
  public val fromIndex: Int,
  /**
   * End index of the search result to be returned.
   */
  public val toIndex: Int,
)

/**
 * Response type for the [DOMDomain.getSearchResults] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSearchResultsResponse(
  /**
   * Ids of the search result nodes.
   */
  public val nodeIds: List<NodeId>,
)

/**
 * A dummy response object for the [DOMDomain.hideHighlight] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object HideHighlightResponse

/**
 * A dummy response object for the [DOMDomain.highlightNode] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object HighlightNodeResponse

/**
 * A dummy response object for the [DOMDomain.highlightRect] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object HighlightRectResponse

/**
 * A dummy response object for the [DOMDomain.markUndoableState] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object MarkUndoableStateResponse

/**
 * Request object containing input parameters for the [DOMDomain.moveTo] command.
 */
@Serializable
public data class MoveToRequest(
  /**
   * Id of the node to move.
   */
  public val nodeId: NodeId,
  /**
   * Id of the element to drop the moved node into.
   */
  public val targetNodeId: NodeId,
  /**
   * Drop node before this one (if absent, the moved node becomes the last child of
   * `targetNodeId`).
   */
  public val insertBeforeNodeId: NodeId? = null,
) {
  public class Builder(
    /**
     * Id of the node to move.
     */
    public val nodeId: NodeId,
    /**
     * Id of the element to drop the moved node into.
     */
    public val targetNodeId: NodeId,
  ) {
    /**
     * Drop node before this one (if absent, the moved node becomes the last child of
     * `targetNodeId`).
     */
    public var insertBeforeNodeId: NodeId? = null

    public fun build(): MoveToRequest = MoveToRequest(nodeId, targetNodeId, insertBeforeNodeId)
  }
}

/**
 * Response type for the [DOMDomain.moveTo] command.
 */
@Serializable
public data class MoveToResponse(
  /**
   * New id of the moved node.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.performSearch] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PerformSearchRequest(
  /**
   * Plain text or query selector or XPath search query.
   */
  public val query: String,
  /**
   * True to search in user agent shadow DOM.
   */
  public val includeUserAgentShadowDOM: Boolean? = null,
) {
  public class Builder(
    /**
     * Plain text or query selector or XPath search query.
     */
    public val query: String,
  ) {
    /**
     * True to search in user agent shadow DOM.
     */
    public var includeUserAgentShadowDOM: Boolean? = null

    public fun build(): PerformSearchRequest = PerformSearchRequest(query,
        includeUserAgentShadowDOM)
  }
}

/**
 * Response type for the [DOMDomain.performSearch] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PerformSearchResponse(
  /**
   * Unique search session identifier.
   */
  public val searchId: String,
  /**
   * Number of search results.
   */
  public val resultCount: Int,
)

/**
 * Request object containing input parameters for the [DOMDomain.pushNodeByPathToFrontend] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodeByPathToFrontendRequest(
  /**
   * Path to node in the proprietary format.
   */
  public val path: String,
)

/**
 * Response type for the [DOMDomain.pushNodeByPathToFrontend] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodeByPathToFrontendResponse(
  /**
   * Id of the node for given path.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.pushNodesByBackendIdsToFrontend]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodesByBackendIdsToFrontendRequest(
  /**
   * The array of backend node ids.
   */
  public val backendNodeIds: List<BackendNodeId>,
)

/**
 * Response type for the [DOMDomain.pushNodesByBackendIdsToFrontend] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodesByBackendIdsToFrontendResponse(
  /**
   * The array of ids of pushed nodes that correspond to the backend ids specified in
   * backendNodeIds.
   */
  public val nodeIds: List<NodeId>,
)

/**
 * Request object containing input parameters for the [DOMDomain.querySelector] command.
 */
@Serializable
public data class QuerySelectorRequest(
  /**
   * Id of the node to query upon.
   */
  public val nodeId: NodeId,
  /**
   * Selector string.
   */
  public val selector: String,
)

/**
 * Response type for the [DOMDomain.querySelector] command.
 */
@Serializable
public data class QuerySelectorResponse(
  /**
   * Query selector result.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.querySelectorAll] command.
 */
@Serializable
public data class QuerySelectorAllRequest(
  /**
   * Id of the node to query upon.
   */
  public val nodeId: NodeId,
  /**
   * Selector string.
   */
  public val selector: String,
)

/**
 * Response type for the [DOMDomain.querySelectorAll] command.
 */
@Serializable
public data class QuerySelectorAllResponse(
  /**
   * Query selector result.
   */
  public val nodeIds: List<NodeId>,
)

/**
 * Response type for the [DOMDomain.getTopLayerElements] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetTopLayerElementsResponse(
  /**
   * NodeIds of top layer elements
   */
  public val nodeIds: List<NodeId>,
)

/**
 * A dummy response object for the [DOMDomain.redo] command. This command doesn't return any result
 * at the moment, but this could happen in the future, or could have happened in the past. For forwards
 * and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
@ExperimentalChromeApi
public object RedoResponse

/**
 * Request object containing input parameters for the [DOMDomain.removeAttribute] command.
 */
@Serializable
public data class RemoveAttributeRequest(
  /**
   * Id of the element to remove attribute from.
   */
  public val nodeId: NodeId,
  /**
   * Name of the attribute to remove.
   */
  public val name: String,
)

/**
 * A dummy response object for the [DOMDomain.removeAttribute] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object RemoveAttributeResponse

/**
 * Request object containing input parameters for the [DOMDomain.removeNode] command.
 */
@Serializable
public data class RemoveNodeRequest(
  /**
   * Id of the node to remove.
   */
  public val nodeId: NodeId,
)

/**
 * A dummy response object for the [DOMDomain.removeNode] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object RemoveNodeResponse

/**
 * Request object containing input parameters for the [DOMDomain.requestChildNodes] command.
 */
@Serializable
public data class RequestChildNodesRequest(
  /**
   * Id of the node to get children for.
   */
  public val nodeId: NodeId,
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the sub-tree
   * (default is false).
   */
  public val pierce: Boolean? = null,
) {
  public class Builder(
    /**
     * Id of the node to get children for.
     */
    public val nodeId: NodeId,
  ) {
    /**
     * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
     * entire subtree or provide an integer larger than 0.
     */
    public var depth: Int? = null

    /**
     * Whether or not iframes and shadow roots should be traversed when returning the sub-tree
     * (default is false).
     */
    public var pierce: Boolean? = null

    public fun build(): RequestChildNodesRequest = RequestChildNodesRequest(nodeId, depth, pierce)
  }
}

/**
 * A dummy response object for the [DOMDomain.requestChildNodes] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object RequestChildNodesResponse

/**
 * Request object containing input parameters for the [DOMDomain.requestNode] command.
 */
@Serializable
public data class RequestNodeRequest(
  /**
   * JavaScript object id to convert into node.
   */
  public val objectId: RemoteObjectId,
)

/**
 * Response type for the [DOMDomain.requestNode] command.
 */
@Serializable
public data class RequestNodeResponse(
  /**
   * Node id for given object.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.resolveNode] command.
 */
@Serializable
public data class ResolveNodeRequest(
  /**
   * Id of the node to resolve.
   */
  public val nodeId: NodeId? = null,
  /**
   * Backend identifier of the node to resolve.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
  /**
   * Execution context in which to resolve the node.
   */
  public val executionContextId: ExecutionContextId? = null,
) {
  public class Builder() {
    /**
     * Id of the node to resolve.
     */
    public var nodeId: NodeId? = null

    /**
     * Backend identifier of the node to resolve.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * Symbolic group name that can be used to release multiple objects.
     */
    public var objectGroup: String? = null

    /**
     * Execution context in which to resolve the node.
     */
    public var executionContextId: ExecutionContextId? = null

    public fun build(): ResolveNodeRequest = ResolveNodeRequest(nodeId, backendNodeId, objectGroup,
        executionContextId)
  }
}

/**
 * Response type for the [DOMDomain.resolveNode] command.
 */
@Serializable
public data class ResolveNodeResponse(
  /**
   * JavaScript object wrapper for given node.
   */
  public val `object`: RemoteObject,
)

/**
 * Request object containing input parameters for the [DOMDomain.setAttributeValue] command.
 */
@Serializable
public data class SetAttributeValueRequest(
  /**
   * Id of the element to set attribute for.
   */
  public val nodeId: NodeId,
  /**
   * Attribute name.
   */
  public val name: String,
  /**
   * Attribute value.
   */
  public val `value`: String,
)

/**
 * A dummy response object for the [DOMDomain.setAttributeValue] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetAttributeValueResponse

/**
 * Request object containing input parameters for the [DOMDomain.setAttributesAsText] command.
 */
@Serializable
public data class SetAttributesAsTextRequest(
  /**
   * Id of the element to set attributes for.
   */
  public val nodeId: NodeId,
  /**
   * Text with a number of attributes. Will parse this text using HTML parser.
   */
  public val text: String,
  /**
   * Attribute name to replace with new attributes derived from text in case text parsed
   * successfully.
   */
  public val name: String? = null,
) {
  public class Builder(
    /**
     * Id of the element to set attributes for.
     */
    public val nodeId: NodeId,
    /**
     * Text with a number of attributes. Will parse this text using HTML parser.
     */
    public val text: String,
  ) {
    /**
     * Attribute name to replace with new attributes derived from text in case text parsed
     * successfully.
     */
    public var name: String? = null

    public fun build(): SetAttributesAsTextRequest = SetAttributesAsTextRequest(nodeId, text, name)
  }
}

/**
 * A dummy response object for the [DOMDomain.setAttributesAsText] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetAttributesAsTextResponse

/**
 * Request object containing input parameters for the [DOMDomain.setFileInputFiles] command.
 */
@Serializable
public data class SetFileInputFilesRequest(
  /**
   * Array of file paths to set.
   */
  public val files: List<String>,
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder(
    /**
     * Array of file paths to set.
     */
    public val files: List<String>,
  ) {
    /**
     * Identifier of the node.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node wrapper.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): SetFileInputFilesRequest = SetFileInputFilesRequest(files, nodeId,
        backendNodeId, objectId)
  }
}

/**
 * A dummy response object for the [DOMDomain.setFileInputFiles] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetFileInputFilesResponse

/**
 * Request object containing input parameters for the [DOMDomain.setNodeStackTracesEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetNodeStackTracesEnabledRequest(
  /**
   * Enable or disable.
   */
  public val enable: Boolean,
)

/**
 * A dummy response object for the [DOMDomain.setNodeStackTracesEnabled] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetNodeStackTracesEnabledResponse

/**
 * Request object containing input parameters for the [DOMDomain.getNodeStackTraces] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodeStackTracesRequest(
  /**
   * Id of the node to get stack traces for.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [DOMDomain.getNodeStackTraces] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodeStackTracesResponse(
  /**
   * Creation stack trace, if available.
   */
  public val creation: StackTrace? = null,
)

/**
 * Request object containing input parameters for the [DOMDomain.getFileInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFileInfoRequest(
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId,
)

/**
 * Response type for the [DOMDomain.getFileInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFileInfoResponse(
  public val path: String,
)

/**
 * Request object containing input parameters for the [DOMDomain.setInspectedNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInspectedNodeRequest(
  /**
   * DOM node id to be accessible by means of $x command line API.
   */
  public val nodeId: NodeId,
)

/**
 * A dummy response object for the [DOMDomain.setInspectedNode] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetInspectedNodeResponse

/**
 * Request object containing input parameters for the [DOMDomain.setNodeName] command.
 */
@Serializable
public data class SetNodeNameRequest(
  /**
   * Id of the node to set name for.
   */
  public val nodeId: NodeId,
  /**
   * New node's name.
   */
  public val name: String,
)

/**
 * Response type for the [DOMDomain.setNodeName] command.
 */
@Serializable
public data class SetNodeNameResponse(
  /**
   * New node's id.
   */
  public val nodeId: NodeId,
)

/**
 * Request object containing input parameters for the [DOMDomain.setNodeValue] command.
 */
@Serializable
public data class SetNodeValueRequest(
  /**
   * Id of the node to set value for.
   */
  public val nodeId: NodeId,
  /**
   * New node's value.
   */
  public val `value`: String,
)

/**
 * A dummy response object for the [DOMDomain.setNodeValue] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object SetNodeValueResponse

/**
 * Request object containing input parameters for the [DOMDomain.setOuterHTML] command.
 */
@Serializable
public data class SetOuterHTMLRequest(
  /**
   * Id of the node to set markup for.
   */
  public val nodeId: NodeId,
  /**
   * Outer HTML markup to set.
   */
  public val outerHTML: String,
)

/**
 * A dummy response object for the [DOMDomain.setOuterHTML] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object SetOuterHTMLResponse

/**
 * A dummy response object for the [DOMDomain.undo] command. This command doesn't return any result
 * at the moment, but this could happen in the future, or could have happened in the past. For forwards
 * and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
@ExperimentalChromeApi
public object UndoResponse

/**
 * Request object containing input parameters for the [DOMDomain.getFrameOwner] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFrameOwnerRequest(
  public val frameId: FrameId,
)

/**
 * Response type for the [DOMDomain.getFrameOwner] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFrameOwnerResponse(
  /**
   * Resulting node.
   */
  public val backendNodeId: BackendNodeId,
  /**
   * Id of the node at given coordinates, only when enabled and requested document.
   */
  public val nodeId: NodeId? = null,
)

/**
 * Request object containing input parameters for the [DOMDomain.getContainerForNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetContainerForNodeRequest(
  public val nodeId: NodeId,
  public val containerName: String? = null,
) {
  public class Builder(
    public val nodeId: NodeId,
  ) {
    public var containerName: String? = null

    public fun build(): GetContainerForNodeRequest = GetContainerForNodeRequest(nodeId,
        containerName)
  }
}

/**
 * Response type for the [DOMDomain.getContainerForNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetContainerForNodeResponse(
  /**
   * The container node for the given node, or null if not found.
   */
  public val nodeId: NodeId? = null,
)

/**
 * Request object containing input parameters for the [DOMDomain.getQueryingDescendantsForContainer]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetQueryingDescendantsForContainerRequest(
  /**
   * Id of the container node to find querying descendants from.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [DOMDomain.getQueryingDescendantsForContainer] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetQueryingDescendantsForContainerResponse(
  /**
   * Descendant nodes with container queries against the given container.
   */
  public val nodeIds: List<NodeId>,
)

/**
 * This domain exposes DOM read/write operations. Each DOM Node is represented with its mirror
 * object
 * that has an `id`. This `id` can be used to get additional information on the Node, resolve it
 * into
 * the JavaScript object wrapper, etc. It is important that client receives DOM events only for the
 * nodes that are known to the client. Backend keeps track of the nodes that were sent to the client
 * and never sends the same node twice. It is client's responsibility to collect information about
 * the nodes that were sent to the client.<p>Note that `iframe` owner elements will return
 * corresponding document elements as their child nodes.</p>
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM)
 */
public class DOMDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DOMEvent>> = mapOf(
      "DOM.attributeModified" to serializer<DOMEvent.AttributeModified>(),
      "DOM.attributeRemoved" to serializer<DOMEvent.AttributeRemoved>(),
      "DOM.characterDataModified" to serializer<DOMEvent.CharacterDataModified>(),
      "DOM.childNodeCountUpdated" to serializer<DOMEvent.ChildNodeCountUpdated>(),
      "DOM.childNodeInserted" to serializer<DOMEvent.ChildNodeInserted>(),
      "DOM.childNodeRemoved" to serializer<DOMEvent.ChildNodeRemoved>(),
      "DOM.distributedNodesUpdated" to serializer<DOMEvent.DistributedNodesUpdated>(),
      "DOM.documentUpdated" to serializer<DOMEvent.DocumentUpdated>(),
      "DOM.inlineStyleInvalidated" to serializer<DOMEvent.InlineStyleInvalidated>(),
      "DOM.pseudoElementAdded" to serializer<DOMEvent.PseudoElementAdded>(),
      "DOM.topLayerElementsUpdated" to serializer<DOMEvent.TopLayerElementsUpdated>(),
      "DOM.pseudoElementRemoved" to serializer<DOMEvent.PseudoElementRemoved>(),
      "DOM.setChildNodes" to serializer<DOMEvent.SetChildNodes>(),
      "DOM.shadowRootPopped" to serializer<DOMEvent.ShadowRootPopped>(),
      "DOM.shadowRootPushed" to serializer<DOMEvent.ShadowRootPushed>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DOMEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fired when `Element`'s attribute is modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-attributeModified)
   */
  public fun attributeModifiedEvents(): Flow<DOMEvent.AttributeModified> =
      session.typedEvents("DOM.attributeModified")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("attributeModifiedEvents()"),
  )
  public fun attributeModified(): Flow<DOMEvent.AttributeModified> = attributeModifiedEvents()

  /**
   * Fired when `Element`'s attribute is removed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-attributeRemoved)
   */
  public fun attributeRemovedEvents(): Flow<DOMEvent.AttributeRemoved> =
      session.typedEvents("DOM.attributeRemoved")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("attributeRemovedEvents()"),
  )
  public fun attributeRemoved(): Flow<DOMEvent.AttributeRemoved> = attributeRemovedEvents()

  /**
   * Mirrors `DOMCharacterDataModified` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-characterDataModified)
   */
  public fun characterDataModifiedEvents(): Flow<DOMEvent.CharacterDataModified> =
      session.typedEvents("DOM.characterDataModified")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("characterDataModifiedEvents()"),
  )
  public fun characterDataModified(): Flow<DOMEvent.CharacterDataModified> =
      characterDataModifiedEvents()

  /**
   * Fired when `Container`'s child node count has changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeCountUpdated)
   */
  public fun childNodeCountUpdatedEvents(): Flow<DOMEvent.ChildNodeCountUpdated> =
      session.typedEvents("DOM.childNodeCountUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("childNodeCountUpdatedEvents()"),
  )
  public fun childNodeCountUpdated(): Flow<DOMEvent.ChildNodeCountUpdated> =
      childNodeCountUpdatedEvents()

  /**
   * Mirrors `DOMNodeInserted` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeInserted)
   */
  public fun childNodeInsertedEvents(): Flow<DOMEvent.ChildNodeInserted> =
      session.typedEvents("DOM.childNodeInserted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("childNodeInsertedEvents()"),
  )
  public fun childNodeInserted(): Flow<DOMEvent.ChildNodeInserted> = childNodeInsertedEvents()

  /**
   * Mirrors `DOMNodeRemoved` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeRemoved)
   */
  public fun childNodeRemovedEvents(): Flow<DOMEvent.ChildNodeRemoved> =
      session.typedEvents("DOM.childNodeRemoved")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("childNodeRemovedEvents()"),
  )
  public fun childNodeRemoved(): Flow<DOMEvent.ChildNodeRemoved> = childNodeRemovedEvents()

  /**
   * Called when distribution is changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-distributedNodesUpdated)
   */
  @ExperimentalChromeApi
  public fun distributedNodesUpdatedEvents(): Flow<DOMEvent.DistributedNodesUpdated> =
      session.typedEvents("DOM.distributedNodesUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("distributedNodesUpdatedEvents()"),
  )
  public fun distributedNodesUpdated(): Flow<DOMEvent.DistributedNodesUpdated> =
      distributedNodesUpdatedEvents()

  /**
   * Fired when `Document` has been totally updated. Node ids are no longer valid.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-documentUpdated)
   */
  public fun documentUpdatedEvents(): Flow<DOMEvent.DocumentUpdated> =
      session.typedEvents("DOM.documentUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("documentUpdatedEvents()"),
  )
  public fun documentUpdated(): Flow<DOMEvent.DocumentUpdated> = documentUpdatedEvents()

  /**
   * Fired when `Element`'s inline style is modified via a CSS property modification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-inlineStyleInvalidated)
   */
  @ExperimentalChromeApi
  public fun inlineStyleInvalidatedEvents(): Flow<DOMEvent.InlineStyleInvalidated> =
      session.typedEvents("DOM.inlineStyleInvalidated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("inlineStyleInvalidatedEvents()"),
  )
  public fun inlineStyleInvalidated(): Flow<DOMEvent.InlineStyleInvalidated> =
      inlineStyleInvalidatedEvents()

  /**
   * Called when a pseudo element is added to an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-pseudoElementAdded)
   */
  @ExperimentalChromeApi
  public fun pseudoElementAddedEvents(): Flow<DOMEvent.PseudoElementAdded> =
      session.typedEvents("DOM.pseudoElementAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("pseudoElementAddedEvents()"),
  )
  public fun pseudoElementAdded(): Flow<DOMEvent.PseudoElementAdded> = pseudoElementAddedEvents()

  /**
   * Called when top layer elements are changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-topLayerElementsUpdated)
   */
  @ExperimentalChromeApi
  public fun topLayerElementsUpdatedEvents(): Flow<DOMEvent.TopLayerElementsUpdated> =
      session.typedEvents("DOM.topLayerElementsUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("topLayerElementsUpdatedEvents()"),
  )
  public fun topLayerElementsUpdated(): Flow<DOMEvent.TopLayerElementsUpdated> =
      topLayerElementsUpdatedEvents()

  /**
   * Called when a pseudo element is removed from an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-pseudoElementRemoved)
   */
  @ExperimentalChromeApi
  public fun pseudoElementRemovedEvents(): Flow<DOMEvent.PseudoElementRemoved> =
      session.typedEvents("DOM.pseudoElementRemoved")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("pseudoElementRemovedEvents()"),
  )
  public fun pseudoElementRemoved(): Flow<DOMEvent.PseudoElementRemoved> =
      pseudoElementRemovedEvents()

  /**
   * Fired when backend wants to provide client with the missing DOM structure. This happens upon
   * most of the calls requesting node ids.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-setChildNodes)
   */
  public fun setChildNodesEvents(): Flow<DOMEvent.SetChildNodes> =
      session.typedEvents("DOM.setChildNodes")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("setChildNodesEvents()"),
  )
  public fun setChildNodes(): Flow<DOMEvent.SetChildNodes> = setChildNodesEvents()

  /**
   * Called when shadow root is popped from the element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-shadowRootPopped)
   */
  @ExperimentalChromeApi
  public fun shadowRootPoppedEvents(): Flow<DOMEvent.ShadowRootPopped> =
      session.typedEvents("DOM.shadowRootPopped")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("shadowRootPoppedEvents()"),
  )
  public fun shadowRootPopped(): Flow<DOMEvent.ShadowRootPopped> = shadowRootPoppedEvents()

  /**
   * Called when shadow root is pushed into the element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-shadowRootPushed)
   */
  @ExperimentalChromeApi
  public fun shadowRootPushedEvents(): Flow<DOMEvent.ShadowRootPushed> =
      session.typedEvents("DOM.shadowRootPushed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("shadowRootPushedEvents()"),
  )
  public fun shadowRootPushed(): Flow<DOMEvent.ShadowRootPushed> = shadowRootPushedEvents()

  /**
   * Collects class names for the node with given id and all of it's child nodes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-collectClassNamesFromSubtree)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun collectClassNamesFromSubtree(input: CollectClassNamesFromSubtreeRequest):
      CollectClassNamesFromSubtreeResponse = session.request("DOM.collectClassNamesFromSubtree",
      input)

  /**
   * Collects class names for the node with given id and all of it's child nodes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-collectClassNamesFromSubtree)
   *
   * @param nodeId Id of the node to collect class names.
   */
  @ExperimentalChromeApi
  public suspend fun collectClassNamesFromSubtree(nodeId: NodeId):
      CollectClassNamesFromSubtreeResponse {
    val input = CollectClassNamesFromSubtreeRequest(nodeId)
    return collectClassNamesFromSubtree(input)
  }

  /**
   * Creates a deep copy of the specified node and places it into the target container before the
   * given anchor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-copyTo)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun copyTo(input: CopyToRequest): CopyToResponse = session.request("DOM.copyTo",
      input)

  /**
   * Creates a deep copy of the specified node and places it into the target container before the
   * given anchor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-copyTo)
   *
   * @param nodeId Id of the node to copy.
   * @param targetNodeId Id of the element to drop the copy into.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun copyTo(
    nodeId: NodeId,
    targetNodeId: NodeId,
    optionalArgs: CopyToRequest.Builder.() -> Unit = {},
  ): CopyToResponse {
    val builder = CopyToRequest.Builder(nodeId, targetNodeId)
    val input = builder.apply(optionalArgs).build()
    return copyTo(input)
  }

  /**
   * Describes node given its id, does not require domain to be enabled. Does not start tracking any
   * objects, can be used for automation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-describeNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun describeNode(input: DescribeNodeRequest): DescribeNodeResponse =
      session.request("DOM.describeNode", input)

  /**
   * Describes node given its id, does not require domain to be enabled. Does not start tracking any
   * objects, can be used for automation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-describeNode)
   */
  @JvmOverloads
  public suspend inline fun describeNode(optionalArgs: DescribeNodeRequest.Builder.() -> Unit = {}):
      DescribeNodeResponse {
    val builder = DescribeNodeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return describeNode(input)
  }

  /**
   * Scrolls the specified rect of the given node into view if not already visible.
   * Note: exactly one between nodeId, backendNodeId and objectId should be passed
   * to identify the node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-scrollIntoViewIfNeeded)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun scrollIntoViewIfNeeded(input: ScrollIntoViewIfNeededRequest):
      ScrollIntoViewIfNeededResponse = session.request("DOM.scrollIntoViewIfNeeded", input)

  /**
   * Scrolls the specified rect of the given node into view if not already visible.
   * Note: exactly one between nodeId, backendNodeId and objectId should be passed
   * to identify the node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-scrollIntoViewIfNeeded)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun scrollIntoViewIfNeeded(optionalArgs: ScrollIntoViewIfNeededRequest.Builder.() -> Unit =
      {}): ScrollIntoViewIfNeededResponse {
    val builder = ScrollIntoViewIfNeededRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return scrollIntoViewIfNeeded(input)
  }

  /**
   * Disables DOM agent for the given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("DOM.disable", Unit)

  /**
   * Discards search results from the session with the given id. `getSearchResults` should no longer
   * be called for that search.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-discardSearchResults)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun discardSearchResults(input: DiscardSearchResultsRequest):
      DiscardSearchResultsResponse = session.request("DOM.discardSearchResults", input)

  /**
   * Discards search results from the session with the given id. `getSearchResults` should no longer
   * be called for that search.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-discardSearchResults)
   *
   * @param searchId Unique search session identifier.
   */
  @ExperimentalChromeApi
  public suspend fun discardSearchResults(searchId: String): DiscardSearchResultsResponse {
    val input = DiscardSearchResultsRequest(searchId)
    return discardSearchResults(input)
  }

  /**
   * Enables DOM agent for the given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse = session.request("DOM.enable",
      input)

  /**
   * Enables DOM agent for the given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Focuses the given element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-focus)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun focus(input: FocusRequest): FocusResponse = session.request("DOM.focus", input)

  /**
   * Focuses the given element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-focus)
   */
  @JvmOverloads
  public suspend inline fun focus(optionalArgs: FocusRequest.Builder.() -> Unit = {}):
      FocusResponse {
    val builder = FocusRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return focus(input)
  }

  /**
   * Returns attributes for the specified node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getAttributes)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getAttributes(input: GetAttributesRequest): GetAttributesResponse =
      session.request("DOM.getAttributes", input)

  /**
   * Returns attributes for the specified node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getAttributes)
   *
   * @param nodeId Id of the node to retrieve attibutes for.
   */
  public suspend fun getAttributes(nodeId: NodeId): GetAttributesResponse {
    val input = GetAttributesRequest(nodeId)
    return getAttributes(input)
  }

  /**
   * Returns boxes for the given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getBoxModel)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getBoxModel(input: GetBoxModelRequest): GetBoxModelResponse =
      session.request("DOM.getBoxModel", input)

  /**
   * Returns boxes for the given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getBoxModel)
   */
  @JvmOverloads
  public suspend inline fun getBoxModel(optionalArgs: GetBoxModelRequest.Builder.() -> Unit = {}):
      GetBoxModelResponse {
    val builder = GetBoxModelRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getBoxModel(input)
  }

  /**
   * Returns quads that describe node position on the page. This method
   * might return multiple quads for inline nodes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getContentQuads)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getContentQuads(input: GetContentQuadsRequest): GetContentQuadsResponse =
      session.request("DOM.getContentQuads", input)

  /**
   * Returns quads that describe node position on the page. This method
   * might return multiple quads for inline nodes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getContentQuads)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getContentQuads(optionalArgs: GetContentQuadsRequest.Builder.() -> Unit
      = {}): GetContentQuadsResponse {
    val builder = GetContentQuadsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getContentQuads(input)
  }

  /**
   * Returns the root DOM node (and optionally the subtree) to the caller.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getDocument)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getDocument(input: GetDocumentRequest): GetDocumentResponse =
      session.request("DOM.getDocument", input)

  /**
   * Returns the root DOM node (and optionally the subtree) to the caller.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getDocument)
   */
  @JvmOverloads
  public suspend inline fun getDocument(optionalArgs: GetDocumentRequest.Builder.() -> Unit = {}):
      GetDocumentResponse {
    val builder = GetDocumentRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getDocument(input)
  }

  /**
   * Returns the root DOM node (and optionally the subtree) to the caller.
   * Deprecated, as it is not designed to work well with the rest of the DOM agent.
   * Use DOMSnapshot.captureSnapshot instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFlattenedDocument)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun getFlattenedDocument(input: GetFlattenedDocumentRequest):
      GetFlattenedDocumentResponse = session.request("DOM.getFlattenedDocument", input)

  /**
   * Returns the root DOM node (and optionally the subtree) to the caller.
   * Deprecated, as it is not designed to work well with the rest of the DOM agent.
   * Use DOMSnapshot.captureSnapshot instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFlattenedDocument)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @JvmOverloads
  public suspend inline
      fun getFlattenedDocument(optionalArgs: GetFlattenedDocumentRequest.Builder.() -> Unit = {}):
      GetFlattenedDocumentResponse {
    val builder = GetFlattenedDocumentRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getFlattenedDocument(input)
  }

  /**
   * Finds nodes with a given computed style in a subtree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodesForSubtreeByStyle)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getNodesForSubtreeByStyle(input: GetNodesForSubtreeByStyleRequest):
      GetNodesForSubtreeByStyleResponse = session.request("DOM.getNodesForSubtreeByStyle", input)

  /**
   * Finds nodes with a given computed style in a subtree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodesForSubtreeByStyle)
   *
   * @param nodeId Node ID pointing to the root of a subtree.
   * @param computedStyles The style to filter nodes by (includes nodes if any of properties
   * matches).
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getNodesForSubtreeByStyle(
    nodeId: NodeId,
    computedStyles: List<CSSComputedStyleProperty>,
    optionalArgs: GetNodesForSubtreeByStyleRequest.Builder.() -> Unit = {},
  ): GetNodesForSubtreeByStyleResponse {
    val builder = GetNodesForSubtreeByStyleRequest.Builder(nodeId, computedStyles)
    val input = builder.apply(optionalArgs).build()
    return getNodesForSubtreeByStyle(input)
  }

  /**
   * Returns node id at given location. Depending on whether DOM domain is enabled, nodeId is
   * either returned or not.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodeForLocation)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getNodeForLocation(input: GetNodeForLocationRequest):
      GetNodeForLocationResponse = session.request("DOM.getNodeForLocation", input)

  /**
   * Returns node id at given location. Depending on whether DOM domain is enabled, nodeId is
   * either returned or not.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodeForLocation)
   *
   * @param x X coordinate.
   * @param y Y coordinate.
   */
  @JvmOverloads
  public suspend inline fun getNodeForLocation(
    x: Int,
    y: Int,
    optionalArgs: GetNodeForLocationRequest.Builder.() -> Unit = {},
  ): GetNodeForLocationResponse {
    val builder = GetNodeForLocationRequest.Builder(x, y)
    val input = builder.apply(optionalArgs).build()
    return getNodeForLocation(input)
  }

  /**
   * Returns node's HTML markup.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getOuterHTML)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getOuterHTML(input: GetOuterHTMLRequest): GetOuterHTMLResponse =
      session.request("DOM.getOuterHTML", input)

  /**
   * Returns node's HTML markup.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getOuterHTML)
   */
  @JvmOverloads
  public suspend inline fun getOuterHTML(optionalArgs: GetOuterHTMLRequest.Builder.() -> Unit = {}):
      GetOuterHTMLResponse {
    val builder = GetOuterHTMLRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getOuterHTML(input)
  }

  /**
   * Returns the id of the nearest ancestor that is a relayout boundary.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getRelayoutBoundary)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getRelayoutBoundary(input: GetRelayoutBoundaryRequest):
      GetRelayoutBoundaryResponse = session.request("DOM.getRelayoutBoundary", input)

  /**
   * Returns the id of the nearest ancestor that is a relayout boundary.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getRelayoutBoundary)
   *
   * @param nodeId Id of the node.
   */
  @ExperimentalChromeApi
  public suspend fun getRelayoutBoundary(nodeId: NodeId): GetRelayoutBoundaryResponse {
    val input = GetRelayoutBoundaryRequest(nodeId)
    return getRelayoutBoundary(input)
  }

  /**
   * Returns search results from given `fromIndex` to given `toIndex` from the search with the given
   * identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getSearchResults)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getSearchResults(input: GetSearchResultsRequest): GetSearchResultsResponse =
      session.request("DOM.getSearchResults", input)

  /**
   * Returns search results from given `fromIndex` to given `toIndex` from the search with the given
   * identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getSearchResults)
   *
   * @param searchId Unique search session identifier.
   * @param fromIndex Start index of the search result to be returned.
   * @param toIndex End index of the search result to be returned.
   */
  @ExperimentalChromeApi
  public suspend fun getSearchResults(
    searchId: String,
    fromIndex: Int,
    toIndex: Int,
  ): GetSearchResultsResponse {
    val input = GetSearchResultsRequest(searchId, fromIndex, toIndex)
    return getSearchResults(input)
  }

  /**
   * Hides any highlight.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-hideHighlight)
   */
  public suspend fun hideHighlight(): HideHighlightResponse = session.request("DOM.hideHighlight",
      Unit)

  /**
   * Highlights DOM node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-highlightNode)
   */
  public suspend fun highlightNode(): HighlightNodeResponse = session.request("DOM.highlightNode",
      Unit)

  /**
   * Highlights given rectangle.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-highlightRect)
   */
  public suspend fun highlightRect(): HighlightRectResponse = session.request("DOM.highlightRect",
      Unit)

  /**
   * Marks last undoable state.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-markUndoableState)
   */
  @ExperimentalChromeApi
  public suspend fun markUndoableState(): MarkUndoableStateResponse =
      session.request("DOM.markUndoableState", Unit)

  /**
   * Moves node into the new container, places it before the given anchor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-moveTo)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun moveTo(input: MoveToRequest): MoveToResponse = session.request("DOM.moveTo",
      input)

  /**
   * Moves node into the new container, places it before the given anchor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-moveTo)
   *
   * @param nodeId Id of the node to move.
   * @param targetNodeId Id of the element to drop the moved node into.
   */
  @JvmOverloads
  public suspend inline fun moveTo(
    nodeId: NodeId,
    targetNodeId: NodeId,
    optionalArgs: MoveToRequest.Builder.() -> Unit = {},
  ): MoveToResponse {
    val builder = MoveToRequest.Builder(nodeId, targetNodeId)
    val input = builder.apply(optionalArgs).build()
    return moveTo(input)
  }

  /**
   * Searches for a given string in the DOM tree. Use `getSearchResults` to access search results or
   * `cancelSearch` to end this search session.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-performSearch)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun performSearch(input: PerformSearchRequest): PerformSearchResponse =
      session.request("DOM.performSearch", input)

  /**
   * Searches for a given string in the DOM tree. Use `getSearchResults` to access search results or
   * `cancelSearch` to end this search session.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-performSearch)
   *
   * @param query Plain text or query selector or XPath search query.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun performSearch(query: String,
      optionalArgs: PerformSearchRequest.Builder.() -> Unit = {}): PerformSearchResponse {
    val builder = PerformSearchRequest.Builder(query)
    val input = builder.apply(optionalArgs).build()
    return performSearch(input)
  }

  /**
   * Requests that the node is sent to the caller given its path. // FIXME, use XPath
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-pushNodeByPathToFrontend)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun pushNodeByPathToFrontend(input: PushNodeByPathToFrontendRequest):
      PushNodeByPathToFrontendResponse = session.request("DOM.pushNodeByPathToFrontend", input)

  /**
   * Requests that the node is sent to the caller given its path. // FIXME, use XPath
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-pushNodeByPathToFrontend)
   *
   * @param path Path to node in the proprietary format.
   */
  @ExperimentalChromeApi
  public suspend fun pushNodeByPathToFrontend(path: String): PushNodeByPathToFrontendResponse {
    val input = PushNodeByPathToFrontendRequest(path)
    return pushNodeByPathToFrontend(input)
  }

  /**
   * Requests that a batch of nodes is sent to the caller given their backend node ids.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-pushNodesByBackendIdsToFrontend)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun pushNodesByBackendIdsToFrontend(input: PushNodesByBackendIdsToFrontendRequest):
      PushNodesByBackendIdsToFrontendResponse =
      session.request("DOM.pushNodesByBackendIdsToFrontend", input)

  /**
   * Requests that a batch of nodes is sent to the caller given their backend node ids.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-pushNodesByBackendIdsToFrontend)
   *
   * @param backendNodeIds The array of backend node ids.
   */
  @ExperimentalChromeApi
  public suspend fun pushNodesByBackendIdsToFrontend(backendNodeIds: List<BackendNodeId>):
      PushNodesByBackendIdsToFrontendResponse {
    val input = PushNodesByBackendIdsToFrontendRequest(backendNodeIds)
    return pushNodesByBackendIdsToFrontend(input)
  }

  /**
   * Executes `querySelector` on a given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-querySelector)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun querySelector(input: QuerySelectorRequest): QuerySelectorResponse =
      session.request("DOM.querySelector", input)

  /**
   * Executes `querySelector` on a given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-querySelector)
   *
   * @param nodeId Id of the node to query upon.
   * @param selector Selector string.
   */
  public suspend fun querySelector(nodeId: NodeId, selector: String): QuerySelectorResponse {
    val input = QuerySelectorRequest(nodeId, selector)
    return querySelector(input)
  }

  /**
   * Executes `querySelectorAll` on a given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-querySelectorAll)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun querySelectorAll(input: QuerySelectorAllRequest): QuerySelectorAllResponse =
      session.request("DOM.querySelectorAll", input)

  /**
   * Executes `querySelectorAll` on a given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-querySelectorAll)
   *
   * @param nodeId Id of the node to query upon.
   * @param selector Selector string.
   */
  public suspend fun querySelectorAll(nodeId: NodeId, selector: String): QuerySelectorAllResponse {
    val input = QuerySelectorAllRequest(nodeId, selector)
    return querySelectorAll(input)
  }

  /**
   * Returns NodeIds of current top layer elements.
   * Top layer is rendered closest to the user within a viewport, therefore its elements always
   * appear on top of all other content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getTopLayerElements)
   */
  @ExperimentalChromeApi
  public suspend fun getTopLayerElements(): GetTopLayerElementsResponse =
      session.request("DOM.getTopLayerElements", Unit)

  /**
   * Re-does the last undone action.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-redo)
   */
  @ExperimentalChromeApi
  public suspend fun redo(): RedoResponse = session.request("DOM.redo", Unit)

  /**
   * Removes attribute with given name from an element with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-removeAttribute)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeAttribute(input: RemoveAttributeRequest): RemoveAttributeResponse =
      session.request("DOM.removeAttribute", input)

  /**
   * Removes attribute with given name from an element with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-removeAttribute)
   *
   * @param nodeId Id of the element to remove attribute from.
   * @param name Name of the attribute to remove.
   */
  public suspend fun removeAttribute(nodeId: NodeId, name: String): RemoveAttributeResponse {
    val input = RemoveAttributeRequest(nodeId, name)
    return removeAttribute(input)
  }

  /**
   * Removes node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-removeNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeNode(input: RemoveNodeRequest): RemoveNodeResponse =
      session.request("DOM.removeNode", input)

  /**
   * Removes node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-removeNode)
   *
   * @param nodeId Id of the node to remove.
   */
  public suspend fun removeNode(nodeId: NodeId): RemoveNodeResponse {
    val input = RemoveNodeRequest(nodeId)
    return removeNode(input)
  }

  /**
   * Requests that children of the node with given id are returned to the caller in form of
   * `setChildNodes` events where not only immediate children are retrieved, but all children down
   * to
   * the specified depth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-requestChildNodes)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestChildNodes(input: RequestChildNodesRequest): RequestChildNodesResponse =
      session.request("DOM.requestChildNodes", input)

  /**
   * Requests that children of the node with given id are returned to the caller in form of
   * `setChildNodes` events where not only immediate children are retrieved, but all children down
   * to
   * the specified depth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-requestChildNodes)
   *
   * @param nodeId Id of the node to get children for.
   */
  @JvmOverloads
  public suspend inline fun requestChildNodes(nodeId: NodeId,
      optionalArgs: RequestChildNodesRequest.Builder.() -> Unit = {}): RequestChildNodesResponse {
    val builder = RequestChildNodesRequest.Builder(nodeId)
    val input = builder.apply(optionalArgs).build()
    return requestChildNodes(input)
  }

  /**
   * Requests that the node is sent to the caller given the JavaScript node object reference. All
   * nodes that form the path from the node to the root are also sent to the client as a series of
   * `setChildNodes` notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-requestNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestNode(input: RequestNodeRequest): RequestNodeResponse =
      session.request("DOM.requestNode", input)

  /**
   * Requests that the node is sent to the caller given the JavaScript node object reference. All
   * nodes that form the path from the node to the root are also sent to the client as a series of
   * `setChildNodes` notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-requestNode)
   *
   * @param objectId JavaScript object id to convert into node.
   */
  public suspend fun requestNode(objectId: RemoteObjectId): RequestNodeResponse {
    val input = RequestNodeRequest(objectId)
    return requestNode(input)
  }

  /**
   * Resolves the JavaScript node object for a given NodeId or BackendNodeId.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-resolveNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun resolveNode(input: ResolveNodeRequest): ResolveNodeResponse =
      session.request("DOM.resolveNode", input)

  /**
   * Resolves the JavaScript node object for a given NodeId or BackendNodeId.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-resolveNode)
   */
  @JvmOverloads
  public suspend inline fun resolveNode(optionalArgs: ResolveNodeRequest.Builder.() -> Unit = {}):
      ResolveNodeResponse {
    val builder = ResolveNodeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return resolveNode(input)
  }

  /**
   * Sets attribute for an element with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setAttributeValue)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setAttributeValue(input: SetAttributeValueRequest): SetAttributeValueResponse =
      session.request("DOM.setAttributeValue", input)

  /**
   * Sets attribute for an element with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setAttributeValue)
   *
   * @param nodeId Id of the element to set attribute for.
   * @param name Attribute name.
   * @param value Attribute value.
   */
  public suspend fun setAttributeValue(
    nodeId: NodeId,
    name: String,
    `value`: String,
  ): SetAttributeValueResponse {
    val input = SetAttributeValueRequest(nodeId, name, value)
    return setAttributeValue(input)
  }

  /**
   * Sets attributes on element with given id. This method is useful when user edits some existing
   * attribute value and types in several attribute name/value pairs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setAttributesAsText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setAttributesAsText(input: SetAttributesAsTextRequest):
      SetAttributesAsTextResponse = session.request("DOM.setAttributesAsText", input)

  /**
   * Sets attributes on element with given id. This method is useful when user edits some existing
   * attribute value and types in several attribute name/value pairs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setAttributesAsText)
   *
   * @param nodeId Id of the element to set attributes for.
   * @param text Text with a number of attributes. Will parse this text using HTML parser.
   */
  @JvmOverloads
  public suspend inline fun setAttributesAsText(
    nodeId: NodeId,
    text: String,
    optionalArgs: SetAttributesAsTextRequest.Builder.() -> Unit = {},
  ): SetAttributesAsTextResponse {
    val builder = SetAttributesAsTextRequest.Builder(nodeId, text)
    val input = builder.apply(optionalArgs).build()
    return setAttributesAsText(input)
  }

  /**
   * Sets files for the given file input element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setFileInputFiles)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setFileInputFiles(input: SetFileInputFilesRequest): SetFileInputFilesResponse =
      session.request("DOM.setFileInputFiles", input)

  /**
   * Sets files for the given file input element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setFileInputFiles)
   *
   * @param files Array of file paths to set.
   */
  @JvmOverloads
  public suspend inline fun setFileInputFiles(files: List<String>,
      optionalArgs: SetFileInputFilesRequest.Builder.() -> Unit = {}): SetFileInputFilesResponse {
    val builder = SetFileInputFilesRequest.Builder(files)
    val input = builder.apply(optionalArgs).build()
    return setFileInputFiles(input)
  }

  /**
   * Sets if stack traces should be captured for Nodes. See `Node.getNodeStackTraces`. Default is
   * disabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeStackTracesEnabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setNodeStackTracesEnabled(input: SetNodeStackTracesEnabledRequest):
      SetNodeStackTracesEnabledResponse = session.request("DOM.setNodeStackTracesEnabled", input)

  /**
   * Sets if stack traces should be captured for Nodes. See `Node.getNodeStackTraces`. Default is
   * disabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeStackTracesEnabled)
   *
   * @param enable Enable or disable.
   */
  @ExperimentalChromeApi
  public suspend fun setNodeStackTracesEnabled(enable: Boolean): SetNodeStackTracesEnabledResponse {
    val input = SetNodeStackTracesEnabledRequest(enable)
    return setNodeStackTracesEnabled(input)
  }

  /**
   * Gets stack traces associated with a Node. As of now, only provides stack trace for Node
   * creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodeStackTraces)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getNodeStackTraces(input: GetNodeStackTracesRequest):
      GetNodeStackTracesResponse = session.request("DOM.getNodeStackTraces", input)

  /**
   * Gets stack traces associated with a Node. As of now, only provides stack trace for Node
   * creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodeStackTraces)
   *
   * @param nodeId Id of the node to get stack traces for.
   */
  @ExperimentalChromeApi
  public suspend fun getNodeStackTraces(nodeId: NodeId): GetNodeStackTracesResponse {
    val input = GetNodeStackTracesRequest(nodeId)
    return getNodeStackTraces(input)
  }

  /**
   * Returns file information for the given
   * File wrapper.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFileInfo)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getFileInfo(input: GetFileInfoRequest): GetFileInfoResponse =
      session.request("DOM.getFileInfo", input)

  /**
   * Returns file information for the given
   * File wrapper.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFileInfo)
   *
   * @param objectId JavaScript object id of the node wrapper.
   */
  @ExperimentalChromeApi
  public suspend fun getFileInfo(objectId: RemoteObjectId): GetFileInfoResponse {
    val input = GetFileInfoRequest(objectId)
    return getFileInfo(input)
  }

  /**
   * Enables console to refer to the node with given id via $x (see Command Line API for more
   * details
   * $x functions).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setInspectedNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setInspectedNode(input: SetInspectedNodeRequest): SetInspectedNodeResponse =
      session.request("DOM.setInspectedNode", input)

  /**
   * Enables console to refer to the node with given id via $x (see Command Line API for more
   * details
   * $x functions).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setInspectedNode)
   *
   * @param nodeId DOM node id to be accessible by means of $x command line API.
   */
  @ExperimentalChromeApi
  public suspend fun setInspectedNode(nodeId: NodeId): SetInspectedNodeResponse {
    val input = SetInspectedNodeRequest(nodeId)
    return setInspectedNode(input)
  }

  /**
   * Sets node name for a node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeName)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setNodeName(input: SetNodeNameRequest): SetNodeNameResponse =
      session.request("DOM.setNodeName", input)

  /**
   * Sets node name for a node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeName)
   *
   * @param nodeId Id of the node to set name for.
   * @param name New node's name.
   */
  public suspend fun setNodeName(nodeId: NodeId, name: String): SetNodeNameResponse {
    val input = SetNodeNameRequest(nodeId, name)
    return setNodeName(input)
  }

  /**
   * Sets node value for a node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeValue)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setNodeValue(input: SetNodeValueRequest): SetNodeValueResponse =
      session.request("DOM.setNodeValue", input)

  /**
   * Sets node value for a node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeValue)
   *
   * @param nodeId Id of the node to set value for.
   * @param value New node's value.
   */
  public suspend fun setNodeValue(nodeId: NodeId, `value`: String): SetNodeValueResponse {
    val input = SetNodeValueRequest(nodeId, value)
    return setNodeValue(input)
  }

  /**
   * Sets node HTML markup, returns new node id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setOuterHTML)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setOuterHTML(input: SetOuterHTMLRequest): SetOuterHTMLResponse =
      session.request("DOM.setOuterHTML", input)

  /**
   * Sets node HTML markup, returns new node id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setOuterHTML)
   *
   * @param nodeId Id of the node to set markup for.
   * @param outerHTML Outer HTML markup to set.
   */
  public suspend fun setOuterHTML(nodeId: NodeId, outerHTML: String): SetOuterHTMLResponse {
    val input = SetOuterHTMLRequest(nodeId, outerHTML)
    return setOuterHTML(input)
  }

  /**
   * Undoes the last performed action.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-undo)
   */
  @ExperimentalChromeApi
  public suspend fun undo(): UndoResponse = session.request("DOM.undo", Unit)

  /**
   * Returns iframe node that owns iframe with the given domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFrameOwner)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getFrameOwner(input: GetFrameOwnerRequest): GetFrameOwnerResponse =
      session.request("DOM.getFrameOwner", input)

  /**
   * Returns iframe node that owns iframe with the given domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFrameOwner)
   */
  @ExperimentalChromeApi
  public suspend fun getFrameOwner(frameId: FrameId): GetFrameOwnerResponse {
    val input = GetFrameOwnerRequest(frameId)
    return getFrameOwner(input)
  }

  /**
   * Returns the container of the given node based on container query conditions.
   * If containerName is given, it will find the nearest container with a matching name;
   * otherwise it will find the nearest container regardless of its container name.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getContainerForNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getContainerForNode(input: GetContainerForNodeRequest):
      GetContainerForNodeResponse = session.request("DOM.getContainerForNode", input)

  /**
   * Returns the container of the given node based on container query conditions.
   * If containerName is given, it will find the nearest container with a matching name;
   * otherwise it will find the nearest container regardless of its container name.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getContainerForNode)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getContainerForNode(nodeId: NodeId,
      optionalArgs: GetContainerForNodeRequest.Builder.() -> Unit = {}):
      GetContainerForNodeResponse {
    val builder = GetContainerForNodeRequest.Builder(nodeId)
    val input = builder.apply(optionalArgs).build()
    return getContainerForNode(input)
  }

  /**
   * Returns the descendants of a container query container that have
   * container queries against this container.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getQueryingDescendantsForContainer)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend
      fun getQueryingDescendantsForContainer(input: GetQueryingDescendantsForContainerRequest):
      GetQueryingDescendantsForContainerResponse =
      session.request("DOM.getQueryingDescendantsForContainer", input)

  /**
   * Returns the descendants of a container query container that have
   * container queries against this container.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getQueryingDescendantsForContainer)
   *
   * @param nodeId Id of the container node to find querying descendants from.
   */
  @ExperimentalChromeApi
  public suspend fun getQueryingDescendantsForContainer(nodeId: NodeId):
      GetQueryingDescendantsForContainerResponse {
    val input = GetQueryingDescendantsForContainerRequest(nodeId)
    return getQueryingDescendantsForContainer(input)
  }
}
