@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.dom.events

import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNode
import org.hildan.chrome.devtools.domains.dom.Node
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class DOMEvent {
  /**
   * Fired when `Element`'s attribute is modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-attributeModified)
   */
  @Serializable
  public data class AttributeModified(
    /**
     * Id of the node that has changed.
     */
    public val nodeId: NodeId,
    /**
     * Attribute name.
     */
    public val name: String,
    /**
     * Attribute value.
     */
    public val `value`: String,
  ) : DOMEvent()

  /**
   * Fired when `Element`'s attribute is removed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-attributeRemoved)
   */
  @Serializable
  public data class AttributeRemoved(
    /**
     * Id of the node that has changed.
     */
    public val nodeId: NodeId,
    /**
     * A ttribute name.
     */
    public val name: String,
  ) : DOMEvent()

  /**
   * Mirrors `DOMCharacterDataModified` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-characterDataModified)
   */
  @Serializable
  public data class CharacterDataModified(
    /**
     * Id of the node that has changed.
     */
    public val nodeId: NodeId,
    /**
     * New text value.
     */
    public val characterData: String,
  ) : DOMEvent()

  /**
   * Fired when `Container`'s child node count has changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeCountUpdated)
   */
  @Serializable
  public data class ChildNodeCountUpdated(
    /**
     * Id of the node that has changed.
     */
    public val nodeId: NodeId,
    /**
     * New node count.
     */
    public val childNodeCount: Int,
  ) : DOMEvent()

  /**
   * Mirrors `DOMNodeInserted` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeInserted)
   */
  @Serializable
  public data class ChildNodeInserted(
    /**
     * Id of the node that has changed.
     */
    public val parentNodeId: NodeId,
    /**
     * Id of the previous sibling.
     */
    public val previousNodeId: NodeId,
    /**
     * Inserted node data.
     */
    public val node: Node,
  ) : DOMEvent()

  /**
   * Mirrors `DOMNodeRemoved` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeRemoved)
   */
  @Serializable
  public data class ChildNodeRemoved(
    /**
     * Parent id.
     */
    public val parentNodeId: NodeId,
    /**
     * Id of the node that has been removed.
     */
    public val nodeId: NodeId,
  ) : DOMEvent()

  /**
   * Called when distribution is changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-distributedNodesUpdated)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class DistributedNodesUpdated(
    /**
     * Insertion point where distributed nodes were updated.
     */
    public val insertionPointId: NodeId,
    /**
     * Distributed nodes for given insertion point.
     */
    public val distributedNodes: List<BackendNode>,
  ) : DOMEvent()

  /**
   * Fired when `Document` has been totally updated. Node ids are no longer valid.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-documentUpdated)
   */
  @Serializable
  public object DocumentUpdated : DOMEvent()

  /**
   * Fired when `Element`'s inline style is modified via a CSS property modification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-inlineStyleInvalidated)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class InlineStyleInvalidated(
    /**
     * Ids of the nodes for which the inline styles have been invalidated.
     */
    public val nodeIds: List<NodeId>,
  ) : DOMEvent()

  /**
   * Called when a pseudo element is added to an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-pseudoElementAdded)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class PseudoElementAdded(
    /**
     * Pseudo element's parent element id.
     */
    public val parentId: NodeId,
    /**
     * The added pseudo element.
     */
    public val pseudoElement: Node,
  ) : DOMEvent()

  /**
   * Called when top layer elements are changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-topLayerElementsUpdated)
   */
  @ExperimentalChromeApi
  @Serializable
  public object TopLayerElementsUpdated : DOMEvent()

  /**
   * Called when a pseudo element is removed from an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-pseudoElementRemoved)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class PseudoElementRemoved(
    /**
     * Pseudo element's parent element id.
     */
    public val parentId: NodeId,
    /**
     * The removed pseudo element id.
     */
    public val pseudoElementId: NodeId,
  ) : DOMEvent()

  /**
   * Fired when backend wants to provide client with the missing DOM structure. This happens upon
   * most of the calls requesting node ids.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-setChildNodes)
   */
  @Serializable
  public data class SetChildNodes(
    /**
     * Parent node id to populate with children.
     */
    public val parentId: NodeId,
    /**
     * Child nodes array.
     */
    public val nodes: List<Node>,
  ) : DOMEvent()

  /**
   * Called when shadow root is popped from the element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-shadowRootPopped)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ShadowRootPopped(
    /**
     * Host element id.
     */
    public val hostId: NodeId,
    /**
     * Shadow root id.
     */
    public val rootId: NodeId,
  ) : DOMEvent()

  /**
   * Called when shadow root is pushed into the element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-shadowRootPushed)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ShadowRootPushed(
    /**
     * Host element id.
     */
    public val hostId: NodeId,
    /**
     * Shadow root.
     */
    public val root: Node,
  ) : DOMEvent()
}
