@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.domdebugger

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.getEventListeners] command.
 */
@Serializable
public data class GetEventListenersRequest(
  /**
   * Identifier of the object to return listeners for.
   */
  public val objectId: RemoteObjectId,
  /**
   * The maximum depth at which Node children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false). Reports listeners for all contexts if pierce is enabled.
   */
  public val pierce: Boolean? = null,
) {
  public class Builder(
    /**
     * Identifier of the object to return listeners for.
     */
    public val objectId: RemoteObjectId,
  ) {
    /**
     * The maximum depth at which Node children should be retrieved, defaults to 1. Use -1 for the
     * entire subtree or provide an integer larger than 0.
     */
    public var depth: Int? = null

    /**
     * Whether or not iframes and shadow roots should be traversed when returning the subtree
     * (default is false). Reports listeners for all contexts if pierce is enabled.
     */
    public var pierce: Boolean? = null

    public fun build(): GetEventListenersRequest = GetEventListenersRequest(objectId, depth, pierce)
  }
}

/**
 * Response type for the [DOMDebuggerDomain.getEventListeners] command.
 */
@Serializable
public data class GetEventListenersResponse(
  /**
   * Array of relevant listeners.
   */
  public val listeners: List<EventListener>,
)

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.removeDOMBreakpoint]
 * command.
 */
@Serializable
public data class RemoveDOMBreakpointRequest(
  /**
   * Identifier of the node to remove breakpoint from.
   */
  public val nodeId: NodeId,
  /**
   * Type of the breakpoint to remove.
   */
  public val type: DOMBreakpointType,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.removeDOMBreakpoint] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object RemoveDOMBreakpointResponse

/**
 * Request object containing input parameters for the
 * [DOMDebuggerDomain.removeEventListenerBreakpoint] command.
 */
@Serializable
public data class RemoveEventListenerBreakpointRequest(
  /**
   * Event name.
   */
  public val eventName: String,
  /**
   * EventTarget interface name.
   */
  @ExperimentalChromeApi
  public val targetName: String? = null,
) {
  public class Builder(
    /**
     * Event name.
     */
    public val eventName: String,
  ) {
    /**
     * EventTarget interface name.
     */
    @ExperimentalChromeApi
    public var targetName: String? = null

    public fun build(): RemoveEventListenerBreakpointRequest =
        RemoveEventListenerBreakpointRequest(eventName, targetName)
  }
}

/**
 * A dummy response object for the [DOMDebuggerDomain.removeEventListenerBreakpoint] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object RemoveEventListenerBreakpointResponse

/**
 * Request object containing input parameters for the
 * [DOMDebuggerDomain.removeInstrumentationBreakpoint] command.
 */
@Serializable
@ExperimentalChromeApi
public data class RemoveInstrumentationBreakpointRequest(
  /**
   * Instrumentation name to stop on.
   */
  public val eventName: String,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.removeInstrumentationBreakpoint] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object RemoveInstrumentationBreakpointResponse

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.removeXHRBreakpoint]
 * command.
 */
@Serializable
public data class RemoveXHRBreakpointRequest(
  /**
   * Resource URL substring.
   */
  public val url: String,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.removeXHRBreakpoint] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object RemoveXHRBreakpointResponse

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setBreakOnCSPViolation]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBreakOnCSPViolationRequest(
  /**
   * CSP Violations to stop upon.
   */
  public val violationTypes: List<CSPViolationType>,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.setBreakOnCSPViolation] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetBreakOnCSPViolationResponse

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setDOMBreakpoint] command.
 */
@Serializable
public data class SetDOMBreakpointRequest(
  /**
   * Identifier of the node to set breakpoint on.
   */
  public val nodeId: NodeId,
  /**
   * Type of the operation to stop upon.
   */
  public val type: DOMBreakpointType,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.setDOMBreakpoint] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetDOMBreakpointResponse

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setEventListenerBreakpoint]
 * command.
 */
@Serializable
public data class SetEventListenerBreakpointRequest(
  /**
   * DOM Event name to stop on (any DOM event will do).
   */
  public val eventName: String,
  /**
   * EventTarget interface name to stop on. If equal to `"*"` or not provided, will stop on any
   * EventTarget.
   */
  @ExperimentalChromeApi
  public val targetName: String? = null,
) {
  public class Builder(
    /**
     * DOM Event name to stop on (any DOM event will do).
     */
    public val eventName: String,
  ) {
    /**
     * EventTarget interface name to stop on. If equal to `"*"` or not provided, will stop on any
     * EventTarget.
     */
    @ExperimentalChromeApi
    public var targetName: String? = null

    public fun build(): SetEventListenerBreakpointRequest =
        SetEventListenerBreakpointRequest(eventName, targetName)
  }
}

/**
 * A dummy response object for the [DOMDebuggerDomain.setEventListenerBreakpoint] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetEventListenerBreakpointResponse

/**
 * Request object containing input parameters for the
 * [DOMDebuggerDomain.setInstrumentationBreakpoint] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInstrumentationBreakpointRequest(
  /**
   * Instrumentation name to stop on.
   */
  public val eventName: String,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.setInstrumentationBreakpoint] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetInstrumentationBreakpointResponse

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setXHRBreakpoint] command.
 */
@Serializable
public data class SetXHRBreakpointRequest(
  /**
   * Resource URL substring. All XHRs having this substring in the URL will get stopped upon.
   */
  public val url: String,
)

/**
 * A dummy response object for the [DOMDebuggerDomain.setXHRBreakpoint] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetXHRBreakpointResponse

/**
 * DOM debugging allows setting breakpoints on particular DOM operations and events. JavaScript
 * execution will stop on these operations as if there was a regular breakpoint set.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger)
 */
public class DOMDebuggerDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Returns event listeners of the given object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-getEventListeners)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getEventListeners(input: GetEventListenersRequest): GetEventListenersResponse =
      session.request("DOMDebugger.getEventListeners", input)

  /**
   * Returns event listeners of the given object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-getEventListeners)
   *
   * @param objectId Identifier of the object to return listeners for.
   */
  @JvmOverloads
  public suspend inline fun getEventListeners(objectId: RemoteObjectId,
      optionalArgs: GetEventListenersRequest.Builder.() -> Unit = {}): GetEventListenersResponse {
    val builder = GetEventListenersRequest.Builder(objectId)
    val input = builder.apply(optionalArgs).build()
    return getEventListeners(input)
  }

  /**
   * Removes DOM breakpoint that was set using `setDOMBreakpoint`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeDOMBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeDOMBreakpoint(input: RemoveDOMBreakpointRequest):
      RemoveDOMBreakpointResponse = session.request("DOMDebugger.removeDOMBreakpoint", input)

  /**
   * Removes DOM breakpoint that was set using `setDOMBreakpoint`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeDOMBreakpoint)
   *
   * @param nodeId Identifier of the node to remove breakpoint from.
   * @param type Type of the breakpoint to remove.
   */
  public suspend fun removeDOMBreakpoint(nodeId: NodeId, type: DOMBreakpointType):
      RemoveDOMBreakpointResponse {
    val input = RemoveDOMBreakpointRequest(nodeId, type)
    return removeDOMBreakpoint(input)
  }

  /**
   * Removes breakpoint on particular DOM event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeEventListenerBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeEventListenerBreakpoint(input: RemoveEventListenerBreakpointRequest):
      RemoveEventListenerBreakpointResponse =
      session.request("DOMDebugger.removeEventListenerBreakpoint", input)

  /**
   * Removes breakpoint on particular DOM event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeEventListenerBreakpoint)
   *
   * @param eventName Event name.
   */
  @JvmOverloads
  public suspend inline fun removeEventListenerBreakpoint(eventName: String,
      optionalArgs: RemoveEventListenerBreakpointRequest.Builder.() -> Unit = {}):
      RemoveEventListenerBreakpointResponse {
    val builder = RemoveEventListenerBreakpointRequest.Builder(eventName)
    val input = builder.apply(optionalArgs).build()
    return removeEventListenerBreakpoint(input)
  }

  /**
   * Removes breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeInstrumentationBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun removeInstrumentationBreakpoint(input: RemoveInstrumentationBreakpointRequest):
      RemoveInstrumentationBreakpointResponse =
      session.request("DOMDebugger.removeInstrumentationBreakpoint", input)

  /**
   * Removes breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeInstrumentationBreakpoint)
   *
   * @param eventName Instrumentation name to stop on.
   */
  @ExperimentalChromeApi
  public suspend fun removeInstrumentationBreakpoint(eventName: String):
      RemoveInstrumentationBreakpointResponse {
    val input = RemoveInstrumentationBreakpointRequest(eventName)
    return removeInstrumentationBreakpoint(input)
  }

  /**
   * Removes breakpoint from XMLHttpRequest.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeXHRBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeXHRBreakpoint(input: RemoveXHRBreakpointRequest):
      RemoveXHRBreakpointResponse = session.request("DOMDebugger.removeXHRBreakpoint", input)

  /**
   * Removes breakpoint from XMLHttpRequest.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-removeXHRBreakpoint)
   *
   * @param url Resource URL substring.
   */
  public suspend fun removeXHRBreakpoint(url: String): RemoveXHRBreakpointResponse {
    val input = RemoveXHRBreakpointRequest(url)
    return removeXHRBreakpoint(input)
  }

  /**
   * Sets breakpoint on particular CSP violations.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setBreakOnCSPViolation)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setBreakOnCSPViolation(input: SetBreakOnCSPViolationRequest):
      SetBreakOnCSPViolationResponse = session.request("DOMDebugger.setBreakOnCSPViolation", input)

  /**
   * Sets breakpoint on particular CSP violations.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setBreakOnCSPViolation)
   *
   * @param violationTypes CSP Violations to stop upon.
   */
  @ExperimentalChromeApi
  public suspend fun setBreakOnCSPViolation(violationTypes: List<CSPViolationType>):
      SetBreakOnCSPViolationResponse {
    val input = SetBreakOnCSPViolationRequest(violationTypes)
    return setBreakOnCSPViolation(input)
  }

  /**
   * Sets breakpoint on particular operation with DOM.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setDOMBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setDOMBreakpoint(input: SetDOMBreakpointRequest): SetDOMBreakpointResponse =
      session.request("DOMDebugger.setDOMBreakpoint", input)

  /**
   * Sets breakpoint on particular operation with DOM.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setDOMBreakpoint)
   *
   * @param nodeId Identifier of the node to set breakpoint on.
   * @param type Type of the operation to stop upon.
   */
  public suspend fun setDOMBreakpoint(nodeId: NodeId, type: DOMBreakpointType):
      SetDOMBreakpointResponse {
    val input = SetDOMBreakpointRequest(nodeId, type)
    return setDOMBreakpoint(input)
  }

  /**
   * Sets breakpoint on particular DOM event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setEventListenerBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setEventListenerBreakpoint(input: SetEventListenerBreakpointRequest):
      SetEventListenerBreakpointResponse = session.request("DOMDebugger.setEventListenerBreakpoint",
      input)

  /**
   * Sets breakpoint on particular DOM event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setEventListenerBreakpoint)
   *
   * @param eventName DOM Event name to stop on (any DOM event will do).
   */
  @JvmOverloads
  public suspend inline fun setEventListenerBreakpoint(eventName: String,
      optionalArgs: SetEventListenerBreakpointRequest.Builder.() -> Unit = {}):
      SetEventListenerBreakpointResponse {
    val builder = SetEventListenerBreakpointRequest.Builder(eventName)
    val input = builder.apply(optionalArgs).build()
    return setEventListenerBreakpoint(input)
  }

  /**
   * Sets breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setInstrumentationBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setInstrumentationBreakpoint(input: SetInstrumentationBreakpointRequest):
      SetInstrumentationBreakpointResponse =
      session.request("DOMDebugger.setInstrumentationBreakpoint", input)

  /**
   * Sets breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setInstrumentationBreakpoint)
   *
   * @param eventName Instrumentation name to stop on.
   */
  @ExperimentalChromeApi
  public suspend fun setInstrumentationBreakpoint(eventName: String):
      SetInstrumentationBreakpointResponse {
    val input = SetInstrumentationBreakpointRequest(eventName)
    return setInstrumentationBreakpoint(input)
  }

  /**
   * Sets breakpoint on XMLHttpRequest.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setXHRBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setXHRBreakpoint(input: SetXHRBreakpointRequest): SetXHRBreakpointResponse =
      session.request("DOMDebugger.setXHRBreakpoint", input)

  /**
   * Sets breakpoint on XMLHttpRequest.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMDebugger/#method-setXHRBreakpoint)
   *
   * @param url Resource URL substring. All XHRs having this substring in the URL will get stopped
   * upon.
   */
  public suspend fun setXHRBreakpoint(url: String): SetXHRBreakpointResponse {
    val input = SetXHRBreakpointRequest(url)
    return setXHRBreakpoint(input)
  }
}
