@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.domstorage

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.domstorage.events.DOMStorageEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [DOMStorageDomain.clear] command.
 */
@Serializable
public data class ClearRequest(
  public val storageId: StorageId,
)

/**
 * A dummy response object for the [DOMStorageDomain.clear] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object ClearResponse

/**
 * A dummy response object for the [DOMStorageDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [DOMStorageDomain.enable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [DOMStorageDomain.getDOMStorageItems] command.
 */
@Serializable
public data class GetDOMStorageItemsRequest(
  public val storageId: StorageId,
)

/**
 * Response type for the [DOMStorageDomain.getDOMStorageItems] command.
 */
@Serializable
public data class GetDOMStorageItemsResponse(
  public val entries: List<Item>,
)

/**
 * Request object containing input parameters for the [DOMStorageDomain.removeDOMStorageItem]
 * command.
 */
@Serializable
public data class RemoveDOMStorageItemRequest(
  public val storageId: StorageId,
  public val key: String,
)

/**
 * A dummy response object for the [DOMStorageDomain.removeDOMStorageItem] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object RemoveDOMStorageItemResponse

/**
 * Request object containing input parameters for the [DOMStorageDomain.setDOMStorageItem] command.
 */
@Serializable
public data class SetDOMStorageItemRequest(
  public val storageId: StorageId,
  public val key: String,
  public val `value`: String,
)

/**
 * A dummy response object for the [DOMStorageDomain.setDOMStorageItem] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetDOMStorageItemResponse

/**
 * Query and modify DOM storage.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage)
 */
@ExperimentalChromeApi
public class DOMStorageDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DOMStorageEvent>> =
      mapOf(
      "DOMStorage.domStorageItemAdded" to serializer<DOMStorageEvent.DomStorageItemAdded>(),
      "DOMStorage.domStorageItemRemoved" to serializer<DOMStorageEvent.DomStorageItemRemoved>(),
      "DOMStorage.domStorageItemUpdated" to serializer<DOMStorageEvent.DomStorageItemUpdated>(),
      "DOMStorage.domStorageItemsCleared" to serializer<DOMStorageEvent.DomStorageItemsCleared>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DOMStorageEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#event-domStorageItemAdded)
   */
  public fun domStorageItemAddedEvents(): Flow<DOMStorageEvent.DomStorageItemAdded> =
      session.typedEvents("DOMStorage.domStorageItemAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("domStorageItemAddedEvents()"),
  )
  public fun domStorageItemAdded(): Flow<DOMStorageEvent.DomStorageItemAdded> =
      domStorageItemAddedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#event-domStorageItemRemoved)
   */
  public fun domStorageItemRemovedEvents(): Flow<DOMStorageEvent.DomStorageItemRemoved> =
      session.typedEvents("DOMStorage.domStorageItemRemoved")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("domStorageItemRemovedEvents()"),
  )
  public fun domStorageItemRemoved(): Flow<DOMStorageEvent.DomStorageItemRemoved> =
      domStorageItemRemovedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#event-domStorageItemUpdated)
   */
  public fun domStorageItemUpdatedEvents(): Flow<DOMStorageEvent.DomStorageItemUpdated> =
      session.typedEvents("DOMStorage.domStorageItemUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("domStorageItemUpdatedEvents()"),
  )
  public fun domStorageItemUpdated(): Flow<DOMStorageEvent.DomStorageItemUpdated> =
      domStorageItemUpdatedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#event-domStorageItemsCleared)
   */
  public fun domStorageItemsClearedEvents(): Flow<DOMStorageEvent.DomStorageItemsCleared> =
      session.typedEvents("DOMStorage.domStorageItemsCleared")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("domStorageItemsClearedEvents()"),
  )
  public fun domStorageItemsCleared(): Flow<DOMStorageEvent.DomStorageItemsCleared> =
      domStorageItemsClearedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-clear)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clear(input: ClearRequest): ClearResponse = session.request("DOMStorage.clear",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-clear)
   */
  public suspend fun clear(storageId: StorageId): ClearResponse {
    val input = ClearRequest(storageId)
    return clear(input)
  }

  /**
   * Disables storage tracking, prevents storage events from being sent to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("DOMStorage.disable", Unit)

  /**
   * Enables storage tracking, storage events will now be delivered to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("DOMStorage.enable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-getDOMStorageItems)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getDOMStorageItems(input: GetDOMStorageItemsRequest):
      GetDOMStorageItemsResponse = session.request("DOMStorage.getDOMStorageItems", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-getDOMStorageItems)
   */
  public suspend fun getDOMStorageItems(storageId: StorageId): GetDOMStorageItemsResponse {
    val input = GetDOMStorageItemsRequest(storageId)
    return getDOMStorageItems(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-removeDOMStorageItem)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeDOMStorageItem(input: RemoveDOMStorageItemRequest):
      RemoveDOMStorageItemResponse = session.request("DOMStorage.removeDOMStorageItem", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-removeDOMStorageItem)
   */
  public suspend fun removeDOMStorageItem(storageId: StorageId, key: String):
      RemoveDOMStorageItemResponse {
    val input = RemoveDOMStorageItemRequest(storageId, key)
    return removeDOMStorageItem(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-setDOMStorageItem)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setDOMStorageItem(input: SetDOMStorageItemRequest): SetDOMStorageItemResponse =
      session.request("DOMStorage.setDOMStorageItem", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMStorage/#method-setDOMStorageItem)
   */
  public suspend fun setDOMStorageItem(
    storageId: StorageId,
    key: String,
    `value`: String,
  ): SetDOMStorageItemResponse {
    val input = SetDOMStorageItemRequest(storageId, key, value)
    return setDOMStorageItem(input)
  }
}
