@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.emulation

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.RGBA
import org.hildan.chrome.devtools.domains.emulation.events.EmulationEvent
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.page.Viewport
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Response type for the [EmulationDomain.canEmulate] command.
 */
@Serializable
public data class CanEmulateResponse(
  /**
   * True if emulation is supported.
   */
  public val result: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.clearDeviceMetricsOverride] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object ClearDeviceMetricsOverrideResponse

/**
 * A dummy response object for the [EmulationDomain.clearGeolocationOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object ClearGeolocationOverrideResponse

/**
 * A dummy response object for the [EmulationDomain.resetPageScaleFactor] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ResetPageScaleFactorResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setFocusEmulationEnabled]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetFocusEmulationEnabledRequest(
  /**
   * Whether to enable to disable focus emulation.
   */
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.setFocusEmulationEnabled] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetFocusEmulationEnabledResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setAutoDarkModeOverride]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAutoDarkModeOverrideRequest(
  /**
   * Whether to enable or disable automatic dark mode.
   * If not specified, any existing override will be cleared.
   */
  public val enabled: Boolean? = null,
) {
  public class Builder() {
    /**
     * Whether to enable or disable automatic dark mode.
     * If not specified, any existing override will be cleared.
     */
    public var enabled: Boolean? = null

    public fun build(): SetAutoDarkModeOverrideRequest = SetAutoDarkModeOverrideRequest(enabled)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setAutoDarkModeOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetAutoDarkModeOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setCPUThrottlingRate]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetCPUThrottlingRateRequest(
  /**
   * Throttling rate as a slowdown factor (1 is no throttle, 2 is 2x slowdown, etc).
   */
  public val rate: Double,
)

/**
 * A dummy response object for the [EmulationDomain.setCPUThrottlingRate] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetCPUThrottlingRateResponse

/**
 * Request object containing input parameters for the
 * [EmulationDomain.setDefaultBackgroundColorOverride] command.
 */
@Serializable
public data class SetDefaultBackgroundColorOverrideRequest(
  /**
   * RGBA of the default background color. If not specified, any existing override will be
   * cleared.
   */
  public val color: RGBA? = null,
) {
  public class Builder() {
    /**
     * RGBA of the default background color. If not specified, any existing override will be
     * cleared.
     */
    public var color: RGBA? = null

    public fun build(): SetDefaultBackgroundColorOverrideRequest =
        SetDefaultBackgroundColorOverrideRequest(color)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setDefaultBackgroundColorOverride] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetDefaultBackgroundColorOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setDeviceMetricsOverride]
 * command.
 */
@Serializable
public data class SetDeviceMetricsOverrideRequest(
  /**
   * Overriding width value in pixels (minimum 0, maximum 10000000). 0 disables the override.
   */
  public val width: Int,
  /**
   * Overriding height value in pixels (minimum 0, maximum 10000000). 0 disables the override.
   */
  public val height: Int,
  /**
   * Overriding device scale factor value. 0 disables the override.
   */
  public val deviceScaleFactor: Double,
  /**
   * Whether to emulate mobile device. This includes viewport meta tag, overlay scrollbars, text
   * autosizing and more.
   */
  public val mobile: Boolean,
  /**
   * Scale to apply to resulting view image.
   */
  @ExperimentalChromeApi
  public val scale: Double? = null,
  /**
   * Overriding screen width value in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val screenWidth: Int? = null,
  /**
   * Overriding screen height value in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val screenHeight: Int? = null,
  /**
   * Overriding view X position on screen in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val positionX: Int? = null,
  /**
   * Overriding view Y position on screen in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val positionY: Int? = null,
  /**
   * Do not set visible view size, rely upon explicit setVisibleSize call.
   */
  @ExperimentalChromeApi
  public val dontSetVisibleSize: Boolean? = null,
  /**
   * Screen orientation override.
   */
  public val screenOrientation: ScreenOrientation? = null,
  /**
   * If set, the visible area of the page will be overridden to this viewport. This viewport
   * change is not observed by the page, e.g. viewport-relative elements do not change positions.
   */
  @ExperimentalChromeApi
  public val viewport: Viewport? = null,
  /**
   * If set, the display feature of a multi-segment screen. If not set, multi-segment support
   * is turned-off.
   */
  @ExperimentalChromeApi
  public val displayFeature: DisplayFeature? = null,
) {
  public class Builder(
    /**
     * Overriding width value in pixels (minimum 0, maximum 10000000). 0 disables the override.
     */
    public val width: Int,
    /**
     * Overriding height value in pixels (minimum 0, maximum 10000000). 0 disables the override.
     */
    public val height: Int,
    /**
     * Overriding device scale factor value. 0 disables the override.
     */
    public val deviceScaleFactor: Double,
    /**
     * Whether to emulate mobile device. This includes viewport meta tag, overlay scrollbars, text
     * autosizing and more.
     */
    public val mobile: Boolean,
  ) {
    /**
     * Scale to apply to resulting view image.
     */
    @ExperimentalChromeApi
    public var scale: Double? = null

    /**
     * Overriding screen width value in pixels (minimum 0, maximum 10000000).
     */
    @ExperimentalChromeApi
    public var screenWidth: Int? = null

    /**
     * Overriding screen height value in pixels (minimum 0, maximum 10000000).
     */
    @ExperimentalChromeApi
    public var screenHeight: Int? = null

    /**
     * Overriding view X position on screen in pixels (minimum 0, maximum 10000000).
     */
    @ExperimentalChromeApi
    public var positionX: Int? = null

    /**
     * Overriding view Y position on screen in pixels (minimum 0, maximum 10000000).
     */
    @ExperimentalChromeApi
    public var positionY: Int? = null

    /**
     * Do not set visible view size, rely upon explicit setVisibleSize call.
     */
    @ExperimentalChromeApi
    public var dontSetVisibleSize: Boolean? = null

    /**
     * Screen orientation override.
     */
    public var screenOrientation: ScreenOrientation? = null

    /**
     * If set, the visible area of the page will be overridden to this viewport. This viewport
     * change is not observed by the page, e.g. viewport-relative elements do not change positions.
     */
    @ExperimentalChromeApi
    public var viewport: Viewport? = null

    /**
     * If set, the display feature of a multi-segment screen. If not set, multi-segment support
     * is turned-off.
     */
    @ExperimentalChromeApi
    public var displayFeature: DisplayFeature? = null

    public fun build(): SetDeviceMetricsOverrideRequest = SetDeviceMetricsOverrideRequest(width,
        height, deviceScaleFactor, mobile, scale, screenWidth, screenHeight, positionX, positionY,
        dontSetVisibleSize, screenOrientation, viewport, displayFeature)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setDeviceMetricsOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetDeviceMetricsOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setScrollbarsHidden] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetScrollbarsHiddenRequest(
  /**
   * Whether scrollbars should be always hidden.
   */
  public val hidden: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.setScrollbarsHidden] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetScrollbarsHiddenResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setDocumentCookieDisabled]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDocumentCookieDisabledRequest(
  /**
   * Whether document.coookie API should be disabled.
   */
  public val disabled: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.setDocumentCookieDisabled] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetDocumentCookieDisabledResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setEmitTouchEventsForMouse]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetEmitTouchEventsForMouseRequest(
  /**
   * Whether touch emulation based on mouse input should be enabled.
   */
  public val enabled: Boolean,
  /**
   * Touch/gesture events configuration. Default: current platform.
   */
  public val configuration: TouchEventsConfiguration? = null,
) {
  public class Builder(
    /**
     * Whether touch emulation based on mouse input should be enabled.
     */
    public val enabled: Boolean,
  ) {
    /**
     * Touch/gesture events configuration. Default: current platform.
     */
    public var configuration: TouchEventsConfiguration? = null

    public fun build(): SetEmitTouchEventsForMouseRequest =
        SetEmitTouchEventsForMouseRequest(enabled, configuration)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setEmitTouchEventsForMouse] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetEmitTouchEventsForMouseResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setEmulatedMedia] command.
 */
@Serializable
public data class SetEmulatedMediaRequest(
  /**
   * Media type to emulate. Empty string disables the override.
   */
  public val media: String? = null,
  /**
   * Media features to emulate.
   */
  public val features: List<MediaFeature>? = null,
) {
  public class Builder() {
    /**
     * Media type to emulate. Empty string disables the override.
     */
    public var media: String? = null

    /**
     * Media features to emulate.
     */
    public var features: List<MediaFeature>? = null

    public fun build(): SetEmulatedMediaRequest = SetEmulatedMediaRequest(media, features)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setEmulatedMedia] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetEmulatedMediaResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setEmulatedVisionDeficiency]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetEmulatedVisionDeficiencyRequest(
  /**
   * Vision deficiency to emulate.
   */
  public val type: EmulatedVisionDeficiencyType,
)

/**
 * A dummy response object for the [EmulationDomain.setEmulatedVisionDeficiency] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetEmulatedVisionDeficiencyResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setGeolocationOverride]
 * command.
 */
@Serializable
public data class SetGeolocationOverrideRequest(
  /**
   * Mock latitude
   */
  public val latitude: Double? = null,
  /**
   * Mock longitude
   */
  public val longitude: Double? = null,
  /**
   * Mock accuracy
   */
  public val accuracy: Double? = null,
) {
  public class Builder() {
    /**
     * Mock latitude
     */
    public var latitude: Double? = null

    /**
     * Mock longitude
     */
    public var longitude: Double? = null

    /**
     * Mock accuracy
     */
    public var accuracy: Double? = null

    public fun build(): SetGeolocationOverrideRequest = SetGeolocationOverrideRequest(latitude,
        longitude, accuracy)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setGeolocationOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetGeolocationOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setIdleOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetIdleOverrideRequest(
  /**
   * Mock isUserActive
   */
  public val isUserActive: Boolean,
  /**
   * Mock isScreenUnlocked
   */
  public val isScreenUnlocked: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.setIdleOverride] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetIdleOverrideResponse

/**
 * A dummy response object for the [EmulationDomain.clearIdleOverride] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ClearIdleOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setNavigatorOverrides]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetNavigatorOverridesRequest(
  /**
   * The platform navigator.platform should return.
   */
  public val platform: String,
)

/**
 * A dummy response object for the [EmulationDomain.setNavigatorOverrides] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public object SetNavigatorOverridesResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setPageScaleFactor] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetPageScaleFactorRequest(
  /**
   * Page scale factor.
   */
  public val pageScaleFactor: Double,
)

/**
 * A dummy response object for the [EmulationDomain.setPageScaleFactor] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetPageScaleFactorResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setScriptExecutionDisabled]
 * command.
 */
@Serializable
public data class SetScriptExecutionDisabledRequest(
  /**
   * Whether script execution should be disabled in the page.
   */
  public val `value`: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.setScriptExecutionDisabled] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetScriptExecutionDisabledResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setTouchEmulationEnabled]
 * command.
 */
@Serializable
public data class SetTouchEmulationEnabledRequest(
  /**
   * Whether the touch event emulation should be enabled.
   */
  public val enabled: Boolean,
  /**
   * Maximum touch points supported. Defaults to one.
   */
  public val maxTouchPoints: Int? = null,
) {
  public class Builder(
    /**
     * Whether the touch event emulation should be enabled.
     */
    public val enabled: Boolean,
  ) {
    /**
     * Maximum touch points supported. Defaults to one.
     */
    public var maxTouchPoints: Int? = null

    public fun build(): SetTouchEmulationEnabledRequest = SetTouchEmulationEnabledRequest(enabled,
        maxTouchPoints)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setTouchEmulationEnabled] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetTouchEmulationEnabledResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setVirtualTimePolicy]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetVirtualTimePolicyRequest(
  public val policy: VirtualTimePolicy,
  /**
   * If set, after this many virtual milliseconds have elapsed virtual time will be paused and a
   * virtualTimeBudgetExpired event is sent.
   */
  public val budget: Double? = null,
  /**
   * If set this specifies the maximum number of tasks that can be run before virtual is forced
   * forwards to prevent deadlock.
   */
  public val maxVirtualTimeTaskStarvationCount: Int? = null,
  /**
   * If set, base::Time::Now will be overridden to initially return this value.
   */
  public val initialVirtualTime: TimeSinceEpoch? = null,
) {
  public class Builder(
    public val policy: VirtualTimePolicy,
  ) {
    /**
     * If set, after this many virtual milliseconds have elapsed virtual time will be paused and a
     * virtualTimeBudgetExpired event is sent.
     */
    public var budget: Double? = null

    /**
     * If set this specifies the maximum number of tasks that can be run before virtual is forced
     * forwards to prevent deadlock.
     */
    public var maxVirtualTimeTaskStarvationCount: Int? = null

    /**
     * If set, base::Time::Now will be overridden to initially return this value.
     */
    public var initialVirtualTime: TimeSinceEpoch? = null

    public fun build(): SetVirtualTimePolicyRequest = SetVirtualTimePolicyRequest(policy, budget,
        maxVirtualTimeTaskStarvationCount, initialVirtualTime)
  }
}

/**
 * Response type for the [EmulationDomain.setVirtualTimePolicy] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetVirtualTimePolicyResponse(
  /**
   * Absolute timestamp at which virtual time was first enabled (up time in milliseconds).
   */
  public val virtualTimeTicksBase: Double,
)

/**
 * Request object containing input parameters for the [EmulationDomain.setLocaleOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetLocaleOverrideRequest(
  /**
   * ICU style C locale (e.g. "en_US"). If not specified or empty, disables the override and
   * restores default host system locale.
   */
  public val locale: String? = null,
) {
  public class Builder() {
    /**
     * ICU style C locale (e.g. "en_US"). If not specified or empty, disables the override and
     * restores default host system locale.
     */
    public var locale: String? = null

    public fun build(): SetLocaleOverrideRequest = SetLocaleOverrideRequest(locale)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setLocaleOverride] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetLocaleOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setTimezoneOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetTimezoneOverrideRequest(
  /**
   * The timezone identifier. If empty, disables the override and
   * restores default host system timezone.
   */
  public val timezoneId: String,
)

/**
 * A dummy response object for the [EmulationDomain.setTimezoneOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetTimezoneOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setVisibleSize] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetVisibleSizeRequest(
  /**
   * Frame width (DIP).
   */
  public val width: Int,
  /**
   * Frame height (DIP).
   */
  public val height: Int,
)

/**
 * A dummy response object for the [EmulationDomain.setVisibleSize] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public object SetVisibleSizeResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setDisabledImageTypes]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDisabledImageTypesRequest(
  /**
   * Image types to disable.
   */
  public val imageTypes: List<DisabledImageType>,
)

/**
 * A dummy response object for the [EmulationDomain.setDisabledImageTypes] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetDisabledImageTypesResponse

/**
 * Request object containing input parameters for the
 * [EmulationDomain.setHardwareConcurrencyOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetHardwareConcurrencyOverrideRequest(
  /**
   * Hardware concurrency to report
   */
  public val hardwareConcurrency: Int,
)

/**
 * A dummy response object for the [EmulationDomain.setHardwareConcurrencyOverride] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetHardwareConcurrencyOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setUserAgentOverride]
 * command.
 */
@Serializable
public data class SetUserAgentOverrideRequest(
  /**
   * User agent to use.
   */
  public val userAgent: String,
  /**
   * Browser langugage to emulate.
   */
  public val acceptLanguage: String? = null,
  /**
   * The platform navigator.platform should return.
   */
  public val platform: String? = null,
  /**
   * To be sent in Sec-CH-UA-* headers and returned in navigator.userAgentData
   */
  @ExperimentalChromeApi
  public val userAgentMetadata: UserAgentMetadata? = null,
) {
  public class Builder(
    /**
     * User agent to use.
     */
    public val userAgent: String,
  ) {
    /**
     * Browser langugage to emulate.
     */
    public var acceptLanguage: String? = null

    /**
     * The platform navigator.platform should return.
     */
    public var platform: String? = null

    /**
     * To be sent in Sec-CH-UA-* headers and returned in navigator.userAgentData
     */
    @ExperimentalChromeApi
    public var userAgentMetadata: UserAgentMetadata? = null

    public fun build(): SetUserAgentOverrideRequest = SetUserAgentOverrideRequest(userAgent,
        acceptLanguage, platform, userAgentMetadata)
  }
}

/**
 * A dummy response object for the [EmulationDomain.setUserAgentOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetUserAgentOverrideResponse

/**
 * Request object containing input parameters for the [EmulationDomain.setAutomationOverride]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAutomationOverrideRequest(
  /**
   * Whether the override should be enabled.
   */
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [EmulationDomain.setAutomationOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetAutomationOverrideResponse

/**
 * This domain emulates different environments for the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation)
 */
public class EmulationDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out EmulationEvent>> =
      mapOf(
      "Emulation.virtualTimeBudgetExpired" to serializer<EmulationEvent.VirtualTimeBudgetExpired>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<EmulationEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Notification sent after the virtual time budget for the current VirtualTimePolicy has run out.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#event-virtualTimeBudgetExpired)
   */
  @ExperimentalChromeApi
  public fun virtualTimeBudgetExpiredEvents(): Flow<EmulationEvent.VirtualTimeBudgetExpired> =
      session.typedEvents("Emulation.virtualTimeBudgetExpired")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("virtualTimeBudgetExpiredEvents()"),
  )
  public fun virtualTimeBudgetExpired(): Flow<EmulationEvent.VirtualTimeBudgetExpired> =
      virtualTimeBudgetExpiredEvents()

  /**
   * Tells whether emulation is supported.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-canEmulate)
   */
  public suspend fun canEmulate(): CanEmulateResponse = session.request("Emulation.canEmulate",
      Unit)

  /**
   * Clears the overridden device metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-clearDeviceMetricsOverride)
   */
  public suspend fun clearDeviceMetricsOverride(): ClearDeviceMetricsOverrideResponse =
      session.request("Emulation.clearDeviceMetricsOverride", Unit)

  /**
   * Clears the overridden Geolocation Position and Error.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-clearGeolocationOverride)
   */
  public suspend fun clearGeolocationOverride(): ClearGeolocationOverrideResponse =
      session.request("Emulation.clearGeolocationOverride", Unit)

  /**
   * Requests that page scale factor is reset to initial values.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-resetPageScaleFactor)
   */
  @ExperimentalChromeApi
  public suspend fun resetPageScaleFactor(): ResetPageScaleFactorResponse =
      session.request("Emulation.resetPageScaleFactor", Unit)

  /**
   * Enables or disables simulating a focused and active page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setFocusEmulationEnabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setFocusEmulationEnabled(input: SetFocusEmulationEnabledRequest):
      SetFocusEmulationEnabledResponse = session.request("Emulation.setFocusEmulationEnabled",
      input)

  /**
   * Enables or disables simulating a focused and active page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setFocusEmulationEnabled)
   *
   * @param enabled Whether to enable to disable focus emulation.
   */
  @ExperimentalChromeApi
  public suspend fun setFocusEmulationEnabled(enabled: Boolean): SetFocusEmulationEnabledResponse {
    val input = SetFocusEmulationEnabledRequest(enabled)
    return setFocusEmulationEnabled(input)
  }

  /**
   * Automatically render all web contents using a dark theme.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setAutoDarkModeOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setAutoDarkModeOverride(input: SetAutoDarkModeOverrideRequest):
      SetAutoDarkModeOverrideResponse = session.request("Emulation.setAutoDarkModeOverride", input)

  /**
   * Automatically render all web contents using a dark theme.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setAutoDarkModeOverride)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun setAutoDarkModeOverride(optionalArgs: SetAutoDarkModeOverrideRequest.Builder.() -> Unit =
      {}): SetAutoDarkModeOverrideResponse {
    val builder = SetAutoDarkModeOverrideRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setAutoDarkModeOverride(input)
  }

  /**
   * Enables CPU throttling to emulate slow CPUs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setCPUThrottlingRate)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setCPUThrottlingRate(input: SetCPUThrottlingRateRequest):
      SetCPUThrottlingRateResponse = session.request("Emulation.setCPUThrottlingRate", input)

  /**
   * Enables CPU throttling to emulate slow CPUs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setCPUThrottlingRate)
   *
   * @param rate Throttling rate as a slowdown factor (1 is no throttle, 2 is 2x slowdown, etc).
   */
  @ExperimentalChromeApi
  public suspend fun setCPUThrottlingRate(rate: Double): SetCPUThrottlingRateResponse {
    val input = SetCPUThrottlingRateRequest(rate)
    return setCPUThrottlingRate(input)
  }

  /**
   * Sets or clears an override of the default background color of the frame. This override is used
   * if the content does not specify one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDefaultBackgroundColorOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend
      fun setDefaultBackgroundColorOverride(input: SetDefaultBackgroundColorOverrideRequest):
      SetDefaultBackgroundColorOverrideResponse =
      session.request("Emulation.setDefaultBackgroundColorOverride", input)

  /**
   * Sets or clears an override of the default background color of the frame. This override is used
   * if the content does not specify one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDefaultBackgroundColorOverride)
   */
  @JvmOverloads
  public suspend inline
      fun setDefaultBackgroundColorOverride(optionalArgs: SetDefaultBackgroundColorOverrideRequest.Builder.() -> Unit
      = {}): SetDefaultBackgroundColorOverrideResponse {
    val builder = SetDefaultBackgroundColorOverrideRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setDefaultBackgroundColorOverride(input)
  }

  /**
   * Overrides the values of device screen dimensions (window.screen.width, window.screen.height,
   * window.innerWidth, window.innerHeight, and "device-width"/"device-height"-related CSS media
   * query results).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDeviceMetricsOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setDeviceMetricsOverride(input: SetDeviceMetricsOverrideRequest):
      SetDeviceMetricsOverrideResponse = session.request("Emulation.setDeviceMetricsOverride",
      input)

  /**
   * Overrides the values of device screen dimensions (window.screen.width, window.screen.height,
   * window.innerWidth, window.innerHeight, and "device-width"/"device-height"-related CSS media
   * query results).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDeviceMetricsOverride)
   *
   * @param width Overriding width value in pixels (minimum 0, maximum 10000000). 0 disables the
   * override.
   * @param height Overriding height value in pixels (minimum 0, maximum 10000000). 0 disables the
   * override.
   * @param deviceScaleFactor Overriding device scale factor value. 0 disables the override.
   * @param mobile Whether to emulate mobile device. This includes viewport meta tag, overlay
   * scrollbars, text
   * autosizing and more.
   */
  @JvmOverloads
  public suspend inline fun setDeviceMetricsOverride(
    width: Int,
    height: Int,
    deviceScaleFactor: Double,
    mobile: Boolean,
    optionalArgs: SetDeviceMetricsOverrideRequest.Builder.() -> Unit = {},
  ): SetDeviceMetricsOverrideResponse {
    val builder = SetDeviceMetricsOverrideRequest.Builder(width, height, deviceScaleFactor, mobile)
    val input = builder.apply(optionalArgs).build()
    return setDeviceMetricsOverride(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setScrollbarsHidden)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setScrollbarsHidden(input: SetScrollbarsHiddenRequest):
      SetScrollbarsHiddenResponse = session.request("Emulation.setScrollbarsHidden", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setScrollbarsHidden)
   *
   * @param hidden Whether scrollbars should be always hidden.
   */
  @ExperimentalChromeApi
  public suspend fun setScrollbarsHidden(hidden: Boolean): SetScrollbarsHiddenResponse {
    val input = SetScrollbarsHiddenRequest(hidden)
    return setScrollbarsHidden(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDocumentCookieDisabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setDocumentCookieDisabled(input: SetDocumentCookieDisabledRequest):
      SetDocumentCookieDisabledResponse = session.request("Emulation.setDocumentCookieDisabled",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDocumentCookieDisabled)
   *
   * @param disabled Whether document.coookie API should be disabled.
   */
  @ExperimentalChromeApi
  public suspend fun setDocumentCookieDisabled(disabled: Boolean):
      SetDocumentCookieDisabledResponse {
    val input = SetDocumentCookieDisabledRequest(disabled)
    return setDocumentCookieDisabled(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmitTouchEventsForMouse)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setEmitTouchEventsForMouse(input: SetEmitTouchEventsForMouseRequest):
      SetEmitTouchEventsForMouseResponse = session.request("Emulation.setEmitTouchEventsForMouse",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmitTouchEventsForMouse)
   *
   * @param enabled Whether touch emulation based on mouse input should be enabled.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setEmitTouchEventsForMouse(enabled: Boolean,
      optionalArgs: SetEmitTouchEventsForMouseRequest.Builder.() -> Unit = {}):
      SetEmitTouchEventsForMouseResponse {
    val builder = SetEmitTouchEventsForMouseRequest.Builder(enabled)
    val input = builder.apply(optionalArgs).build()
    return setEmitTouchEventsForMouse(input)
  }

  /**
   * Emulates the given media type or media feature for CSS media queries.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmulatedMedia)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setEmulatedMedia(input: SetEmulatedMediaRequest): SetEmulatedMediaResponse =
      session.request("Emulation.setEmulatedMedia", input)

  /**
   * Emulates the given media type or media feature for CSS media queries.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmulatedMedia)
   */
  @JvmOverloads
  public suspend inline
      fun setEmulatedMedia(optionalArgs: SetEmulatedMediaRequest.Builder.() -> Unit = {}):
      SetEmulatedMediaResponse {
    val builder = SetEmulatedMediaRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setEmulatedMedia(input)
  }

  /**
   * Emulates the given vision deficiency.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmulatedVisionDeficiency)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setEmulatedVisionDeficiency(input: SetEmulatedVisionDeficiencyRequest):
      SetEmulatedVisionDeficiencyResponse = session.request("Emulation.setEmulatedVisionDeficiency",
      input)

  /**
   * Emulates the given vision deficiency.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmulatedVisionDeficiency)
   *
   * @param type Vision deficiency to emulate.
   */
  @ExperimentalChromeApi
  public suspend fun setEmulatedVisionDeficiency(type: EmulatedVisionDeficiencyType):
      SetEmulatedVisionDeficiencyResponse {
    val input = SetEmulatedVisionDeficiencyRequest(type)
    return setEmulatedVisionDeficiency(input)
  }

  /**
   * Overrides the Geolocation Position or Error. Omitting any of the parameters emulates position
   * unavailable.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setGeolocationOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setGeolocationOverride(input: SetGeolocationOverrideRequest):
      SetGeolocationOverrideResponse = session.request("Emulation.setGeolocationOverride", input)

  /**
   * Overrides the Geolocation Position or Error. Omitting any of the parameters emulates position
   * unavailable.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setGeolocationOverride)
   */
  @JvmOverloads
  public suspend inline
      fun setGeolocationOverride(optionalArgs: SetGeolocationOverrideRequest.Builder.() -> Unit =
      {}): SetGeolocationOverrideResponse {
    val builder = SetGeolocationOverrideRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setGeolocationOverride(input)
  }

  /**
   * Overrides the Idle state.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setIdleOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setIdleOverride(input: SetIdleOverrideRequest): SetIdleOverrideResponse =
      session.request("Emulation.setIdleOverride", input)

  /**
   * Overrides the Idle state.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setIdleOverride)
   *
   * @param isUserActive Mock isUserActive
   * @param isScreenUnlocked Mock isScreenUnlocked
   */
  @ExperimentalChromeApi
  public suspend fun setIdleOverride(isUserActive: Boolean, isScreenUnlocked: Boolean):
      SetIdleOverrideResponse {
    val input = SetIdleOverrideRequest(isUserActive, isScreenUnlocked)
    return setIdleOverride(input)
  }

  /**
   * Clears Idle state overrides.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-clearIdleOverride)
   */
  @ExperimentalChromeApi
  public suspend fun clearIdleOverride(): ClearIdleOverrideResponse =
      session.request("Emulation.clearIdleOverride", Unit)

  /**
   * Overrides value returned by the javascript navigator object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setNavigatorOverrides)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setNavigatorOverrides(input: SetNavigatorOverridesRequest):
      SetNavigatorOverridesResponse = session.request("Emulation.setNavigatorOverrides", input)

  /**
   * Overrides value returned by the javascript navigator object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setNavigatorOverrides)
   *
   * @param platform The platform navigator.platform should return.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setNavigatorOverrides(platform: String): SetNavigatorOverridesResponse {
    val input = SetNavigatorOverridesRequest(platform)
    return setNavigatorOverrides(input)
  }

  /**
   * Sets a specified page scale factor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setPageScaleFactor)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setPageScaleFactor(input: SetPageScaleFactorRequest):
      SetPageScaleFactorResponse = session.request("Emulation.setPageScaleFactor", input)

  /**
   * Sets a specified page scale factor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setPageScaleFactor)
   *
   * @param pageScaleFactor Page scale factor.
   */
  @ExperimentalChromeApi
  public suspend fun setPageScaleFactor(pageScaleFactor: Double): SetPageScaleFactorResponse {
    val input = SetPageScaleFactorRequest(pageScaleFactor)
    return setPageScaleFactor(input)
  }

  /**
   * Switches script execution in the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setScriptExecutionDisabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setScriptExecutionDisabled(input: SetScriptExecutionDisabledRequest):
      SetScriptExecutionDisabledResponse = session.request("Emulation.setScriptExecutionDisabled",
      input)

  /**
   * Switches script execution in the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setScriptExecutionDisabled)
   *
   * @param value Whether script execution should be disabled in the page.
   */
  public suspend fun setScriptExecutionDisabled(`value`: Boolean):
      SetScriptExecutionDisabledResponse {
    val input = SetScriptExecutionDisabledRequest(value)
    return setScriptExecutionDisabled(input)
  }

  /**
   * Enables touch on platforms which do not support them.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setTouchEmulationEnabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setTouchEmulationEnabled(input: SetTouchEmulationEnabledRequest):
      SetTouchEmulationEnabledResponse = session.request("Emulation.setTouchEmulationEnabled",
      input)

  /**
   * Enables touch on platforms which do not support them.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setTouchEmulationEnabled)
   *
   * @param enabled Whether the touch event emulation should be enabled.
   */
  @JvmOverloads
  public suspend inline fun setTouchEmulationEnabled(enabled: Boolean,
      optionalArgs: SetTouchEmulationEnabledRequest.Builder.() -> Unit = {}):
      SetTouchEmulationEnabledResponse {
    val builder = SetTouchEmulationEnabledRequest.Builder(enabled)
    val input = builder.apply(optionalArgs).build()
    return setTouchEmulationEnabled(input)
  }

  /**
   * Turns on virtual time for all frames (replacing real-time with a synthetic time source) and
   * sets
   * the current virtual time policy.  Note this supersedes any previous time budget.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setVirtualTimePolicy)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setVirtualTimePolicy(input: SetVirtualTimePolicyRequest):
      SetVirtualTimePolicyResponse = session.request("Emulation.setVirtualTimePolicy", input)

  /**
   * Turns on virtual time for all frames (replacing real-time with a synthetic time source) and
   * sets
   * the current virtual time policy.  Note this supersedes any previous time budget.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setVirtualTimePolicy)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setVirtualTimePolicy(policy: VirtualTimePolicy,
      optionalArgs: SetVirtualTimePolicyRequest.Builder.() -> Unit = {}):
      SetVirtualTimePolicyResponse {
    val builder = SetVirtualTimePolicyRequest.Builder(policy)
    val input = builder.apply(optionalArgs).build()
    return setVirtualTimePolicy(input)
  }

  /**
   * Overrides default host system locale with the specified one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setLocaleOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setLocaleOverride(input: SetLocaleOverrideRequest): SetLocaleOverrideResponse =
      session.request("Emulation.setLocaleOverride", input)

  /**
   * Overrides default host system locale with the specified one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setLocaleOverride)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun setLocaleOverride(optionalArgs: SetLocaleOverrideRequest.Builder.() -> Unit = {}):
      SetLocaleOverrideResponse {
    val builder = SetLocaleOverrideRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setLocaleOverride(input)
  }

  /**
   * Overrides default host system timezone with the specified one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setTimezoneOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setTimezoneOverride(input: SetTimezoneOverrideRequest):
      SetTimezoneOverrideResponse = session.request("Emulation.setTimezoneOverride", input)

  /**
   * Overrides default host system timezone with the specified one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setTimezoneOverride)
   *
   * @param timezoneId The timezone identifier. If empty, disables the override and
   * restores default host system timezone.
   */
  @ExperimentalChromeApi
  public suspend fun setTimezoneOverride(timezoneId: String): SetTimezoneOverrideResponse {
    val input = SetTimezoneOverrideRequest(timezoneId)
    return setTimezoneOverride(input)
  }

  /**
   * Resizes the frame/viewport of the page. Note that this does not affect the frame's container
   * (e.g. browser window). Can be used to produce screenshots of the specified size. Not supported
   * on Android.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setVisibleSize)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setVisibleSize(input: SetVisibleSizeRequest): SetVisibleSizeResponse =
      session.request("Emulation.setVisibleSize", input)

  /**
   * Resizes the frame/viewport of the page. Note that this does not affect the frame's container
   * (e.g. browser window). Can be used to produce screenshots of the specified size. Not supported
   * on Android.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setVisibleSize)
   *
   * @param width Frame width (DIP).
   * @param height Frame height (DIP).
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setVisibleSize(width: Int, height: Int): SetVisibleSizeResponse {
    val input = SetVisibleSizeRequest(width, height)
    return setVisibleSize(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDisabledImageTypes)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setDisabledImageTypes(input: SetDisabledImageTypesRequest):
      SetDisabledImageTypesResponse = session.request("Emulation.setDisabledImageTypes", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDisabledImageTypes)
   *
   * @param imageTypes Image types to disable.
   */
  @ExperimentalChromeApi
  public suspend fun setDisabledImageTypes(imageTypes: List<DisabledImageType>):
      SetDisabledImageTypesResponse {
    val input = SetDisabledImageTypesRequest(imageTypes)
    return setDisabledImageTypes(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setHardwareConcurrencyOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setHardwareConcurrencyOverride(input: SetHardwareConcurrencyOverrideRequest):
      SetHardwareConcurrencyOverrideResponse =
      session.request("Emulation.setHardwareConcurrencyOverride", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setHardwareConcurrencyOverride)
   *
   * @param hardwareConcurrency Hardware concurrency to report
   */
  @ExperimentalChromeApi
  public suspend fun setHardwareConcurrencyOverride(hardwareConcurrency: Int):
      SetHardwareConcurrencyOverrideResponse {
    val input = SetHardwareConcurrencyOverrideRequest(hardwareConcurrency)
    return setHardwareConcurrencyOverride(input)
  }

  /**
   * Allows overriding user agent with the given string.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setUserAgentOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setUserAgentOverride(input: SetUserAgentOverrideRequest):
      SetUserAgentOverrideResponse = session.request("Emulation.setUserAgentOverride", input)

  /**
   * Allows overriding user agent with the given string.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setUserAgentOverride)
   *
   * @param userAgent User agent to use.
   */
  @JvmOverloads
  public suspend inline fun setUserAgentOverride(userAgent: String,
      optionalArgs: SetUserAgentOverrideRequest.Builder.() -> Unit = {}):
      SetUserAgentOverrideResponse {
    val builder = SetUserAgentOverrideRequest.Builder(userAgent)
    val input = builder.apply(optionalArgs).build()
    return setUserAgentOverride(input)
  }

  /**
   * Allows overriding the automation flag.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setAutomationOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setAutomationOverride(input: SetAutomationOverrideRequest):
      SetAutomationOverrideResponse = session.request("Emulation.setAutomationOverride", input)

  /**
   * Allows overriding the automation flag.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setAutomationOverride)
   *
   * @param enabled Whether the override should be enabled.
   */
  @ExperimentalChromeApi
  public suspend fun setAutomationOverride(enabled: Boolean): SetAutomationOverrideResponse {
    val input = SetAutomationOverrideRequest(enabled)
    return setAutomationOverride(input)
  }
}
