@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.eventbreakpoints

import kotlin.String
import kotlin.Suppress
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the
 * [EventBreakpointsDomain.setInstrumentationBreakpoint] command.
 */
@Serializable
public data class SetInstrumentationBreakpointRequest(
  /**
   * Instrumentation name to stop on.
   */
  public val eventName: String,
)

/**
 * A dummy response object for the [EventBreakpointsDomain.setInstrumentationBreakpoint] command.
 * This command doesn't return any result at the moment, but this could happen in the future, or could
 * have happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetInstrumentationBreakpointResponse

/**
 * Request object containing input parameters for the
 * [EventBreakpointsDomain.removeInstrumentationBreakpoint] command.
 */
@Serializable
public data class RemoveInstrumentationBreakpointRequest(
  /**
   * Instrumentation name to stop on.
   */
  public val eventName: String,
)

/**
 * A dummy response object for the [EventBreakpointsDomain.removeInstrumentationBreakpoint] command.
 * This command doesn't return any result at the moment, but this could happen in the future, or could
 * have happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object RemoveInstrumentationBreakpointResponse

/**
 * EventBreakpoints permits setting breakpoints on particular operations and
 * events in targets that run JavaScript but do not have a DOM.
 * JavaScript execution will stop on these operations as if there was a regular
 * breakpoint set.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/EventBreakpoints)
 */
@ExperimentalChromeApi
public class EventBreakpointsDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Sets breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/EventBreakpoints/#method-setInstrumentationBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setInstrumentationBreakpoint(input: SetInstrumentationBreakpointRequest):
      SetInstrumentationBreakpointResponse =
      session.request("EventBreakpoints.setInstrumentationBreakpoint", input)

  /**
   * Sets breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/EventBreakpoints/#method-setInstrumentationBreakpoint)
   *
   * @param eventName Instrumentation name to stop on.
   */
  public suspend fun setInstrumentationBreakpoint(eventName: String):
      SetInstrumentationBreakpointResponse {
    val input = SetInstrumentationBreakpointRequest(eventName)
    return setInstrumentationBreakpoint(input)
  }

  /**
   * Removes breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/EventBreakpoints/#method-removeInstrumentationBreakpoint)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeInstrumentationBreakpoint(input: RemoveInstrumentationBreakpointRequest):
      RemoveInstrumentationBreakpointResponse =
      session.request("EventBreakpoints.removeInstrumentationBreakpoint", input)

  /**
   * Removes breakpoint on particular native event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/EventBreakpoints/#method-removeInstrumentationBreakpoint)
   *
   * @param eventName Instrumentation name to stop on.
   */
  public suspend fun removeInstrumentationBreakpoint(eventName: String):
      RemoveInstrumentationBreakpointResponse {
    val input = RemoveInstrumentationBreakpointRequest(eventName)
    return removeInstrumentationBreakpoint(input)
  }
}
