@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.fetch

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.fetch.events.FetchEvent
import org.hildan.chrome.devtools.domains.io.StreamHandle
import org.hildan.chrome.devtools.domains.network.ErrorReason
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [FetchDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [FetchDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * If specified, only requests matching any of these patterns will produce
   * fetchRequested event and will be paused until clients response. If not set,
   * all requests will be affected.
   */
  public val patterns: List<RequestPattern>? = null,
  /**
   * If true, authRequired events will be issued and requests will be paused
   * expecting a call to continueWithAuth.
   */
  public val handleAuthRequests: Boolean? = null,
) {
  public class Builder() {
    /**
     * If specified, only requests matching any of these patterns will produce
     * fetchRequested event and will be paused until clients response. If not set,
     * all requests will be affected.
     */
    public var patterns: List<RequestPattern>? = null

    /**
     * If true, authRequired events will be issued and requests will be paused
     * expecting a call to continueWithAuth.
     */
    public var handleAuthRequests: Boolean? = null

    public fun build(): EnableRequest = EnableRequest(patterns, handleAuthRequests)
  }
}

/**
 * A dummy response object for the [FetchDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [FetchDomain.failRequest] command.
 */
@Serializable
public data class FailRequestRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * Causes the request to fail with the given reason.
   */
  public val errorReason: ErrorReason,
)

/**
 * A dummy response object for the [FetchDomain.failRequest] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object FailRequestResponse

/**
 * Request object containing input parameters for the [FetchDomain.fulfillRequest] command.
 */
@Serializable
public data class FulfillRequestRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * An HTTP response code.
   */
  public val responseCode: Int,
  /**
   * Response headers.
   */
  public val responseHeaders: List<HeaderEntry>? = null,
  /**
   * Alternative way of specifying response headers as a \0-separated
   * series of name: value pairs. Prefer the above method unless you
   * need to represent some non-UTF8 values that can't be transmitted
   * over the protocol as text. (Encoded as a base64 string when passed over JSON)
   */
  public val binaryResponseHeaders: String? = null,
  /**
   * A response body. If absent, original response body will be used if
   * the request is intercepted at the response stage and empty body
   * will be used if the request is intercepted at the request stage. (Encoded as a base64 string
   * when passed over JSON)
   */
  public val body: String? = null,
  /**
   * A textual representation of responseCode.
   * If absent, a standard phrase matching responseCode is used.
   */
  public val responsePhrase: String? = null,
) {
  public class Builder(
    /**
     * An id the client received in requestPaused event.
     */
    public val requestId: RequestId,
    /**
     * An HTTP response code.
     */
    public val responseCode: Int,
  ) {
    /**
     * Response headers.
     */
    public var responseHeaders: List<HeaderEntry>? = null

    /**
     * Alternative way of specifying response headers as a \0-separated
     * series of name: value pairs. Prefer the above method unless you
     * need to represent some non-UTF8 values that can't be transmitted
     * over the protocol as text. (Encoded as a base64 string when passed over JSON)
     */
    public var binaryResponseHeaders: String? = null

    /**
     * A response body. If absent, original response body will be used if
     * the request is intercepted at the response stage and empty body
     * will be used if the request is intercepted at the request stage. (Encoded as a base64 string
     * when passed over JSON)
     */
    public var body: String? = null

    /**
     * A textual representation of responseCode.
     * If absent, a standard phrase matching responseCode is used.
     */
    public var responsePhrase: String? = null

    public fun build(): FulfillRequestRequest = FulfillRequestRequest(requestId, responseCode,
        responseHeaders, binaryResponseHeaders, body, responsePhrase)
  }
}

/**
 * A dummy response object for the [FetchDomain.fulfillRequest] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object FulfillRequestResponse

/**
 * Request object containing input parameters for the [FetchDomain.continueRequest] command.
 */
@Serializable
public data class ContinueRequestRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * If set, the request url will be modified in a way that's not observable by page.
   */
  public val url: String? = null,
  /**
   * If set, the request method is overridden.
   */
  public val method: String? = null,
  /**
   * If set, overrides the post data in the request. (Encoded as a base64 string when passed over
   * JSON)
   */
  public val postData: String? = null,
  /**
   * If set, overrides the request headers. Note that the overrides do not
   * extend to subsequent redirect hops, if a redirect happens. Another override
   * may be applied to a different request produced by a redirect.
   */
  public val headers: List<HeaderEntry>? = null,
  /**
   * If set, overrides response interception behavior for this request.
   */
  @ExperimentalChromeApi
  public val interceptResponse: Boolean? = null,
) {
  public class Builder(
    /**
     * An id the client received in requestPaused event.
     */
    public val requestId: RequestId,
  ) {
    /**
     * If set, the request url will be modified in a way that's not observable by page.
     */
    public var url: String? = null

    /**
     * If set, the request method is overridden.
     */
    public var method: String? = null

    /**
     * If set, overrides the post data in the request. (Encoded as a base64 string when passed over
     * JSON)
     */
    public var postData: String? = null

    /**
     * If set, overrides the request headers. Note that the overrides do not
     * extend to subsequent redirect hops, if a redirect happens. Another override
     * may be applied to a different request produced by a redirect.
     */
    public var headers: List<HeaderEntry>? = null

    /**
     * If set, overrides response interception behavior for this request.
     */
    @ExperimentalChromeApi
    public var interceptResponse: Boolean? = null

    public fun build(): ContinueRequestRequest = ContinueRequestRequest(requestId, url, method,
        postData, headers, interceptResponse)
  }
}

/**
 * A dummy response object for the [FetchDomain.continueRequest] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ContinueRequestResponse

/**
 * Request object containing input parameters for the [FetchDomain.continueWithAuth] command.
 */
@Serializable
public data class ContinueWithAuthRequest(
  /**
   * An id the client received in authRequired event.
   */
  public val requestId: RequestId,
  /**
   * Response to  with an authChallenge.
   */
  public val authChallengeResponse: AuthChallengeResponse,
)

/**
 * A dummy response object for the [FetchDomain.continueWithAuth] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ContinueWithAuthResponse

/**
 * Request object containing input parameters for the [FetchDomain.continueResponse] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ContinueResponseRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * An HTTP response code. If absent, original response code will be used.
   */
  public val responseCode: Int? = null,
  /**
   * A textual representation of responseCode.
   * If absent, a standard phrase matching responseCode is used.
   */
  public val responsePhrase: String? = null,
  /**
   * Response headers. If absent, original response headers will be used.
   */
  public val responseHeaders: List<HeaderEntry>? = null,
  /**
   * Alternative way of specifying response headers as a \0-separated
   * series of name: value pairs. Prefer the above method unless you
   * need to represent some non-UTF8 values that can't be transmitted
   * over the protocol as text. (Encoded as a base64 string when passed over JSON)
   */
  public val binaryResponseHeaders: String? = null,
) {
  public class Builder(
    /**
     * An id the client received in requestPaused event.
     */
    public val requestId: RequestId,
  ) {
    /**
     * An HTTP response code. If absent, original response code will be used.
     */
    public var responseCode: Int? = null

    /**
     * A textual representation of responseCode.
     * If absent, a standard phrase matching responseCode is used.
     */
    public var responsePhrase: String? = null

    /**
     * Response headers. If absent, original response headers will be used.
     */
    public var responseHeaders: List<HeaderEntry>? = null

    /**
     * Alternative way of specifying response headers as a \0-separated
     * series of name: value pairs. Prefer the above method unless you
     * need to represent some non-UTF8 values that can't be transmitted
     * over the protocol as text. (Encoded as a base64 string when passed over JSON)
     */
    public var binaryResponseHeaders: String? = null

    public fun build(): ContinueResponseRequest = ContinueResponseRequest(requestId, responseCode,
        responsePhrase, responseHeaders, binaryResponseHeaders)
  }
}

/**
 * A dummy response object for the [FetchDomain.continueResponse] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ContinueResponseResponse

/**
 * Request object containing input parameters for the [FetchDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyRequest(
  /**
   * Identifier for the intercepted request to get body for.
   */
  public val requestId: RequestId,
)

/**
 * Response type for the [FetchDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyResponse(
  /**
   * Response body.
   */
  public val body: String,
  /**
   * True, if content was sent as base64.
   */
  public val base64Encoded: Boolean,
)

/**
 * Request object containing input parameters for the [FetchDomain.takeResponseBodyAsStream]
 * command.
 */
@Serializable
public data class TakeResponseBodyAsStreamRequest(
  public val requestId: RequestId,
)

/**
 * Response type for the [FetchDomain.takeResponseBodyAsStream] command.
 */
@Serializable
public data class TakeResponseBodyAsStreamResponse(
  public val stream: StreamHandle,
)

/**
 * A domain for letting clients substitute browser's network layer with client code.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch)
 */
public class FetchDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out FetchEvent>> =
      mapOf(
      "Fetch.requestPaused" to serializer<FetchEvent.RequestPaused>(),
      "Fetch.authRequired" to serializer<FetchEvent.AuthRequired>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<FetchEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Issued when the domain is enabled and the request URL matches the
   * specified filter. The request is paused until the client responds
   * with one of continueRequest, failRequest or fulfillRequest.
   * The stage of the request can be determined by presence of responseErrorReason
   * and responseStatusCode -- the request is at the response stage if either
   * of these fields is present and in the request stage otherwise.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#event-requestPaused)
   */
  public fun requestPausedEvents(): Flow<FetchEvent.RequestPaused> =
      session.typedEvents("Fetch.requestPaused")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("requestPausedEvents()"),
  )
  public fun requestPaused(): Flow<FetchEvent.RequestPaused> = requestPausedEvents()

  /**
   * Issued when the domain is enabled with handleAuthRequests set to true.
   * The request is paused until client responds with continueWithAuth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#event-authRequired)
   */
  public fun authRequiredEvents(): Flow<FetchEvent.AuthRequired> =
      session.typedEvents("Fetch.authRequired")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("authRequiredEvents()"),
  )
  public fun authRequired(): Flow<FetchEvent.AuthRequired> = authRequiredEvents()

  /**
   * Disables the fetch domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Fetch.disable", Unit)

  /**
   * Enables issuing of requestPaused events. A request will be paused until client
   * calls one of failRequest, fulfillRequest or continueRequest/continueWithAuth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse = session.request("Fetch.enable",
      input)

  /**
   * Enables issuing of requestPaused events. A request will be paused until client
   * calls one of failRequest, fulfillRequest or continueRequest/continueWithAuth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Causes the request to fail with specified reason.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-failRequest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun failRequest(input: FailRequestRequest): FailRequestResponse =
      session.request("Fetch.failRequest", input)

  /**
   * Causes the request to fail with specified reason.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-failRequest)
   *
   * @param requestId An id the client received in requestPaused event.
   * @param errorReason Causes the request to fail with the given reason.
   */
  public suspend fun failRequest(requestId: RequestId, errorReason: ErrorReason):
      FailRequestResponse {
    val input = FailRequestRequest(requestId, errorReason)
    return failRequest(input)
  }

  /**
   * Provides response to the request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-fulfillRequest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun fulfillRequest(input: FulfillRequestRequest): FulfillRequestResponse =
      session.request("Fetch.fulfillRequest", input)

  /**
   * Provides response to the request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-fulfillRequest)
   *
   * @param requestId An id the client received in requestPaused event.
   * @param responseCode An HTTP response code.
   */
  @JvmOverloads
  public suspend inline fun fulfillRequest(
    requestId: RequestId,
    responseCode: Int,
    optionalArgs: FulfillRequestRequest.Builder.() -> Unit = {},
  ): FulfillRequestResponse {
    val builder = FulfillRequestRequest.Builder(requestId, responseCode)
    val input = builder.apply(optionalArgs).build()
    return fulfillRequest(input)
  }

  /**
   * Continues the request, optionally modifying some of its parameters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueRequest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun continueRequest(input: ContinueRequestRequest): ContinueRequestResponse =
      session.request("Fetch.continueRequest", input)

  /**
   * Continues the request, optionally modifying some of its parameters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueRequest)
   *
   * @param requestId An id the client received in requestPaused event.
   */
  @JvmOverloads
  public suspend inline fun continueRequest(requestId: RequestId,
      optionalArgs: ContinueRequestRequest.Builder.() -> Unit = {}): ContinueRequestResponse {
    val builder = ContinueRequestRequest.Builder(requestId)
    val input = builder.apply(optionalArgs).build()
    return continueRequest(input)
  }

  /**
   * Continues a request supplying authChallengeResponse following authRequired event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueWithAuth)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun continueWithAuth(input: ContinueWithAuthRequest): ContinueWithAuthResponse =
      session.request("Fetch.continueWithAuth", input)

  /**
   * Continues a request supplying authChallengeResponse following authRequired event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueWithAuth)
   *
   * @param requestId An id the client received in authRequired event.
   * @param authChallengeResponse Response to  with an authChallenge.
   */
  public suspend fun continueWithAuth(requestId: RequestId,
      authChallengeResponse: AuthChallengeResponse): ContinueWithAuthResponse {
    val input = ContinueWithAuthRequest(requestId, authChallengeResponse)
    return continueWithAuth(input)
  }

  /**
   * Continues loading of the paused response, optionally modifying the
   * response headers. If either responseCode or headers are modified, all of them
   * must be present.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueResponse)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun continueResponse(input: ContinueResponseRequest): ContinueResponseResponse =
      session.request("Fetch.continueResponse", input)

  /**
   * Continues loading of the paused response, optionally modifying the
   * response headers. If either responseCode or headers are modified, all of them
   * must be present.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueResponse)
   *
   * @param requestId An id the client received in requestPaused event.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun continueResponse(requestId: RequestId,
      optionalArgs: ContinueResponseRequest.Builder.() -> Unit = {}): ContinueResponseResponse {
    val builder = ContinueResponseRequest.Builder(requestId)
    val input = builder.apply(optionalArgs).build()
    return continueResponse(input)
  }

  /**
   * Causes the body of the response to be received from the server and
   * returned as a single string. May only be issued for a request that
   * is paused in the Response stage and is mutually exclusive with
   * takeResponseBodyForInterceptionAsStream. Calling other methods that
   * affect the request or disabling fetch domain before body is received
   * results in an undefined behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-getResponseBody)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getResponseBody(input: GetResponseBodyRequest): GetResponseBodyResponse =
      session.request("Fetch.getResponseBody", input)

  /**
   * Causes the body of the response to be received from the server and
   * returned as a single string. May only be issued for a request that
   * is paused in the Response stage and is mutually exclusive with
   * takeResponseBodyForInterceptionAsStream. Calling other methods that
   * affect the request or disabling fetch domain before body is received
   * results in an undefined behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-getResponseBody)
   *
   * @param requestId Identifier for the intercepted request to get body for.
   */
  public suspend fun getResponseBody(requestId: RequestId): GetResponseBodyResponse {
    val input = GetResponseBodyRequest(requestId)
    return getResponseBody(input)
  }

  /**
   * Returns a handle to the stream representing the response body.
   * The request must be paused in the HeadersReceived stage.
   * Note that after this command the request can't be continued
   * as is -- client either needs to cancel it or to provide the
   * response body.
   * The stream only supports sequential read, IO.read will fail if the position
   * is specified.
   * This method is mutually exclusive with getResponseBody.
   * Calling other methods that affect the request or disabling fetch
   * domain before body is received results in an undefined behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-takeResponseBodyAsStream)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun takeResponseBodyAsStream(input: TakeResponseBodyAsStreamRequest):
      TakeResponseBodyAsStreamResponse = session.request("Fetch.takeResponseBodyAsStream", input)

  /**
   * Returns a handle to the stream representing the response body.
   * The request must be paused in the HeadersReceived stage.
   * Note that after this command the request can't be continued
   * as is -- client either needs to cancel it or to provide the
   * response body.
   * The stream only supports sequential read, IO.read will fail if the position
   * is specified.
   * This method is mutually exclusive with getResponseBody.
   * Calling other methods that affect the request or disabling fetch
   * domain before body is received results in an undefined behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-takeResponseBodyAsStream)
   */
  public suspend fun takeResponseBodyAsStream(requestId: RequestId):
      TakeResponseBodyAsStreamResponse {
    val input = TakeResponseBodyAsStreamRequest(requestId)
    return takeResponseBodyAsStream(input)
  }
}
