@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.headlessexperimental

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the [HeadlessExperimentalDomain.beginFrame]
 * command.
 */
@Serializable
public data class BeginFrameRequest(
  /**
   * Timestamp of this BeginFrame in Renderer TimeTicks (milliseconds of uptime). If not set,
   * the current time will be used.
   */
  public val frameTimeTicks: Double? = null,
  /**
   * The interval between BeginFrames that is reported to the compositor, in milliseconds.
   * Defaults to a 60 frames/second interval, i.e. about 16.666 milliseconds.
   */
  public val interval: Double? = null,
  /**
   * Whether updates should not be committed and drawn onto the display. False by default. If
   * true, only side effects of the BeginFrame will be run, such as layout and animations, but
   * any visual updates may not be visible on the display or in screenshots.
   */
  public val noDisplayUpdates: Boolean? = null,
  /**
   * If set, a screenshot of the frame will be captured and returned in the response. Otherwise,
   * no screenshot will be captured. Note that capturing a screenshot can fail, for example,
   * during renderer initialization. In such a case, no screenshot data will be returned.
   */
  public val screenshot: ScreenshotParams? = null,
) {
  public class Builder() {
    /**
     * Timestamp of this BeginFrame in Renderer TimeTicks (milliseconds of uptime). If not set,
     * the current time will be used.
     */
    public var frameTimeTicks: Double? = null

    /**
     * The interval between BeginFrames that is reported to the compositor, in milliseconds.
     * Defaults to a 60 frames/second interval, i.e. about 16.666 milliseconds.
     */
    public var interval: Double? = null

    /**
     * Whether updates should not be committed and drawn onto the display. False by default. If
     * true, only side effects of the BeginFrame will be run, such as layout and animations, but
     * any visual updates may not be visible on the display or in screenshots.
     */
    public var noDisplayUpdates: Boolean? = null

    /**
     * If set, a screenshot of the frame will be captured and returned in the response. Otherwise,
     * no screenshot will be captured. Note that capturing a screenshot can fail, for example,
     * during renderer initialization. In such a case, no screenshot data will be returned.
     */
    public var screenshot: ScreenshotParams? = null

    public fun build(): BeginFrameRequest = BeginFrameRequest(frameTimeTicks, interval,
        noDisplayUpdates, screenshot)
  }
}

/**
 * Response type for the [HeadlessExperimentalDomain.beginFrame] command.
 */
@Serializable
public data class BeginFrameResponse(
  /**
   * Whether the BeginFrame resulted in damage and, thus, a new frame was committed to the
   * display. Reported for diagnostic uses, may be removed in the future.
   */
  public val hasDamage: Boolean,
  /**
   * Base64-encoded image data of the screenshot, if one was requested and successfully taken.
   * (Encoded as a base64 string when passed over JSON)
   */
  public val screenshotData: String? = null,
)

/**
 * A dummy response object for the [HeadlessExperimentalDomain.disable] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object DisableResponse

/**
 * A dummy response object for the [HeadlessExperimentalDomain.enable] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object EnableResponse

/**
 * This domain provides experimental commands only supported in headless mode.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental)
 */
@ExperimentalChromeApi
public class HeadlessExperimentalDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Sends a BeginFrame to the target and returns when the frame was completed. Optionally captures
   * a
   * screenshot from the resulting frame. Requires that the target was created with enabled
   * BeginFrameControl. Designed for use with --run-all-compositor-stages-before-draw, see also
   * https://goo.gle/chrome-headless-rendering for more background.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-beginFrame)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun beginFrame(input: BeginFrameRequest): BeginFrameResponse =
      session.request("HeadlessExperimental.beginFrame", input)

  /**
   * Sends a BeginFrame to the target and returns when the frame was completed. Optionally captures
   * a
   * screenshot from the resulting frame. Requires that the target was created with enabled
   * BeginFrameControl. Designed for use with --run-all-compositor-stages-before-draw, see also
   * https://goo.gle/chrome-headless-rendering for more background.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-beginFrame)
   */
  @JvmOverloads
  public suspend inline fun beginFrame(optionalArgs: BeginFrameRequest.Builder.() -> Unit = {}):
      BeginFrameResponse {
    val builder = BeginFrameRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return beginFrame(input)
  }

  /**
   * Disables headless events for the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-disable)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun disable(): DisableResponse = session.request("HeadlessExperimental.disable",
      Unit)

  /**
   * Enables headless events for the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-enable)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun enable(): EnableResponse = session.request("HeadlessExperimental.enable", Unit)
}
