@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.heapprofiler

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.heapprofiler.events.HeapProfilerEvent
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [HeapProfilerDomain.addInspectedHeapObject]
 * command.
 */
@Serializable
public data class AddInspectedHeapObjectRequest(
  /**
   * Heap snapshot object id to be accessible by means of $x command line API.
   */
  public val heapObjectId: HeapSnapshotObjectId,
)

/**
 * A dummy response object for the [HeapProfilerDomain.addInspectedHeapObject] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object AddInspectedHeapObjectResponse

/**
 * A dummy response object for the [HeapProfilerDomain.collectGarbage] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object CollectGarbageResponse

/**
 * A dummy response object for the [HeapProfilerDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [HeapProfilerDomain.enable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [HeapProfilerDomain.getHeapObjectId] command.
 */
@Serializable
public data class GetHeapObjectIdRequest(
  /**
   * Identifier of the object to get heap object id for.
   */
  public val objectId: RemoteObjectId,
)

/**
 * Response type for the [HeapProfilerDomain.getHeapObjectId] command.
 */
@Serializable
public data class GetHeapObjectIdResponse(
  /**
   * Id of the heap snapshot object corresponding to the passed remote object id.
   */
  public val heapSnapshotObjectId: HeapSnapshotObjectId,
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.getObjectByHeapObjectId]
 * command.
 */
@Serializable
public data class GetObjectByHeapObjectIdRequest(
  public val objectId: HeapSnapshotObjectId,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
) {
  public class Builder(
    public val objectId: HeapSnapshotObjectId,
  ) {
    /**
     * Symbolic group name that can be used to release multiple objects.
     */
    public var objectGroup: String? = null

    public fun build(): GetObjectByHeapObjectIdRequest = GetObjectByHeapObjectIdRequest(objectId,
        objectGroup)
  }
}

/**
 * Response type for the [HeapProfilerDomain.getObjectByHeapObjectId] command.
 */
@Serializable
public data class GetObjectByHeapObjectIdResponse(
  /**
   * Evaluation result.
   */
  public val result: RemoteObject,
)

/**
 * Response type for the [HeapProfilerDomain.getSamplingProfile] command.
 */
@Serializable
public data class GetSamplingProfileResponse(
  /**
   * Return the sampling profile being collected.
   */
  public val profile: SamplingHeapProfile,
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.startSampling] command.
 */
@Serializable
public data class StartSamplingRequest(
  /**
   * Average sample interval in bytes. Poisson distribution is used for the intervals. The
   * default value is 32768 bytes.
   */
  public val samplingInterval: Double? = null,
  /**
   * By default, the sampling heap profiler reports only objects which are
   * still alive when the profile is returned via getSamplingProfile or
   * stopSampling, which is useful for determining what functions contribute
   * the most to steady-state memory usage. This flag instructs the sampling
   * heap profiler to also include information about objects discarded by
   * major GC, which will show which functions cause large temporary memory
   * usage or long GC pauses.
   */
  public val includeObjectsCollectedByMajorGC: Boolean? = null,
  /**
   * By default, the sampling heap profiler reports only objects which are
   * still alive when the profile is returned via getSamplingProfile or
   * stopSampling, which is useful for determining what functions contribute
   * the most to steady-state memory usage. This flag instructs the sampling
   * heap profiler to also include information about objects discarded by
   * minor GC, which is useful when tuning a latency-sensitive application
   * for minimal GC activity.
   */
  public val includeObjectsCollectedByMinorGC: Boolean? = null,
) {
  public class Builder() {
    /**
     * Average sample interval in bytes. Poisson distribution is used for the intervals. The
     * default value is 32768 bytes.
     */
    public var samplingInterval: Double? = null

    /**
     * By default, the sampling heap profiler reports only objects which are
     * still alive when the profile is returned via getSamplingProfile or
     * stopSampling, which is useful for determining what functions contribute
     * the most to steady-state memory usage. This flag instructs the sampling
     * heap profiler to also include information about objects discarded by
     * major GC, which will show which functions cause large temporary memory
     * usage or long GC pauses.
     */
    public var includeObjectsCollectedByMajorGC: Boolean? = null

    /**
     * By default, the sampling heap profiler reports only objects which are
     * still alive when the profile is returned via getSamplingProfile or
     * stopSampling, which is useful for determining what functions contribute
     * the most to steady-state memory usage. This flag instructs the sampling
     * heap profiler to also include information about objects discarded by
     * minor GC, which is useful when tuning a latency-sensitive application
     * for minimal GC activity.
     */
    public var includeObjectsCollectedByMinorGC: Boolean? = null

    public fun build(): StartSamplingRequest = StartSamplingRequest(samplingInterval,
        includeObjectsCollectedByMajorGC, includeObjectsCollectedByMinorGC)
  }
}

/**
 * A dummy response object for the [HeapProfilerDomain.startSampling] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object StartSamplingResponse

/**
 * Request object containing input parameters for the [HeapProfilerDomain.startTrackingHeapObjects]
 * command.
 */
@Serializable
public data class StartTrackingHeapObjectsRequest(
  public val trackAllocations: Boolean? = null,
) {
  public class Builder() {
    public var trackAllocations: Boolean? = null

    public fun build(): StartTrackingHeapObjectsRequest =
        StartTrackingHeapObjectsRequest(trackAllocations)
  }
}

/**
 * A dummy response object for the [HeapProfilerDomain.startTrackingHeapObjects] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object StartTrackingHeapObjectsResponse

/**
 * Response type for the [HeapProfilerDomain.stopSampling] command.
 */
@Serializable
public data class StopSamplingResponse(
  /**
   * Recorded sampling heap profile.
   */
  public val profile: SamplingHeapProfile,
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.stopTrackingHeapObjects]
 * command.
 */
@Serializable
public data class StopTrackingHeapObjectsRequest(
  /**
   * If true 'reportHeapSnapshotProgress' events will be generated while snapshot is being taken
   * when the tracking is stopped.
   */
  public val reportProgress: Boolean? = null,
  /**
   * Deprecated in favor of `exposeInternals`.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val treatGlobalObjectsAsRoots: Boolean? = null,
  /**
   * If true, numerical values are included in the snapshot
   */
  public val captureNumericValue: Boolean? = null,
  /**
   * If true, exposes internals of the snapshot.
   */
  @ExperimentalChromeApi
  public val exposeInternals: Boolean? = null,
) {
  public class Builder() {
    /**
     * If true 'reportHeapSnapshotProgress' events will be generated while snapshot is being taken
     * when the tracking is stopped.
     */
    public var reportProgress: Boolean? = null

    /**
     * Deprecated in favor of `exposeInternals`.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var treatGlobalObjectsAsRoots: Boolean? = null

    /**
     * If true, numerical values are included in the snapshot
     */
    public var captureNumericValue: Boolean? = null

    /**
     * If true, exposes internals of the snapshot.
     */
    @ExperimentalChromeApi
    public var exposeInternals: Boolean? = null

    public fun build(): StopTrackingHeapObjectsRequest =
        StopTrackingHeapObjectsRequest(reportProgress, treatGlobalObjectsAsRoots,
        captureNumericValue, exposeInternals)
  }
}

/**
 * A dummy response object for the [HeapProfilerDomain.stopTrackingHeapObjects] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object StopTrackingHeapObjectsResponse

/**
 * Request object containing input parameters for the [HeapProfilerDomain.takeHeapSnapshot] command.
 */
@Serializable
public data class TakeHeapSnapshotRequest(
  /**
   * If true 'reportHeapSnapshotProgress' events will be generated while snapshot is being taken.
   */
  public val reportProgress: Boolean? = null,
  /**
   * If true, a raw snapshot without artificial roots will be generated.
   * Deprecated in favor of `exposeInternals`.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val treatGlobalObjectsAsRoots: Boolean? = null,
  /**
   * If true, numerical values are included in the snapshot
   */
  public val captureNumericValue: Boolean? = null,
  /**
   * If true, exposes internals of the snapshot.
   */
  @ExperimentalChromeApi
  public val exposeInternals: Boolean? = null,
) {
  public class Builder() {
    /**
     * If true 'reportHeapSnapshotProgress' events will be generated while snapshot is being taken.
     */
    public var reportProgress: Boolean? = null

    /**
     * If true, a raw snapshot without artificial roots will be generated.
     * Deprecated in favor of `exposeInternals`.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var treatGlobalObjectsAsRoots: Boolean? = null

    /**
     * If true, numerical values are included in the snapshot
     */
    public var captureNumericValue: Boolean? = null

    /**
     * If true, exposes internals of the snapshot.
     */
    @ExperimentalChromeApi
    public var exposeInternals: Boolean? = null

    public fun build(): TakeHeapSnapshotRequest = TakeHeapSnapshotRequest(reportProgress,
        treatGlobalObjectsAsRoots, captureNumericValue, exposeInternals)
  }
}

/**
 * A dummy response object for the [HeapProfilerDomain.takeHeapSnapshot] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object TakeHeapSnapshotResponse

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler)
 */
@ExperimentalChromeApi
public class HeapProfilerDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out HeapProfilerEvent>>
      = mapOf(
      "HeapProfiler.addHeapSnapshotChunk" to serializer<HeapProfilerEvent.AddHeapSnapshotChunk>(),
      "HeapProfiler.heapStatsUpdate" to serializer<HeapProfilerEvent.HeapStatsUpdate>(),
      "HeapProfiler.lastSeenObjectId" to serializer<HeapProfilerEvent.LastSeenObjectId>(),
      "HeapProfiler.reportHeapSnapshotProgress" to
          serializer<HeapProfilerEvent.ReportHeapSnapshotProgress>(),
      "HeapProfiler.resetProfiles" to serializer<HeapProfilerEvent.ResetProfiles>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<HeapProfilerEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#event-addHeapSnapshotChunk)
   */
  public fun addHeapSnapshotChunkEvents(): Flow<HeapProfilerEvent.AddHeapSnapshotChunk> =
      session.typedEvents("HeapProfiler.addHeapSnapshotChunk")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("addHeapSnapshotChunkEvents()"),
  )
  public fun addHeapSnapshotChunk(): Flow<HeapProfilerEvent.AddHeapSnapshotChunk> =
      addHeapSnapshotChunkEvents()

  /**
   * If heap objects tracking has been started then backend may send update for one or more
   * fragments
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#event-heapStatsUpdate)
   */
  public fun heapStatsUpdateEvents(): Flow<HeapProfilerEvent.HeapStatsUpdate> =
      session.typedEvents("HeapProfiler.heapStatsUpdate")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("heapStatsUpdateEvents()"),
  )
  public fun heapStatsUpdate(): Flow<HeapProfilerEvent.HeapStatsUpdate> = heapStatsUpdateEvents()

  /**
   * If heap objects tracking has been started then backend regularly sends a current value for last
   * seen object id and corresponding timestamp. If the were changes in the heap since last event
   * then one or more heapStatsUpdate events will be sent before a new lastSeenObjectId event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#event-lastSeenObjectId)
   */
  public fun lastSeenObjectIdEvents(): Flow<HeapProfilerEvent.LastSeenObjectId> =
      session.typedEvents("HeapProfiler.lastSeenObjectId")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("lastSeenObjectIdEvents()"),
  )
  public fun lastSeenObjectId(): Flow<HeapProfilerEvent.LastSeenObjectId> = lastSeenObjectIdEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#event-reportHeapSnapshotProgress)
   */
  public fun reportHeapSnapshotProgressEvents(): Flow<HeapProfilerEvent.ReportHeapSnapshotProgress>
      = session.typedEvents("HeapProfiler.reportHeapSnapshotProgress")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("reportHeapSnapshotProgressEvents()"),
  )
  public fun reportHeapSnapshotProgress(): Flow<HeapProfilerEvent.ReportHeapSnapshotProgress> =
      reportHeapSnapshotProgressEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#event-resetProfiles)
   */
  public fun resetProfilesEvents(): Flow<HeapProfilerEvent.ResetProfiles> =
      session.typedEvents("HeapProfiler.resetProfiles")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("resetProfilesEvents()"),
  )
  public fun resetProfiles(): Flow<HeapProfilerEvent.ResetProfiles> = resetProfilesEvents()

  /**
   * Enables console to refer to the node with given id via $x (see Command Line API for more
   * details
   * $x functions).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-addInspectedHeapObject)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun addInspectedHeapObject(input: AddInspectedHeapObjectRequest):
      AddInspectedHeapObjectResponse = session.request("HeapProfiler.addInspectedHeapObject", input)

  /**
   * Enables console to refer to the node with given id via $x (see Command Line API for more
   * details
   * $x functions).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-addInspectedHeapObject)
   *
   * @param heapObjectId Heap snapshot object id to be accessible by means of $x command line API.
   */
  public suspend fun addInspectedHeapObject(heapObjectId: HeapSnapshotObjectId):
      AddInspectedHeapObjectResponse {
    val input = AddInspectedHeapObjectRequest(heapObjectId)
    return addInspectedHeapObject(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-collectGarbage)
   */
  public suspend fun collectGarbage(): CollectGarbageResponse =
      session.request("HeapProfiler.collectGarbage", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("HeapProfiler.disable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("HeapProfiler.enable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-getHeapObjectId)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getHeapObjectId(input: GetHeapObjectIdRequest): GetHeapObjectIdResponse =
      session.request("HeapProfiler.getHeapObjectId", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-getHeapObjectId)
   *
   * @param objectId Identifier of the object to get heap object id for.
   */
  public suspend fun getHeapObjectId(objectId: RemoteObjectId): GetHeapObjectIdResponse {
    val input = GetHeapObjectIdRequest(objectId)
    return getHeapObjectId(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-getObjectByHeapObjectId)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getObjectByHeapObjectId(input: GetObjectByHeapObjectIdRequest):
      GetObjectByHeapObjectIdResponse = session.request("HeapProfiler.getObjectByHeapObjectId",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-getObjectByHeapObjectId)
   */
  @JvmOverloads
  public suspend inline fun getObjectByHeapObjectId(objectId: HeapSnapshotObjectId,
      optionalArgs: GetObjectByHeapObjectIdRequest.Builder.() -> Unit = {}):
      GetObjectByHeapObjectIdResponse {
    val builder = GetObjectByHeapObjectIdRequest.Builder(objectId)
    val input = builder.apply(optionalArgs).build()
    return getObjectByHeapObjectId(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-getSamplingProfile)
   */
  public suspend fun getSamplingProfile(): GetSamplingProfileResponse =
      session.request("HeapProfiler.getSamplingProfile", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-startSampling)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startSampling(input: StartSamplingRequest): StartSamplingResponse =
      session.request("HeapProfiler.startSampling", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-startSampling)
   */
  @JvmOverloads
  public suspend inline fun startSampling(optionalArgs: StartSamplingRequest.Builder.() -> Unit =
      {}): StartSamplingResponse {
    val builder = StartSamplingRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return startSampling(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-startTrackingHeapObjects)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startTrackingHeapObjects(input: StartTrackingHeapObjectsRequest):
      StartTrackingHeapObjectsResponse = session.request("HeapProfiler.startTrackingHeapObjects",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-startTrackingHeapObjects)
   */
  @JvmOverloads
  public suspend inline
      fun startTrackingHeapObjects(optionalArgs: StartTrackingHeapObjectsRequest.Builder.() -> Unit
      = {}): StartTrackingHeapObjectsResponse {
    val builder = StartTrackingHeapObjectsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return startTrackingHeapObjects(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-stopSampling)
   */
  public suspend fun stopSampling(): StopSamplingResponse =
      session.request("HeapProfiler.stopSampling", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-stopTrackingHeapObjects)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun stopTrackingHeapObjects(input: StopTrackingHeapObjectsRequest):
      StopTrackingHeapObjectsResponse = session.request("HeapProfiler.stopTrackingHeapObjects",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-stopTrackingHeapObjects)
   */
  @JvmOverloads
  public suspend inline
      fun stopTrackingHeapObjects(optionalArgs: StopTrackingHeapObjectsRequest.Builder.() -> Unit =
      {}): StopTrackingHeapObjectsResponse {
    val builder = StopTrackingHeapObjectsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return stopTrackingHeapObjects(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-takeHeapSnapshot)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun takeHeapSnapshot(input: TakeHeapSnapshotRequest): TakeHeapSnapshotResponse =
      session.request("HeapProfiler.takeHeapSnapshot", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeapProfiler/#method-takeHeapSnapshot)
   */
  @JvmOverloads
  public suspend inline
      fun takeHeapSnapshot(optionalArgs: TakeHeapSnapshotRequest.Builder.() -> Unit = {}):
      TakeHeapSnapshotResponse {
    val builder = TakeHeapSnapshotRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return takeHeapSnapshot(input)
  }
}
