@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.indexeddb

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the [IndexedDBDomain.clearObjectStore] command.
 */
@Serializable
public data class ClearObjectStoreRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
  /**
   * Database name.
   */
  public val databaseName: String,
  /**
   * Object store name.
   */
  public val objectStoreName: String,
) {
  public class Builder(
    /**
     * Database name.
     */
    public val databaseName: String,
    /**
     * Object store name.
     */
    public val objectStoreName: String,
  ) {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    public fun build(): ClearObjectStoreRequest = ClearObjectStoreRequest(securityOrigin,
        storageKey, databaseName, objectStoreName)
  }
}

/**
 * A dummy response object for the [IndexedDBDomain.clearObjectStore] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ClearObjectStoreResponse

/**
 * Request object containing input parameters for the [IndexedDBDomain.deleteDatabase] command.
 */
@Serializable
public data class DeleteDatabaseRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
  /**
   * Database name.
   */
  public val databaseName: String,
) {
  public class Builder(
    /**
     * Database name.
     */
    public val databaseName: String,
  ) {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    public fun build(): DeleteDatabaseRequest = DeleteDatabaseRequest(securityOrigin, storageKey,
        databaseName)
  }
}

/**
 * A dummy response object for the [IndexedDBDomain.deleteDatabase] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DeleteDatabaseResponse

/**
 * Request object containing input parameters for the [IndexedDBDomain.deleteObjectStoreEntries]
 * command.
 */
@Serializable
public data class DeleteObjectStoreEntriesRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
  public val databaseName: String,
  public val objectStoreName: String,
  /**
   * Range of entry keys to delete
   */
  public val keyRange: KeyRange,
) {
  public class Builder(
    public val databaseName: String,
    public val objectStoreName: String,
    /**
     * Range of entry keys to delete
     */
    public val keyRange: KeyRange,
  ) {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    public fun build(): DeleteObjectStoreEntriesRequest =
        DeleteObjectStoreEntriesRequest(securityOrigin, storageKey, databaseName, objectStoreName,
        keyRange)
  }
}

/**
 * A dummy response object for the [IndexedDBDomain.deleteObjectStoreEntries] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object DeleteObjectStoreEntriesResponse

/**
 * A dummy response object for the [IndexedDBDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [IndexedDBDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [IndexedDBDomain.requestData] command.
 */
@Serializable
public data class RequestDataRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
  /**
   * Database name.
   */
  public val databaseName: String,
  /**
   * Object store name.
   */
  public val objectStoreName: String,
  /**
   * Index name, empty string for object store data requests.
   */
  public val indexName: String,
  /**
   * Number of records to skip.
   */
  public val skipCount: Int,
  /**
   * Number of records to fetch.
   */
  public val pageSize: Int,
  /**
   * Key range.
   */
  public val keyRange: KeyRange? = null,
) {
  public class Builder(
    /**
     * Database name.
     */
    public val databaseName: String,
    /**
     * Object store name.
     */
    public val objectStoreName: String,
    /**
     * Index name, empty string for object store data requests.
     */
    public val indexName: String,
    /**
     * Number of records to skip.
     */
    public val skipCount: Int,
    /**
     * Number of records to fetch.
     */
    public val pageSize: Int,
  ) {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    /**
     * Key range.
     */
    public var keyRange: KeyRange? = null

    public fun build(): RequestDataRequest = RequestDataRequest(securityOrigin, storageKey,
        databaseName, objectStoreName, indexName, skipCount, pageSize, keyRange)
  }
}

/**
 * Response type for the [IndexedDBDomain.requestData] command.
 */
@Serializable
public data class RequestDataResponse(
  /**
   * Array of object store data entries.
   */
  public val objectStoreDataEntries: List<DataEntry>,
  /**
   * If true, there are more entries to fetch in the given range.
   */
  public val hasMore: Boolean,
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.getMetadata] command.
 */
@Serializable
public data class GetMetadataRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
  /**
   * Database name.
   */
  public val databaseName: String,
  /**
   * Object store name.
   */
  public val objectStoreName: String,
) {
  public class Builder(
    /**
     * Database name.
     */
    public val databaseName: String,
    /**
     * Object store name.
     */
    public val objectStoreName: String,
  ) {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    public fun build(): GetMetadataRequest = GetMetadataRequest(securityOrigin, storageKey,
        databaseName, objectStoreName)
  }
}

/**
 * Response type for the [IndexedDBDomain.getMetadata] command.
 */
@Serializable
public data class GetMetadataResponse(
  /**
   * the entries count
   */
  public val entriesCount: Double,
  /**
   * the current value of key generator, to become the next inserted
   * key into the object store. Valid if objectStore.autoIncrement
   * is true.
   */
  public val keyGeneratorValue: Double,
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.requestDatabase] command.
 */
@Serializable
public data class RequestDatabaseRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
  /**
   * Database name.
   */
  public val databaseName: String,
) {
  public class Builder(
    /**
     * Database name.
     */
    public val databaseName: String,
  ) {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    public fun build(): RequestDatabaseRequest = RequestDatabaseRequest(securityOrigin, storageKey,
        databaseName)
  }
}

/**
 * Response type for the [IndexedDBDomain.requestDatabase] command.
 */
@Serializable
public data class RequestDatabaseResponse(
  /**
   * Database with an array of object stores.
   */
  public val databaseWithObjectStores: DatabaseWithObjectStores,
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.requestDatabaseNames]
 * command.
 */
@Serializable
public data class RequestDatabaseNamesRequest(
  /**
   * At least and at most one of securityOrigin, storageKey must be specified.
   * Security origin.
   */
  public val securityOrigin: String? = null,
  /**
   * Storage key.
   */
  public val storageKey: String? = null,
) {
  public class Builder() {
    /**
     * At least and at most one of securityOrigin, storageKey must be specified.
     * Security origin.
     */
    public var securityOrigin: String? = null

    /**
     * Storage key.
     */
    public var storageKey: String? = null

    public fun build(): RequestDatabaseNamesRequest = RequestDatabaseNamesRequest(securityOrigin,
        storageKey)
  }
}

/**
 * Response type for the [IndexedDBDomain.requestDatabaseNames] command.
 */
@Serializable
public data class RequestDatabaseNamesResponse(
  /**
   * Database names for origin.
   */
  public val databaseNames: List<String>,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB)
 */
@ExperimentalChromeApi
public class IndexedDBDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Clears all entries from an object store.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-clearObjectStore)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clearObjectStore(input: ClearObjectStoreRequest): ClearObjectStoreResponse =
      session.request("IndexedDB.clearObjectStore", input)

  /**
   * Clears all entries from an object store.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-clearObjectStore)
   *
   * @param databaseName Database name.
   * @param objectStoreName Object store name.
   */
  @JvmOverloads
  public suspend inline fun clearObjectStore(
    databaseName: String,
    objectStoreName: String,
    optionalArgs: ClearObjectStoreRequest.Builder.() -> Unit = {},
  ): ClearObjectStoreResponse {
    val builder = ClearObjectStoreRequest.Builder(databaseName, objectStoreName)
    val input = builder.apply(optionalArgs).build()
    return clearObjectStore(input)
  }

  /**
   * Deletes a database.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-deleteDatabase)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun deleteDatabase(input: DeleteDatabaseRequest): DeleteDatabaseResponse =
      session.request("IndexedDB.deleteDatabase", input)

  /**
   * Deletes a database.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-deleteDatabase)
   *
   * @param databaseName Database name.
   */
  @JvmOverloads
  public suspend inline fun deleteDatabase(databaseName: String,
      optionalArgs: DeleteDatabaseRequest.Builder.() -> Unit = {}): DeleteDatabaseResponse {
    val builder = DeleteDatabaseRequest.Builder(databaseName)
    val input = builder.apply(optionalArgs).build()
    return deleteDatabase(input)
  }

  /**
   * Delete a range of entries from an object store
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-deleteObjectStoreEntries)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun deleteObjectStoreEntries(input: DeleteObjectStoreEntriesRequest):
      DeleteObjectStoreEntriesResponse = session.request("IndexedDB.deleteObjectStoreEntries",
      input)

  /**
   * Delete a range of entries from an object store
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-deleteObjectStoreEntries)
   * @param keyRange Range of entry keys to delete
   */
  @JvmOverloads
  public suspend inline fun deleteObjectStoreEntries(
    databaseName: String,
    objectStoreName: String,
    keyRange: KeyRange,
    optionalArgs: DeleteObjectStoreEntriesRequest.Builder.() -> Unit = {},
  ): DeleteObjectStoreEntriesResponse {
    val builder = DeleteObjectStoreEntriesRequest.Builder(databaseName, objectStoreName, keyRange)
    val input = builder.apply(optionalArgs).build()
    return deleteObjectStoreEntries(input)
  }

  /**
   * Disables events from backend.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("IndexedDB.disable", Unit)

  /**
   * Enables events from backend.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("IndexedDB.enable", Unit)

  /**
   * Requests data from object store or index.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-requestData)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestData(input: RequestDataRequest): RequestDataResponse =
      session.request("IndexedDB.requestData", input)

  /**
   * Requests data from object store or index.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-requestData)
   *
   * @param databaseName Database name.
   * @param objectStoreName Object store name.
   * @param indexName Index name, empty string for object store data requests.
   * @param skipCount Number of records to skip.
   * @param pageSize Number of records to fetch.
   */
  @JvmOverloads
  public suspend inline fun requestData(
    databaseName: String,
    objectStoreName: String,
    indexName: String,
    skipCount: Int,
    pageSize: Int,
    optionalArgs: RequestDataRequest.Builder.() -> Unit = {},
  ): RequestDataResponse {
    val builder = RequestDataRequest.Builder(databaseName, objectStoreName, indexName, skipCount,
        pageSize)
    val input = builder.apply(optionalArgs).build()
    return requestData(input)
  }

  /**
   * Gets metadata of an object store
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-getMetadata)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getMetadata(input: GetMetadataRequest): GetMetadataResponse =
      session.request("IndexedDB.getMetadata", input)

  /**
   * Gets metadata of an object store
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-getMetadata)
   *
   * @param databaseName Database name.
   * @param objectStoreName Object store name.
   */
  @JvmOverloads
  public suspend inline fun getMetadata(
    databaseName: String,
    objectStoreName: String,
    optionalArgs: GetMetadataRequest.Builder.() -> Unit = {},
  ): GetMetadataResponse {
    val builder = GetMetadataRequest.Builder(databaseName, objectStoreName)
    val input = builder.apply(optionalArgs).build()
    return getMetadata(input)
  }

  /**
   * Requests database with given name in given frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-requestDatabase)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestDatabase(input: RequestDatabaseRequest): RequestDatabaseResponse =
      session.request("IndexedDB.requestDatabase", input)

  /**
   * Requests database with given name in given frame.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-requestDatabase)
   *
   * @param databaseName Database name.
   */
  @JvmOverloads
  public suspend inline fun requestDatabase(databaseName: String,
      optionalArgs: RequestDatabaseRequest.Builder.() -> Unit = {}): RequestDatabaseResponse {
    val builder = RequestDatabaseRequest.Builder(databaseName)
    val input = builder.apply(optionalArgs).build()
    return requestDatabase(input)
  }

  /**
   * Requests database names for given security origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-requestDatabaseNames)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestDatabaseNames(input: RequestDatabaseNamesRequest):
      RequestDatabaseNamesResponse = session.request("IndexedDB.requestDatabaseNames", input)

  /**
   * Requests database names for given security origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#method-requestDatabaseNames)
   */
  @JvmOverloads
  public suspend inline
      fun requestDatabaseNames(optionalArgs: RequestDatabaseNamesRequest.Builder.() -> Unit = {}):
      RequestDatabaseNamesResponse {
    val builder = RequestDatabaseNamesRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return requestDatabaseNames(input)
  }
}
