@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.input

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.input.events.InputEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [InputDomain.dispatchDragEvent] command.
 */
@Serializable
@ExperimentalChromeApi
public data class DispatchDragEventRequest(
  /**
   * Type of the drag event.
   */
  public val type: DragEventType,
  /**
   * X coordinate of the event relative to the main frame's viewport in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0 refers to
   * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
   */
  public val y: Double,
  public val `data`: DragData,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
) {
  public class Builder(
    /**
     * Type of the drag event.
     */
    public val type: DragEventType,
    /**
     * X coordinate of the event relative to the main frame's viewport in CSS pixels.
     */
    public val x: Double,
    /**
     * Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0 refers to
     * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
     */
    public val y: Double,
    public val `data`: DragData,
  ) {
    /**
     * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
     * (default: 0).
     */
    public var modifiers: Int? = null

    public fun build(): DispatchDragEventRequest = DispatchDragEventRequest(type, x, y, data,
        modifiers)
  }
}

/**
 * A dummy response object for the [InputDomain.dispatchDragEvent] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object DispatchDragEventResponse

/**
 * Request object containing input parameters for the [InputDomain.dispatchKeyEvent] command.
 */
@Serializable
public data class DispatchKeyEventRequest(
  /**
   * Type of the key event.
   */
  public val type: KeyEventType,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Time at which the event occurred.
   */
  public val timestamp: TimeSinceEpoch? = null,
  /**
   * Text as generated by processing a virtual key code with a keyboard layout. Not needed for
   * for `keyUp` and `rawKeyDown` events (default: "")
   */
  public val text: String? = null,
  /**
   * Text that would have been generated by the keyboard if no modifiers were pressed (except for
   * shift). Useful for shortcut (accelerator) key handling (default: "").
   */
  public val unmodifiedText: String? = null,
  /**
   * Unique key identifier (e.g., 'U+0041') (default: "").
   */
  public val keyIdentifier: String? = null,
  /**
   * Unique DOM defined string value for each physical key (e.g., 'KeyA') (default: "").
   */
  public val code: String? = null,
  /**
   * Unique DOM defined string value describing the meaning of the key in the context of active
   * modifiers, keyboard layout, etc (e.g., 'AltGr') (default: "").
   */
  public val key: String? = null,
  /**
   * Windows virtual key code (default: 0).
   */
  public val windowsVirtualKeyCode: Int? = null,
  /**
   * Native virtual key code (default: 0).
   */
  public val nativeVirtualKeyCode: Int? = null,
  /**
   * Whether the event was generated from auto repeat (default: false).
   */
  public val autoRepeat: Boolean? = null,
  /**
   * Whether the event was generated from the keypad (default: false).
   */
  public val isKeypad: Boolean? = null,
  /**
   * Whether the event was a system key event (default: false).
   */
  public val isSystemKey: Boolean? = null,
  /**
   * Whether the event was from the left or right side of the keyboard. 1=Left, 2=Right (default:
   * 0).
   */
  public val location: Int? = null,
  /**
   * Editing commands to send with the key event (e.g., 'selectAll') (default: []).
   * These are related to but not equal the command names used in `document.execCommand` and
   * NSStandardKeyBindingResponding.
   * See
   * https://source.chromium.org/chromium/chromium/src/+/main:third_party/blink/renderer/core/editing/commands/editor_command_names.h
   * for valid command names.
   */
  @ExperimentalChromeApi
  public val commands: List<String>? = null,
) {
  public class Builder(
    /**
     * Type of the key event.
     */
    public val type: KeyEventType,
  ) {
    /**
     * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
     * (default: 0).
     */
    public var modifiers: Int? = null

    /**
     * Time at which the event occurred.
     */
    public var timestamp: TimeSinceEpoch? = null

    /**
     * Text as generated by processing a virtual key code with a keyboard layout. Not needed for
     * for `keyUp` and `rawKeyDown` events (default: "")
     */
    public var text: String? = null

    /**
     * Text that would have been generated by the keyboard if no modifiers were pressed (except for
     * shift). Useful for shortcut (accelerator) key handling (default: "").
     */
    public var unmodifiedText: String? = null

    /**
     * Unique key identifier (e.g., 'U+0041') (default: "").
     */
    public var keyIdentifier: String? = null

    /**
     * Unique DOM defined string value for each physical key (e.g., 'KeyA') (default: "").
     */
    public var code: String? = null

    /**
     * Unique DOM defined string value describing the meaning of the key in the context of active
     * modifiers, keyboard layout, etc (e.g., 'AltGr') (default: "").
     */
    public var key: String? = null

    /**
     * Windows virtual key code (default: 0).
     */
    public var windowsVirtualKeyCode: Int? = null

    /**
     * Native virtual key code (default: 0).
     */
    public var nativeVirtualKeyCode: Int? = null

    /**
     * Whether the event was generated from auto repeat (default: false).
     */
    public var autoRepeat: Boolean? = null

    /**
     * Whether the event was generated from the keypad (default: false).
     */
    public var isKeypad: Boolean? = null

    /**
     * Whether the event was a system key event (default: false).
     */
    public var isSystemKey: Boolean? = null

    /**
     * Whether the event was from the left or right side of the keyboard. 1=Left, 2=Right (default:
     * 0).
     */
    public var location: Int? = null

    /**
     * Editing commands to send with the key event (e.g., 'selectAll') (default: []).
     * These are related to but not equal the command names used in `document.execCommand` and
     * NSStandardKeyBindingResponding.
     * See
     * https://source.chromium.org/chromium/chromium/src/+/main:third_party/blink/renderer/core/editing/commands/editor_command_names.h
     * for valid command names.
     */
    @ExperimentalChromeApi
    public var commands: List<String>? = null

    public fun build(): DispatchKeyEventRequest = DispatchKeyEventRequest(type, modifiers,
        timestamp, text, unmodifiedText, keyIdentifier, code, key, windowsVirtualKeyCode,
        nativeVirtualKeyCode, autoRepeat, isKeypad, isSystemKey, location, commands)
  }
}

/**
 * A dummy response object for the [InputDomain.dispatchKeyEvent] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DispatchKeyEventResponse

/**
 * Request object containing input parameters for the [InputDomain.insertText] command.
 */
@Serializable
@ExperimentalChromeApi
public data class InsertTextRequest(
  /**
   * The text to insert.
   */
  public val text: String,
)

/**
 * A dummy response object for the [InputDomain.insertText] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
@ExperimentalChromeApi
public object InsertTextResponse

/**
 * Request object containing input parameters for the [InputDomain.imeSetComposition] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ImeSetCompositionRequest(
  /**
   * The text to insert
   */
  public val text: String,
  /**
   * selection start
   */
  public val selectionStart: Int,
  /**
   * selection end
   */
  public val selectionEnd: Int,
  /**
   * replacement start
   */
  public val replacementStart: Int? = null,
  /**
   * replacement end
   */
  public val replacementEnd: Int? = null,
) {
  public class Builder(
    /**
     * The text to insert
     */
    public val text: String,
    /**
     * selection start
     */
    public val selectionStart: Int,
    /**
     * selection end
     */
    public val selectionEnd: Int,
  ) {
    /**
     * replacement start
     */
    public var replacementStart: Int? = null

    /**
     * replacement end
     */
    public var replacementEnd: Int? = null

    public fun build(): ImeSetCompositionRequest = ImeSetCompositionRequest(text, selectionStart,
        selectionEnd, replacementStart, replacementEnd)
  }
}

/**
 * A dummy response object for the [InputDomain.imeSetComposition] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ImeSetCompositionResponse

/**
 * Request object containing input parameters for the [InputDomain.dispatchMouseEvent] command.
 */
@Serializable
public data class DispatchMouseEventRequest(
  /**
   * Type of the mouse event.
   */
  public val type: MouseEventType,
  /**
   * X coordinate of the event relative to the main frame's viewport in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0 refers to
   * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
   */
  public val y: Double,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Time at which the event occurred.
   */
  public val timestamp: TimeSinceEpoch? = null,
  /**
   * Mouse button (default: "none").
   */
  public val button: MouseButton? = null,
  /**
   * A number indicating which buttons are pressed on the mouse when a mouse event is triggered.
   * Left=1, Right=2, Middle=4, Back=8, Forward=16, None=0.
   */
  public val buttons: Int? = null,
  /**
   * Number of times the mouse button was clicked (default: 0).
   */
  public val clickCount: Int? = null,
  /**
   * The normalized pressure, which has a range of [0,1] (default: 0).
   */
  @ExperimentalChromeApi
  public val force: Double? = null,
  /**
   * The normalized tangential pressure, which has a range of [-1,1] (default: 0).
   */
  @ExperimentalChromeApi
  public val tangentialPressure: Double? = null,
  /**
   * The plane angle between the Y-Z plane and the plane containing both the stylus axis and the Y
   * axis, in degrees of the range [-90,90], a positive tiltX is to the right (default: 0).
   */
  @ExperimentalChromeApi
  public val tiltX: Int? = null,
  /**
   * The plane angle between the X-Z plane and the plane containing both the stylus axis and the X
   * axis, in degrees of the range [-90,90], a positive tiltY is towards the user (default: 0).
   */
  @ExperimentalChromeApi
  public val tiltY: Int? = null,
  /**
   * The clockwise rotation of a pen stylus around its own major axis, in degrees in the range
   * [0,359] (default: 0).
   */
  @ExperimentalChromeApi
  public val twist: Int? = null,
  /**
   * X delta in CSS pixels for mouse wheel event (default: 0).
   */
  public val deltaX: Double? = null,
  /**
   * Y delta in CSS pixels for mouse wheel event (default: 0).
   */
  public val deltaY: Double? = null,
  /**
   * Pointer type (default: "mouse").
   */
  public val pointerType: MouseEventPointerType? = null,
) {
  public class Builder(
    /**
     * Type of the mouse event.
     */
    public val type: MouseEventType,
    /**
     * X coordinate of the event relative to the main frame's viewport in CSS pixels.
     */
    public val x: Double,
    /**
     * Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0 refers to
     * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
     */
    public val y: Double,
  ) {
    /**
     * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
     * (default: 0).
     */
    public var modifiers: Int? = null

    /**
     * Time at which the event occurred.
     */
    public var timestamp: TimeSinceEpoch? = null

    /**
     * Mouse button (default: "none").
     */
    public var button: MouseButton? = null

    /**
     * A number indicating which buttons are pressed on the mouse when a mouse event is triggered.
     * Left=1, Right=2, Middle=4, Back=8, Forward=16, None=0.
     */
    public var buttons: Int? = null

    /**
     * Number of times the mouse button was clicked (default: 0).
     */
    public var clickCount: Int? = null

    /**
     * The normalized pressure, which has a range of [0,1] (default: 0).
     */
    @ExperimentalChromeApi
    public var force: Double? = null

    /**
     * The normalized tangential pressure, which has a range of [-1,1] (default: 0).
     */
    @ExperimentalChromeApi
    public var tangentialPressure: Double? = null

    /**
     * The plane angle between the Y-Z plane and the plane containing both the stylus axis and the Y
     * axis, in degrees of the range [-90,90], a positive tiltX is to the right (default: 0).
     */
    @ExperimentalChromeApi
    public var tiltX: Int? = null

    /**
     * The plane angle between the X-Z plane and the plane containing both the stylus axis and the X
     * axis, in degrees of the range [-90,90], a positive tiltY is towards the user (default: 0).
     */
    @ExperimentalChromeApi
    public var tiltY: Int? = null

    /**
     * The clockwise rotation of a pen stylus around its own major axis, in degrees in the range
     * [0,359] (default: 0).
     */
    @ExperimentalChromeApi
    public var twist: Int? = null

    /**
     * X delta in CSS pixels for mouse wheel event (default: 0).
     */
    public var deltaX: Double? = null

    /**
     * Y delta in CSS pixels for mouse wheel event (default: 0).
     */
    public var deltaY: Double? = null

    /**
     * Pointer type (default: "mouse").
     */
    public var pointerType: MouseEventPointerType? = null

    public fun build(): DispatchMouseEventRequest = DispatchMouseEventRequest(type, x, y, modifiers,
        timestamp, button, buttons, clickCount, force, tangentialPressure, tiltX, tiltY, twist,
        deltaX, deltaY, pointerType)
  }
}

/**
 * A dummy response object for the [InputDomain.dispatchMouseEvent] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DispatchMouseEventResponse

/**
 * Request object containing input parameters for the [InputDomain.dispatchTouchEvent] command.
 */
@Serializable
public data class DispatchTouchEventRequest(
  /**
   * Type of the touch event. TouchEnd and TouchCancel must not contain any touch points, while
   * TouchStart and TouchMove must contains at least one.
   */
  public val type: TouchEventType,
  /**
   * Active touch points on the touch device. One event per any changed point (compared to
   * previous touch event in a sequence) is generated, emulating pressing/moving/releasing points
   * one by one.
   */
  public val touchPoints: List<TouchPoint>,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Time at which the event occurred.
   */
  public val timestamp: TimeSinceEpoch? = null,
) {
  public class Builder(
    /**
     * Type of the touch event. TouchEnd and TouchCancel must not contain any touch points, while
     * TouchStart and TouchMove must contains at least one.
     */
    public val type: TouchEventType,
    /**
     * Active touch points on the touch device. One event per any changed point (compared to
     * previous touch event in a sequence) is generated, emulating pressing/moving/releasing points
     * one by one.
     */
    public val touchPoints: List<TouchPoint>,
  ) {
    /**
     * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
     * (default: 0).
     */
    public var modifiers: Int? = null

    /**
     * Time at which the event occurred.
     */
    public var timestamp: TimeSinceEpoch? = null

    public fun build(): DispatchTouchEventRequest = DispatchTouchEventRequest(type, touchPoints,
        modifiers, timestamp)
  }
}

/**
 * A dummy response object for the [InputDomain.dispatchTouchEvent] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DispatchTouchEventResponse

/**
 * Request object containing input parameters for the [InputDomain.emulateTouchFromMouseEvent]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class EmulateTouchFromMouseEventRequest(
  /**
   * Type of the mouse event.
   */
  public val type: MouseEventType,
  /**
   * X coordinate of the mouse pointer in DIP.
   */
  public val x: Int,
  /**
   * Y coordinate of the mouse pointer in DIP.
   */
  public val y: Int,
  /**
   * Mouse button. Only "none", "left", "right" are supported.
   */
  public val button: MouseButton,
  /**
   * Time at which the event occurred (default: current time).
   */
  public val timestamp: TimeSinceEpoch? = null,
  /**
   * X delta in DIP for mouse wheel event (default: 0).
   */
  public val deltaX: Double? = null,
  /**
   * Y delta in DIP for mouse wheel event (default: 0).
   */
  public val deltaY: Double? = null,
  /**
   * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
   * (default: 0).
   */
  public val modifiers: Int? = null,
  /**
   * Number of times the mouse button was clicked (default: 0).
   */
  public val clickCount: Int? = null,
) {
  public class Builder(
    /**
     * Type of the mouse event.
     */
    public val type: MouseEventType,
    /**
     * X coordinate of the mouse pointer in DIP.
     */
    public val x: Int,
    /**
     * Y coordinate of the mouse pointer in DIP.
     */
    public val y: Int,
    /**
     * Mouse button. Only "none", "left", "right" are supported.
     */
    public val button: MouseButton,
  ) {
    /**
     * Time at which the event occurred (default: current time).
     */
    public var timestamp: TimeSinceEpoch? = null

    /**
     * X delta in DIP for mouse wheel event (default: 0).
     */
    public var deltaX: Double? = null

    /**
     * Y delta in DIP for mouse wheel event (default: 0).
     */
    public var deltaY: Double? = null

    /**
     * Bit field representing pressed modifier keys. Alt=1, Ctrl=2, Meta/Command=4, Shift=8
     * (default: 0).
     */
    public var modifiers: Int? = null

    /**
     * Number of times the mouse button was clicked (default: 0).
     */
    public var clickCount: Int? = null

    public fun build(): EmulateTouchFromMouseEventRequest = EmulateTouchFromMouseEventRequest(type,
        x, y, button, timestamp, deltaX, deltaY, modifiers, clickCount)
  }
}

/**
 * A dummy response object for the [InputDomain.emulateTouchFromMouseEvent] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object EmulateTouchFromMouseEventResponse

/**
 * Request object containing input parameters for the [InputDomain.setIgnoreInputEvents] command.
 */
@Serializable
public data class SetIgnoreInputEventsRequest(
  /**
   * Ignores input events processing when set to true.
   */
  public val ignore: Boolean,
)

/**
 * A dummy response object for the [InputDomain.setIgnoreInputEvents] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetIgnoreInputEventsResponse

/**
 * Request object containing input parameters for the [InputDomain.setInterceptDrags] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInterceptDragsRequest(
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [InputDomain.setInterceptDrags] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetInterceptDragsResponse

/**
 * Request object containing input parameters for the [InputDomain.synthesizePinchGesture] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SynthesizePinchGestureRequest(
  /**
   * X coordinate of the start of the gesture in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the start of the gesture in CSS pixels.
   */
  public val y: Double,
  /**
   * Relative scale factor after zooming (>1.0 zooms in, <1.0 zooms out).
   */
  public val scaleFactor: Double,
  /**
   * Relative pointer speed in pixels per second (default: 800).
   */
  public val relativeSpeed: Int? = null,
  /**
   * Which type of input events to be generated (default: 'default', which queries the platform
   * for the preferred input type).
   */
  public val gestureSourceType: GestureSourceType? = null,
) {
  public class Builder(
    /**
     * X coordinate of the start of the gesture in CSS pixels.
     */
    public val x: Double,
    /**
     * Y coordinate of the start of the gesture in CSS pixels.
     */
    public val y: Double,
    /**
     * Relative scale factor after zooming (>1.0 zooms in, <1.0 zooms out).
     */
    public val scaleFactor: Double,
  ) {
    /**
     * Relative pointer speed in pixels per second (default: 800).
     */
    public var relativeSpeed: Int? = null

    /**
     * Which type of input events to be generated (default: 'default', which queries the platform
     * for the preferred input type).
     */
    public var gestureSourceType: GestureSourceType? = null

    public fun build(): SynthesizePinchGestureRequest = SynthesizePinchGestureRequest(x, y,
        scaleFactor, relativeSpeed, gestureSourceType)
  }
}

/**
 * A dummy response object for the [InputDomain.synthesizePinchGesture] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SynthesizePinchGestureResponse

/**
 * Request object containing input parameters for the [InputDomain.synthesizeScrollGesture] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SynthesizeScrollGestureRequest(
  /**
   * X coordinate of the start of the gesture in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the start of the gesture in CSS pixels.
   */
  public val y: Double,
  /**
   * The distance to scroll along the X axis (positive to scroll left).
   */
  public val xDistance: Double? = null,
  /**
   * The distance to scroll along the Y axis (positive to scroll up).
   */
  public val yDistance: Double? = null,
  /**
   * The number of additional pixels to scroll back along the X axis, in addition to the given
   * distance.
   */
  public val xOverscroll: Double? = null,
  /**
   * The number of additional pixels to scroll back along the Y axis, in addition to the given
   * distance.
   */
  public val yOverscroll: Double? = null,
  /**
   * Prevent fling (default: true).
   */
  public val preventFling: Boolean? = null,
  /**
   * Swipe speed in pixels per second (default: 800).
   */
  public val speed: Int? = null,
  /**
   * Which type of input events to be generated (default: 'default', which queries the platform
   * for the preferred input type).
   */
  public val gestureSourceType: GestureSourceType? = null,
  /**
   * The number of times to repeat the gesture (default: 0).
   */
  public val repeatCount: Int? = null,
  /**
   * The number of milliseconds delay between each repeat. (default: 250).
   */
  public val repeatDelayMs: Int? = null,
  /**
   * The name of the interaction markers to generate, if not empty (default: "").
   */
  public val interactionMarkerName: String? = null,
) {
  public class Builder(
    /**
     * X coordinate of the start of the gesture in CSS pixels.
     */
    public val x: Double,
    /**
     * Y coordinate of the start of the gesture in CSS pixels.
     */
    public val y: Double,
  ) {
    /**
     * The distance to scroll along the X axis (positive to scroll left).
     */
    public var xDistance: Double? = null

    /**
     * The distance to scroll along the Y axis (positive to scroll up).
     */
    public var yDistance: Double? = null

    /**
     * The number of additional pixels to scroll back along the X axis, in addition to the given
     * distance.
     */
    public var xOverscroll: Double? = null

    /**
     * The number of additional pixels to scroll back along the Y axis, in addition to the given
     * distance.
     */
    public var yOverscroll: Double? = null

    /**
     * Prevent fling (default: true).
     */
    public var preventFling: Boolean? = null

    /**
     * Swipe speed in pixels per second (default: 800).
     */
    public var speed: Int? = null

    /**
     * Which type of input events to be generated (default: 'default', which queries the platform
     * for the preferred input type).
     */
    public var gestureSourceType: GestureSourceType? = null

    /**
     * The number of times to repeat the gesture (default: 0).
     */
    public var repeatCount: Int? = null

    /**
     * The number of milliseconds delay between each repeat. (default: 250).
     */
    public var repeatDelayMs: Int? = null

    /**
     * The name of the interaction markers to generate, if not empty (default: "").
     */
    public var interactionMarkerName: String? = null

    public fun build(): SynthesizeScrollGestureRequest = SynthesizeScrollGestureRequest(x, y,
        xDistance, yDistance, xOverscroll, yOverscroll, preventFling, speed, gestureSourceType,
        repeatCount, repeatDelayMs, interactionMarkerName)
  }
}

/**
 * A dummy response object for the [InputDomain.synthesizeScrollGesture] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SynthesizeScrollGestureResponse

/**
 * Request object containing input parameters for the [InputDomain.synthesizeTapGesture] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SynthesizeTapGestureRequest(
  /**
   * X coordinate of the start of the gesture in CSS pixels.
   */
  public val x: Double,
  /**
   * Y coordinate of the start of the gesture in CSS pixels.
   */
  public val y: Double,
  /**
   * Duration between touchdown and touchup events in ms (default: 50).
   */
  public val duration: Int? = null,
  /**
   * Number of times to perform the tap (e.g. 2 for double tap, default: 1).
   */
  public val tapCount: Int? = null,
  /**
   * Which type of input events to be generated (default: 'default', which queries the platform
   * for the preferred input type).
   */
  public val gestureSourceType: GestureSourceType? = null,
) {
  public class Builder(
    /**
     * X coordinate of the start of the gesture in CSS pixels.
     */
    public val x: Double,
    /**
     * Y coordinate of the start of the gesture in CSS pixels.
     */
    public val y: Double,
  ) {
    /**
     * Duration between touchdown and touchup events in ms (default: 50).
     */
    public var duration: Int? = null

    /**
     * Number of times to perform the tap (e.g. 2 for double tap, default: 1).
     */
    public var tapCount: Int? = null

    /**
     * Which type of input events to be generated (default: 'default', which queries the platform
     * for the preferred input type).
     */
    public var gestureSourceType: GestureSourceType? = null

    public fun build(): SynthesizeTapGestureRequest = SynthesizeTapGestureRequest(x, y, duration,
        tapCount, gestureSourceType)
  }
}

/**
 * A dummy response object for the [InputDomain.synthesizeTapGesture] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SynthesizeTapGestureResponse

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input)
 */
public class InputDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out InputEvent>> =
      mapOf(
      "Input.dragIntercepted" to serializer<InputEvent.DragIntercepted>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<InputEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Emitted only when `Input.setInterceptDrags` is enabled. Use this data with
   * `Input.dispatchDragEvent` to
   * restore normal drag and drop behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#event-dragIntercepted)
   */
  @ExperimentalChromeApi
  public fun dragInterceptedEvents(): Flow<InputEvent.DragIntercepted> =
      session.typedEvents("Input.dragIntercepted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("dragInterceptedEvents()"),
  )
  public fun dragIntercepted(): Flow<InputEvent.DragIntercepted> = dragInterceptedEvents()

  /**
   * Dispatches a drag event into the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchDragEvent)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun dispatchDragEvent(input: DispatchDragEventRequest): DispatchDragEventResponse =
      session.request("Input.dispatchDragEvent", input)

  /**
   * Dispatches a drag event into the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchDragEvent)
   *
   * @param type Type of the drag event.
   * @param x X coordinate of the event relative to the main frame's viewport in CSS pixels.
   * @param y Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0
   * refers to
   * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun dispatchDragEvent(
    type: DragEventType,
    x: Double,
    y: Double,
    `data`: DragData,
    optionalArgs: DispatchDragEventRequest.Builder.() -> Unit = {},
  ): DispatchDragEventResponse {
    val builder = DispatchDragEventRequest.Builder(type, x, y, data)
    val input = builder.apply(optionalArgs).build()
    return dispatchDragEvent(input)
  }

  /**
   * Dispatches a key event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchKeyEvent)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun dispatchKeyEvent(input: DispatchKeyEventRequest): DispatchKeyEventResponse =
      session.request("Input.dispatchKeyEvent", input)

  /**
   * Dispatches a key event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchKeyEvent)
   *
   * @param type Type of the key event.
   */
  @JvmOverloads
  public suspend inline fun dispatchKeyEvent(type: KeyEventType,
      optionalArgs: DispatchKeyEventRequest.Builder.() -> Unit = {}): DispatchKeyEventResponse {
    val builder = DispatchKeyEventRequest.Builder(type)
    val input = builder.apply(optionalArgs).build()
    return dispatchKeyEvent(input)
  }

  /**
   * This method emulates inserting text that doesn't come from a key press,
   * for example an emoji keyboard or an IME.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-insertText)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun insertText(input: InsertTextRequest): InsertTextResponse =
      session.request("Input.insertText", input)

  /**
   * This method emulates inserting text that doesn't come from a key press,
   * for example an emoji keyboard or an IME.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-insertText)
   *
   * @param text The text to insert.
   */
  @ExperimentalChromeApi
  public suspend fun insertText(text: String): InsertTextResponse {
    val input = InsertTextRequest(text)
    return insertText(input)
  }

  /**
   * This method sets the current candidate text for ime.
   * Use imeCommitComposition to commit the final text.
   * Use imeSetComposition with empty string as text to cancel composition.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-imeSetComposition)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun imeSetComposition(input: ImeSetCompositionRequest): ImeSetCompositionResponse =
      session.request("Input.imeSetComposition", input)

  /**
   * This method sets the current candidate text for ime.
   * Use imeCommitComposition to commit the final text.
   * Use imeSetComposition with empty string as text to cancel composition.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-imeSetComposition)
   *
   * @param text The text to insert
   * @param selectionStart selection start
   * @param selectionEnd selection end
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun imeSetComposition(
    text: String,
    selectionStart: Int,
    selectionEnd: Int,
    optionalArgs: ImeSetCompositionRequest.Builder.() -> Unit = {},
  ): ImeSetCompositionResponse {
    val builder = ImeSetCompositionRequest.Builder(text, selectionStart, selectionEnd)
    val input = builder.apply(optionalArgs).build()
    return imeSetComposition(input)
  }

  /**
   * Dispatches a mouse event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchMouseEvent)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun dispatchMouseEvent(input: DispatchMouseEventRequest):
      DispatchMouseEventResponse = session.request("Input.dispatchMouseEvent", input)

  /**
   * Dispatches a mouse event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchMouseEvent)
   *
   * @param type Type of the mouse event.
   * @param x X coordinate of the event relative to the main frame's viewport in CSS pixels.
   * @param y Y coordinate of the event relative to the main frame's viewport in CSS pixels. 0
   * refers to
   * the top of the viewport and Y increases as it proceeds towards the bottom of the viewport.
   */
  @JvmOverloads
  public suspend inline fun dispatchMouseEvent(
    type: MouseEventType,
    x: Double,
    y: Double,
    optionalArgs: DispatchMouseEventRequest.Builder.() -> Unit = {},
  ): DispatchMouseEventResponse {
    val builder = DispatchMouseEventRequest.Builder(type, x, y)
    val input = builder.apply(optionalArgs).build()
    return dispatchMouseEvent(input)
  }

  /**
   * Dispatches a touch event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchTouchEvent)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun dispatchTouchEvent(input: DispatchTouchEventRequest):
      DispatchTouchEventResponse = session.request("Input.dispatchTouchEvent", input)

  /**
   * Dispatches a touch event to the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-dispatchTouchEvent)
   *
   * @param type Type of the touch event. TouchEnd and TouchCancel must not contain any touch
   * points, while
   * TouchStart and TouchMove must contains at least one.
   * @param touchPoints Active touch points on the touch device. One event per any changed point
   * (compared to
   * previous touch event in a sequence) is generated, emulating pressing/moving/releasing points
   * one by one.
   */
  @JvmOverloads
  public suspend inline fun dispatchTouchEvent(
    type: TouchEventType,
    touchPoints: List<TouchPoint>,
    optionalArgs: DispatchTouchEventRequest.Builder.() -> Unit = {},
  ): DispatchTouchEventResponse {
    val builder = DispatchTouchEventRequest.Builder(type, touchPoints)
    val input = builder.apply(optionalArgs).build()
    return dispatchTouchEvent(input)
  }

  /**
   * Emulates touch event from the mouse event parameters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-emulateTouchFromMouseEvent)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun emulateTouchFromMouseEvent(input: EmulateTouchFromMouseEventRequest):
      EmulateTouchFromMouseEventResponse = session.request("Input.emulateTouchFromMouseEvent",
      input)

  /**
   * Emulates touch event from the mouse event parameters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-emulateTouchFromMouseEvent)
   *
   * @param type Type of the mouse event.
   * @param x X coordinate of the mouse pointer in DIP.
   * @param y Y coordinate of the mouse pointer in DIP.
   * @param button Mouse button. Only "none", "left", "right" are supported.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun emulateTouchFromMouseEvent(
    type: MouseEventType,
    x: Int,
    y: Int,
    button: MouseButton,
    optionalArgs: EmulateTouchFromMouseEventRequest.Builder.() -> Unit = {},
  ): EmulateTouchFromMouseEventResponse {
    val builder = EmulateTouchFromMouseEventRequest.Builder(type, x, y, button)
    val input = builder.apply(optionalArgs).build()
    return emulateTouchFromMouseEvent(input)
  }

  /**
   * Ignores input events (useful while auditing page).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-setIgnoreInputEvents)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setIgnoreInputEvents(input: SetIgnoreInputEventsRequest):
      SetIgnoreInputEventsResponse = session.request("Input.setIgnoreInputEvents", input)

  /**
   * Ignores input events (useful while auditing page).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-setIgnoreInputEvents)
   *
   * @param ignore Ignores input events processing when set to true.
   */
  public suspend fun setIgnoreInputEvents(ignore: Boolean): SetIgnoreInputEventsResponse {
    val input = SetIgnoreInputEventsRequest(ignore)
    return setIgnoreInputEvents(input)
  }

  /**
   * Prevents default drag and drop behavior and instead emits `Input.dragIntercepted` events.
   * Drag and drop behavior can be directly controlled via `Input.dispatchDragEvent`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-setInterceptDrags)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setInterceptDrags(input: SetInterceptDragsRequest): SetInterceptDragsResponse =
      session.request("Input.setInterceptDrags", input)

  /**
   * Prevents default drag and drop behavior and instead emits `Input.dragIntercepted` events.
   * Drag and drop behavior can be directly controlled via `Input.dispatchDragEvent`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-setInterceptDrags)
   */
  @ExperimentalChromeApi
  public suspend fun setInterceptDrags(enabled: Boolean): SetInterceptDragsResponse {
    val input = SetInterceptDragsRequest(enabled)
    return setInterceptDrags(input)
  }

  /**
   * Synthesizes a pinch gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizePinchGesture)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun synthesizePinchGesture(input: SynthesizePinchGestureRequest):
      SynthesizePinchGestureResponse = session.request("Input.synthesizePinchGesture", input)

  /**
   * Synthesizes a pinch gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizePinchGesture)
   *
   * @param x X coordinate of the start of the gesture in CSS pixels.
   * @param y Y coordinate of the start of the gesture in CSS pixels.
   * @param scaleFactor Relative scale factor after zooming (>1.0 zooms in, <1.0 zooms out).
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun synthesizePinchGesture(
    x: Double,
    y: Double,
    scaleFactor: Double,
    optionalArgs: SynthesizePinchGestureRequest.Builder.() -> Unit = {},
  ): SynthesizePinchGestureResponse {
    val builder = SynthesizePinchGestureRequest.Builder(x, y, scaleFactor)
    val input = builder.apply(optionalArgs).build()
    return synthesizePinchGesture(input)
  }

  /**
   * Synthesizes a scroll gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizeScrollGesture)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun synthesizeScrollGesture(input: SynthesizeScrollGestureRequest):
      SynthesizeScrollGestureResponse = session.request("Input.synthesizeScrollGesture", input)

  /**
   * Synthesizes a scroll gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizeScrollGesture)
   *
   * @param x X coordinate of the start of the gesture in CSS pixels.
   * @param y Y coordinate of the start of the gesture in CSS pixels.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun synthesizeScrollGesture(
    x: Double,
    y: Double,
    optionalArgs: SynthesizeScrollGestureRequest.Builder.() -> Unit = {},
  ): SynthesizeScrollGestureResponse {
    val builder = SynthesizeScrollGestureRequest.Builder(x, y)
    val input = builder.apply(optionalArgs).build()
    return synthesizeScrollGesture(input)
  }

  /**
   * Synthesizes a tap gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizeTapGesture)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun synthesizeTapGesture(input: SynthesizeTapGestureRequest):
      SynthesizeTapGestureResponse = session.request("Input.synthesizeTapGesture", input)

  /**
   * Synthesizes a tap gesture over a time period by issuing appropriate touch events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Input/#method-synthesizeTapGesture)
   *
   * @param x X coordinate of the start of the gesture in CSS pixels.
   * @param y Y coordinate of the start of the gesture in CSS pixels.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun synthesizeTapGesture(
    x: Double,
    y: Double,
    optionalArgs: SynthesizeTapGestureRequest.Builder.() -> Unit = {},
  ): SynthesizeTapGestureResponse {
    val builder = SynthesizeTapGestureRequest.Builder(x, y)
    val input = builder.apply(optionalArgs).build()
    return synthesizeTapGesture(input)
  }
}
