@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.inspector

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.inspector.events.InspectorEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [InspectorDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [InspectorDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector)
 */
@ExperimentalChromeApi
public class InspectorDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out InspectorEvent>> =
      mapOf(
      "Inspector.detached" to serializer<InspectorEvent.Detached>(),
      "Inspector.targetCrashed" to serializer<InspectorEvent.TargetCrashed>(),
      "Inspector.targetReloadedAfterCrash" to serializer<InspectorEvent.TargetReloadedAfterCrash>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<InspectorEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fired when remote debugging connection is about to be terminated. Contains detach reason.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#event-detached)
   */
  public fun detachedEvents(): Flow<InspectorEvent.Detached> =
      session.typedEvents("Inspector.detached")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("detachedEvents()"),
  )
  public fun detached(): Flow<InspectorEvent.Detached> = detachedEvents()

  /**
   * Fired when debugging target has crashed
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#event-targetCrashed)
   */
  public fun targetCrashedEvents(): Flow<InspectorEvent.TargetCrashed> =
      session.typedEvents("Inspector.targetCrashed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("targetCrashedEvents()"),
  )
  public fun targetCrashed(): Flow<InspectorEvent.TargetCrashed> = targetCrashedEvents()

  /**
   * Fired when debugging target has reloaded after crash
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#event-targetReloadedAfterCrash)
   */
  public fun targetReloadedAfterCrashEvents(): Flow<InspectorEvent.TargetReloadedAfterCrash> =
      session.typedEvents("Inspector.targetReloadedAfterCrash")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("targetReloadedAfterCrashEvents()"),
  )
  public fun targetReloadedAfterCrash(): Flow<InspectorEvent.TargetReloadedAfterCrash> =
      targetReloadedAfterCrashEvents()

  /**
   * Disables inspector domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Inspector.disable", Unit)

  /**
   * Enables inspector domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Inspector.enable", Unit)
}
