@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.io

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the [IODomain.close] command.
 */
@Serializable
public data class CloseRequest(
  /**
   * Handle of the stream to close.
   */
  public val handle: StreamHandle,
)

/**
 * A dummy response object for the [IODomain.close] command. This command doesn't return any result
 * at the moment, but this could happen in the future, or could have happened in the past. For forwards
 * and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object CloseResponse

/**
 * Request object containing input parameters for the [IODomain.read] command.
 */
@Serializable
public data class ReadRequest(
  /**
   * Handle of the stream to read.
   */
  public val handle: StreamHandle,
  /**
   * Seek to the specified offset before reading (if not specificed, proceed with offset
   * following the last read). Some types of streams may only support sequential reads.
   */
  public val offset: Int? = null,
  /**
   * Maximum number of bytes to read (left upon the agent discretion if not specified).
   */
  public val size: Int? = null,
) {
  public class Builder(
    /**
     * Handle of the stream to read.
     */
    public val handle: StreamHandle,
  ) {
    /**
     * Seek to the specified offset before reading (if not specificed, proceed with offset
     * following the last read). Some types of streams may only support sequential reads.
     */
    public var offset: Int? = null

    /**
     * Maximum number of bytes to read (left upon the agent discretion if not specified).
     */
    public var size: Int? = null

    public fun build(): ReadRequest = ReadRequest(handle, offset, size)
  }
}

/**
 * Response type for the [IODomain.read] command.
 */
@Serializable
public data class ReadResponse(
  /**
   * Set if the data is base64-encoded
   */
  public val base64Encoded: Boolean? = null,
  /**
   * Data that were read.
   */
  public val `data`: String,
  /**
   * Set if the end-of-file condition occurred while reading.
   */
  public val eof: Boolean,
)

/**
 * Request object containing input parameters for the [IODomain.resolveBlob] command.
 */
@Serializable
public data class ResolveBlobRequest(
  /**
   * Object id of a Blob object wrapper.
   */
  public val objectId: RemoteObjectId,
)

/**
 * Response type for the [IODomain.resolveBlob] command.
 */
@Serializable
public data class ResolveBlobResponse(
  /**
   * UUID of the specified Blob.
   */
  public val uuid: String,
)

/**
 * Input/Output operations for streams produced by DevTools.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO)
 */
public class IODomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Close the stream, discard any temporary backing storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-close)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun close(input: CloseRequest): CloseResponse = session.request("IO.close", input)

  /**
   * Close the stream, discard any temporary backing storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-close)
   *
   * @param handle Handle of the stream to close.
   */
  public suspend fun close(handle: StreamHandle): CloseResponse {
    val input = CloseRequest(handle)
    return close(input)
  }

  /**
   * Read a chunk of the stream
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-read)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun read(input: ReadRequest): ReadResponse = session.request("IO.read", input)

  /**
   * Read a chunk of the stream
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-read)
   *
   * @param handle Handle of the stream to read.
   */
  @JvmOverloads
  public suspend inline fun read(handle: StreamHandle, optionalArgs: ReadRequest.Builder.() -> Unit
      = {}): ReadResponse {
    val builder = ReadRequest.Builder(handle)
    val input = builder.apply(optionalArgs).build()
    return read(input)
  }

  /**
   * Return UUID of Blob object specified by a remote object id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-resolveBlob)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun resolveBlob(input: ResolveBlobRequest): ResolveBlobResponse =
      session.request("IO.resolveBlob", input)

  /**
   * Return UUID of Blob object specified by a remote object id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IO/#method-resolveBlob)
   *
   * @param objectId Object id of a Blob object wrapper.
   */
  public suspend fun resolveBlob(objectId: RemoteObjectId): ResolveBlobResponse {
    val input = ResolveBlobRequest(objectId)
    return resolveBlob(input)
  }
}
