@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.layertree

import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.Rect
import org.hildan.chrome.devtools.domains.layertree.events.LayerTreeEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [LayerTreeDomain.compositingReasons] command.
 */
@Serializable
public data class CompositingReasonsRequest(
  /**
   * The id of the layer for which we want to get the reasons it was composited.
   */
  public val layerId: LayerId,
)

/**
 * Response type for the [LayerTreeDomain.compositingReasons] command.
 */
@Serializable
public data class CompositingReasonsResponse(
  /**
   * A list of strings specifying reasons for the given layer to become composited.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val compositingReasons: List<String>,
  /**
   * A list of strings specifying reason IDs for the given layer to become composited.
   */
  public val compositingReasonIds: List<String>,
)

/**
 * A dummy response object for the [LayerTreeDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [LayerTreeDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [LayerTreeDomain.loadSnapshot] command.
 */
@Serializable
public data class LoadSnapshotRequest(
  /**
   * An array of tiles composing the snapshot.
   */
  public val tiles: List<PictureTile>,
)

/**
 * Response type for the [LayerTreeDomain.loadSnapshot] command.
 */
@Serializable
public data class LoadSnapshotResponse(
  /**
   * The id of the snapshot.
   */
  public val snapshotId: SnapshotId,
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.makeSnapshot] command.
 */
@Serializable
public data class MakeSnapshotRequest(
  /**
   * The id of the layer.
   */
  public val layerId: LayerId,
)

/**
 * Response type for the [LayerTreeDomain.makeSnapshot] command.
 */
@Serializable
public data class MakeSnapshotResponse(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.profileSnapshot] command.
 */
@Serializable
public data class ProfileSnapshotRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
  /**
   * The maximum number of times to replay the snapshot (1, if not specified).
   */
  public val minRepeatCount: Int? = null,
  /**
   * The minimum duration (in seconds) to replay the snapshot.
   */
  public val minDuration: Double? = null,
  /**
   * The clip rectangle to apply when replaying the snapshot.
   */
  public val clipRect: Rect? = null,
) {
  public class Builder(
    /**
     * The id of the layer snapshot.
     */
    public val snapshotId: SnapshotId,
  ) {
    /**
     * The maximum number of times to replay the snapshot (1, if not specified).
     */
    public var minRepeatCount: Int? = null

    /**
     * The minimum duration (in seconds) to replay the snapshot.
     */
    public var minDuration: Double? = null

    /**
     * The clip rectangle to apply when replaying the snapshot.
     */
    public var clipRect: Rect? = null

    public fun build(): ProfileSnapshotRequest = ProfileSnapshotRequest(snapshotId, minRepeatCount,
        minDuration, clipRect)
  }
}

/**
 * Response type for the [LayerTreeDomain.profileSnapshot] command.
 */
@Serializable
public data class ProfileSnapshotResponse(
  /**
   * The array of paint profiles, one per run.
   */
  public val timings: List<PaintProfile>,
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.releaseSnapshot] command.
 */
@Serializable
public data class ReleaseSnapshotRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
)

/**
 * A dummy response object for the [LayerTreeDomain.releaseSnapshot] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ReleaseSnapshotResponse

/**
 * Request object containing input parameters for the [LayerTreeDomain.replaySnapshot] command.
 */
@Serializable
public data class ReplaySnapshotRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
  /**
   * The first step to replay from (replay from the very start if not specified).
   */
  public val fromStep: Int? = null,
  /**
   * The last step to replay to (replay till the end if not specified).
   */
  public val toStep: Int? = null,
  /**
   * The scale to apply while replaying (defaults to 1).
   */
  public val scale: Double? = null,
) {
  public class Builder(
    /**
     * The id of the layer snapshot.
     */
    public val snapshotId: SnapshotId,
  ) {
    /**
     * The first step to replay from (replay from the very start if not specified).
     */
    public var fromStep: Int? = null

    /**
     * The last step to replay to (replay till the end if not specified).
     */
    public var toStep: Int? = null

    /**
     * The scale to apply while replaying (defaults to 1).
     */
    public var scale: Double? = null

    public fun build(): ReplaySnapshotRequest = ReplaySnapshotRequest(snapshotId, fromStep, toStep,
        scale)
  }
}

/**
 * Response type for the [LayerTreeDomain.replaySnapshot] command.
 */
@Serializable
public data class ReplaySnapshotResponse(
  /**
   * A data: URL for resulting image.
   */
  public val dataURL: String,
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.snapshotCommandLog] command.
 */
@Serializable
public data class SnapshotCommandLogRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
)

/**
 * Response type for the [LayerTreeDomain.snapshotCommandLog] command.
 */
@Serializable
public data class SnapshotCommandLogResponse(
  /**
   * The array of canvas function calls.
   */
  public val commandLog: List<JsonObject>,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree)
 */
@ExperimentalChromeApi
public class LayerTreeDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out LayerTreeEvent>> =
      mapOf(
      "LayerTree.layerPainted" to serializer<LayerTreeEvent.LayerPainted>(),
      "LayerTree.layerTreeDidChange" to serializer<LayerTreeEvent.LayerTreeDidChange>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<LayerTreeEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#event-layerPainted)
   */
  public fun layerPaintedEvents(): Flow<LayerTreeEvent.LayerPainted> =
      session.typedEvents("LayerTree.layerPainted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("layerPaintedEvents()"),
  )
  public fun layerPainted(): Flow<LayerTreeEvent.LayerPainted> = layerPaintedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#event-layerTreeDidChange)
   */
  public fun layerTreeDidChangeEvents(): Flow<LayerTreeEvent.LayerTreeDidChange> =
      session.typedEvents("LayerTree.layerTreeDidChange")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("layerTreeDidChangeEvents()"),
  )
  public fun layerTreeDidChange(): Flow<LayerTreeEvent.LayerTreeDidChange> =
      layerTreeDidChangeEvents()

  /**
   * Provides the reasons why the given layer was composited.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-compositingReasons)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun compositingReasons(input: CompositingReasonsRequest):
      CompositingReasonsResponse = session.request("LayerTree.compositingReasons", input)

  /**
   * Provides the reasons why the given layer was composited.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-compositingReasons)
   *
   * @param layerId The id of the layer for which we want to get the reasons it was composited.
   */
  public suspend fun compositingReasons(layerId: LayerId): CompositingReasonsResponse {
    val input = CompositingReasonsRequest(layerId)
    return compositingReasons(input)
  }

  /**
   * Disables compositing tree inspection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("LayerTree.disable", Unit)

  /**
   * Enables compositing tree inspection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("LayerTree.enable", Unit)

  /**
   * Returns the snapshot identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-loadSnapshot)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun loadSnapshot(input: LoadSnapshotRequest): LoadSnapshotResponse =
      session.request("LayerTree.loadSnapshot", input)

  /**
   * Returns the snapshot identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-loadSnapshot)
   *
   * @param tiles An array of tiles composing the snapshot.
   */
  public suspend fun loadSnapshot(tiles: List<PictureTile>): LoadSnapshotResponse {
    val input = LoadSnapshotRequest(tiles)
    return loadSnapshot(input)
  }

  /**
   * Returns the layer snapshot identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-makeSnapshot)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun makeSnapshot(input: MakeSnapshotRequest): MakeSnapshotResponse =
      session.request("LayerTree.makeSnapshot", input)

  /**
   * Returns the layer snapshot identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-makeSnapshot)
   *
   * @param layerId The id of the layer.
   */
  public suspend fun makeSnapshot(layerId: LayerId): MakeSnapshotResponse {
    val input = MakeSnapshotRequest(layerId)
    return makeSnapshot(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-profileSnapshot)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun profileSnapshot(input: ProfileSnapshotRequest): ProfileSnapshotResponse =
      session.request("LayerTree.profileSnapshot", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-profileSnapshot)
   *
   * @param snapshotId The id of the layer snapshot.
   */
  @JvmOverloads
  public suspend inline fun profileSnapshot(snapshotId: SnapshotId,
      optionalArgs: ProfileSnapshotRequest.Builder.() -> Unit = {}): ProfileSnapshotResponse {
    val builder = ProfileSnapshotRequest.Builder(snapshotId)
    val input = builder.apply(optionalArgs).build()
    return profileSnapshot(input)
  }

  /**
   * Releases layer snapshot captured by the back-end.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-releaseSnapshot)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun releaseSnapshot(input: ReleaseSnapshotRequest): ReleaseSnapshotResponse =
      session.request("LayerTree.releaseSnapshot", input)

  /**
   * Releases layer snapshot captured by the back-end.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-releaseSnapshot)
   *
   * @param snapshotId The id of the layer snapshot.
   */
  public suspend fun releaseSnapshot(snapshotId: SnapshotId): ReleaseSnapshotResponse {
    val input = ReleaseSnapshotRequest(snapshotId)
    return releaseSnapshot(input)
  }

  /**
   * Replays the layer snapshot and returns the resulting bitmap.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-replaySnapshot)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun replaySnapshot(input: ReplaySnapshotRequest): ReplaySnapshotResponse =
      session.request("LayerTree.replaySnapshot", input)

  /**
   * Replays the layer snapshot and returns the resulting bitmap.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-replaySnapshot)
   *
   * @param snapshotId The id of the layer snapshot.
   */
  @JvmOverloads
  public suspend inline fun replaySnapshot(snapshotId: SnapshotId,
      optionalArgs: ReplaySnapshotRequest.Builder.() -> Unit = {}): ReplaySnapshotResponse {
    val builder = ReplaySnapshotRequest.Builder(snapshotId)
    val input = builder.apply(optionalArgs).build()
    return replaySnapshot(input)
  }

  /**
   * Replays the layer snapshot and returns canvas log.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-snapshotCommandLog)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun snapshotCommandLog(input: SnapshotCommandLogRequest):
      SnapshotCommandLogResponse = session.request("LayerTree.snapshotCommandLog", input)

  /**
   * Replays the layer snapshot and returns canvas log.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-snapshotCommandLog)
   *
   * @param snapshotId The id of the layer snapshot.
   */
  public suspend fun snapshotCommandLog(snapshotId: SnapshotId): SnapshotCommandLogResponse {
    val input = SnapshotCommandLogRequest(snapshotId)
    return snapshotCommandLog(input)
  }
}
