@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.log

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.log.events.LogEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [LogDomain.clear] command. This command doesn't return any result
 * at the moment, but this could happen in the future, or could have happened in the past. For forwards
 * and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object ClearResponse

/**
 * A dummy response object for the [LogDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [LogDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [LogDomain.startViolationsReport] command.
 */
@Serializable
public data class StartViolationsReportRequest(
  /**
   * Configuration for violations.
   */
  public val config: List<ViolationSetting>,
)

/**
 * A dummy response object for the [LogDomain.startViolationsReport] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object StartViolationsReportResponse

/**
 * A dummy response object for the [LogDomain.stopViolationsReport] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object StopViolationsReportResponse

/**
 * Provides access to log entries.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log)
 */
public class LogDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out LogEvent>> = mapOf(
      "Log.entryAdded" to serializer<LogEvent.EntryAdded>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<LogEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Issued when new message was logged.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#event-entryAdded)
   */
  public fun entryAddedEvents(): Flow<LogEvent.EntryAdded> = session.typedEvents("Log.entryAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("entryAddedEvents()"),
  )
  public fun entryAdded(): Flow<LogEvent.EntryAdded> = entryAddedEvents()

  /**
   * Clears the log.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#method-clear)
   */
  public suspend fun clear(): ClearResponse = session.request("Log.clear", Unit)

  /**
   * Disables log domain, prevents further log entries from being reported to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Log.disable", Unit)

  /**
   * Enables log domain, sends the entries collected so far to the client by means of the
   * `entryAdded` notification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Log.enable", Unit)

  /**
   * start violation reporting.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#method-startViolationsReport)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startViolationsReport(input: StartViolationsReportRequest):
      StartViolationsReportResponse = session.request("Log.startViolationsReport", input)

  /**
   * start violation reporting.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#method-startViolationsReport)
   *
   * @param config Configuration for violations.
   */
  public suspend fun startViolationsReport(config: List<ViolationSetting>):
      StartViolationsReportResponse {
    val input = StartViolationsReportRequest(config)
    return startViolationsReport(input)
  }

  /**
   * Stop violation reporting.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Log/#method-stopViolationsReport)
   */
  public suspend fun stopViolationsReport(): StopViolationsReportResponse =
      session.request("Log.stopViolationsReport", Unit)
}
