@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.media

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.media.events.MediaEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [MediaDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * A dummy response object for the [MediaDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * This domain allows detailed inspection of media elements
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media)
 */
@ExperimentalChromeApi
public class MediaDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out MediaEvent>> =
      mapOf(
      "Media.playerPropertiesChanged" to serializer<MediaEvent.PlayerPropertiesChanged>(),
      "Media.playerEventsAdded" to serializer<MediaEvent.PlayerEventsAdded>(),
      "Media.playerMessagesLogged" to serializer<MediaEvent.PlayerMessagesLogged>(),
      "Media.playerErrorsRaised" to serializer<MediaEvent.PlayerErrorsRaised>(),
      "Media.playersCreated" to serializer<MediaEvent.PlayersCreated>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<MediaEvent> = session.typedEvents(deserializersByEventName)

  /**
   * This can be called multiple times, and can be used to set / override /
   * remove player properties. A null propValue indicates removal.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerPropertiesChanged)
   */
  public fun playerPropertiesChangedEvents(): Flow<MediaEvent.PlayerPropertiesChanged> =
      session.typedEvents("Media.playerPropertiesChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("playerPropertiesChangedEvents()"),
  )
  public fun playerPropertiesChanged(): Flow<MediaEvent.PlayerPropertiesChanged> =
      playerPropertiesChangedEvents()

  /**
   * Send events as a list, allowing them to be batched on the browser for less
   * congestion. If batched, events must ALWAYS be in chronological order.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerEventsAdded)
   */
  public fun playerEventsAddedEvents(): Flow<MediaEvent.PlayerEventsAdded> =
      session.typedEvents("Media.playerEventsAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("playerEventsAddedEvents()"),
  )
  public fun playerEventsAdded(): Flow<MediaEvent.PlayerEventsAdded> = playerEventsAddedEvents()

  /**
   * Send a list of any messages that need to be delivered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerMessagesLogged)
   */
  public fun playerMessagesLoggedEvents(): Flow<MediaEvent.PlayerMessagesLogged> =
      session.typedEvents("Media.playerMessagesLogged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("playerMessagesLoggedEvents()"),
  )
  public fun playerMessagesLogged(): Flow<MediaEvent.PlayerMessagesLogged> =
      playerMessagesLoggedEvents()

  /**
   * Send a list of any errors that need to be delivered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerErrorsRaised)
   */
  public fun playerErrorsRaisedEvents(): Flow<MediaEvent.PlayerErrorsRaised> =
      session.typedEvents("Media.playerErrorsRaised")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("playerErrorsRaisedEvents()"),
  )
  public fun playerErrorsRaised(): Flow<MediaEvent.PlayerErrorsRaised> = playerErrorsRaisedEvents()

  /**
   * Called whenever a player is created, or when a new agent joins and receives
   * a list of active players. If an agent is restored, it will receive the full
   * list of player ids and all events again.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playersCreated)
   */
  public fun playersCreatedEvents(): Flow<MediaEvent.PlayersCreated> =
      session.typedEvents("Media.playersCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("playersCreatedEvents()"),
  )
  public fun playersCreated(): Flow<MediaEvent.PlayersCreated> = playersCreatedEvents()

  /**
   * Enables the Media domain
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Media.enable", Unit)

  /**
   * Disables the Media domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Media.disable", Unit)
}
