@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.media

import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject

/**
 * Players will get an ID that is unique within the agent context.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerId)
 */
public typealias PlayerId = String

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-Timestamp)
 */
public typealias Timestamp = Double

/**
 * Have one type per entry in MediaLogRecord::Type
 * Corresponds to kMessage
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerMessage)
 */
@Serializable
public data class PlayerMessage(
  /**
   * Keep in sync with MediaLogMessageLevel
   * We are currently keeping the message level 'error' separate from the
   * PlayerError type because right now they represent different things,
   * this one being a DVLOG(ERROR) style log message that gets printed
   * based on what log level is selected in the UI, and the other is a
   * representation of a media::PipelineStatus object. Soon however we're
   * going to be moving away from using PipelineStatus for errors and
   * introducing a new error type which should hopefully let us integrate
   * the error log level into the PlayerError type.
   */
  public val level: PlayerMessageLevel,
  public val message: String,
)

/**
 * Corresponds to kMediaPropertyChange
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerProperty)
 */
@Serializable
public data class PlayerProperty(
  public val name: String,
  public val `value`: String,
)

/**
 * Corresponds to kMediaEventTriggered
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerEvent)
 */
@Serializable
public data class PlayerEvent(
  public val timestamp: Timestamp,
  public val `value`: String,
)

/**
 * Represents logged source line numbers reported in an error.
 * NOTE: file and line are from chromium c++ implementation code, not js.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerErrorSourceLocation)
 */
@Serializable
public data class PlayerErrorSourceLocation(
  public val `file`: String,
  public val line: Int,
)

/**
 * Corresponds to kMediaError
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerError)
 */
@Serializable
public data class PlayerError(
  public val errorType: String,
  /**
   * Code is the numeric enum entry for a specific set of error codes, such
   * as PipelineStatusCodes in media/base/pipeline_status.h
   */
  public val code: Int,
  /**
   * A trace of where this error was caused / where it passed through.
   */
  public val stack: List<PlayerErrorSourceLocation>,
  /**
   * Errors potentially have a root cause error, ie, a DecoderError might be
   * caused by an WindowsError
   */
  public val cause: List<PlayerError>,
  /**
   * Extra data attached to an error, such as an HRESULT, Video Codec, etc.
   */
  public val `data`: JsonObject,
)

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the property 'level' of the type 'PlayerMessage'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#type-PlayerMessageLevel)
 */
@Serializable
public enum class PlayerMessageLevel {
  error,
  warning,
  info,
  debug,
}
