@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.memory

import kotlin.Boolean
import kotlin.Int
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Response type for the [MemoryDomain.getDOMCounters] command.
 */
@Serializable
public data class GetDOMCountersResponse(
  public val documents: Int,
  public val nodes: Int,
  public val jsEventListeners: Int,
)

/**
 * A dummy response object for the [MemoryDomain.prepareForLeakDetection] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object PrepareForLeakDetectionResponse

/**
 * A dummy response object for the [MemoryDomain.forciblyPurgeJavaScriptMemory] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object ForciblyPurgeJavaScriptMemoryResponse

/**
 * Request object containing input parameters for the
 * [MemoryDomain.setPressureNotificationsSuppressed] command.
 */
@Serializable
public data class SetPressureNotificationsSuppressedRequest(
  /**
   * If true, memory pressure notifications will be suppressed.
   */
  public val suppressed: Boolean,
)

/**
 * A dummy response object for the [MemoryDomain.setPressureNotificationsSuppressed] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetPressureNotificationsSuppressedResponse

/**
 * Request object containing input parameters for the [MemoryDomain.simulatePressureNotification]
 * command.
 */
@Serializable
public data class SimulatePressureNotificationRequest(
  /**
   * Memory pressure level of the notification.
   */
  public val level: PressureLevel,
)

/**
 * A dummy response object for the [MemoryDomain.simulatePressureNotification] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SimulatePressureNotificationResponse

/**
 * Request object containing input parameters for the [MemoryDomain.startSampling] command.
 */
@Serializable
public data class StartSamplingRequest(
  /**
   * Average number of bytes between samples.
   */
  public val samplingInterval: Int? = null,
  /**
   * Do not randomize intervals between samples.
   */
  public val suppressRandomness: Boolean? = null,
) {
  public class Builder() {
    /**
     * Average number of bytes between samples.
     */
    public var samplingInterval: Int? = null

    /**
     * Do not randomize intervals between samples.
     */
    public var suppressRandomness: Boolean? = null

    public fun build(): StartSamplingRequest = StartSamplingRequest(samplingInterval,
        suppressRandomness)
  }
}

/**
 * A dummy response object for the [MemoryDomain.startSampling] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object StartSamplingResponse

/**
 * A dummy response object for the [MemoryDomain.stopSampling] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object StopSamplingResponse

/**
 * Response type for the [MemoryDomain.getAllTimeSamplingProfile] command.
 */
@Serializable
public data class GetAllTimeSamplingProfileResponse(
  public val profile: SamplingProfile,
)

/**
 * Response type for the [MemoryDomain.getBrowserSamplingProfile] command.
 */
@Serializable
public data class GetBrowserSamplingProfileResponse(
  public val profile: SamplingProfile,
)

/**
 * Response type for the [MemoryDomain.getSamplingProfile] command.
 */
@Serializable
public data class GetSamplingProfileResponse(
  public val profile: SamplingProfile,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory)
 */
@ExperimentalChromeApi
public class MemoryDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-getDOMCounters)
   */
  public suspend fun getDOMCounters(): GetDOMCountersResponse =
      session.request("Memory.getDOMCounters", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-prepareForLeakDetection)
   */
  public suspend fun prepareForLeakDetection(): PrepareForLeakDetectionResponse =
      session.request("Memory.prepareForLeakDetection", Unit)

  /**
   * Simulate OomIntervention by purging V8 memory.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-forciblyPurgeJavaScriptMemory)
   */
  public suspend fun forciblyPurgeJavaScriptMemory(): ForciblyPurgeJavaScriptMemoryResponse =
      session.request("Memory.forciblyPurgeJavaScriptMemory", Unit)

  /**
   * Enable/disable suppressing memory pressure notifications in all processes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-setPressureNotificationsSuppressed)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend
      fun setPressureNotificationsSuppressed(input: SetPressureNotificationsSuppressedRequest):
      SetPressureNotificationsSuppressedResponse =
      session.request("Memory.setPressureNotificationsSuppressed", input)

  /**
   * Enable/disable suppressing memory pressure notifications in all processes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-setPressureNotificationsSuppressed)
   *
   * @param suppressed If true, memory pressure notifications will be suppressed.
   */
  public suspend fun setPressureNotificationsSuppressed(suppressed: Boolean):
      SetPressureNotificationsSuppressedResponse {
    val input = SetPressureNotificationsSuppressedRequest(suppressed)
    return setPressureNotificationsSuppressed(input)
  }

  /**
   * Simulate a memory pressure notification in all processes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-simulatePressureNotification)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun simulatePressureNotification(input: SimulatePressureNotificationRequest):
      SimulatePressureNotificationResponse = session.request("Memory.simulatePressureNotification",
      input)

  /**
   * Simulate a memory pressure notification in all processes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-simulatePressureNotification)
   *
   * @param level Memory pressure level of the notification.
   */
  public suspend fun simulatePressureNotification(level: PressureLevel):
      SimulatePressureNotificationResponse {
    val input = SimulatePressureNotificationRequest(level)
    return simulatePressureNotification(input)
  }

  /**
   * Start collecting native memory profile.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-startSampling)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startSampling(input: StartSamplingRequest): StartSamplingResponse =
      session.request("Memory.startSampling", input)

  /**
   * Start collecting native memory profile.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-startSampling)
   */
  @JvmOverloads
  public suspend inline fun startSampling(optionalArgs: StartSamplingRequest.Builder.() -> Unit =
      {}): StartSamplingResponse {
    val builder = StartSamplingRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return startSampling(input)
  }

  /**
   * Stop collecting native memory profile.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-stopSampling)
   */
  public suspend fun stopSampling(): StopSamplingResponse = session.request("Memory.stopSampling",
      Unit)

  /**
   * Retrieve native memory allocations profile
   * collected since renderer process startup.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-getAllTimeSamplingProfile)
   */
  public suspend fun getAllTimeSamplingProfile(): GetAllTimeSamplingProfileResponse =
      session.request("Memory.getAllTimeSamplingProfile", Unit)

  /**
   * Retrieve native memory allocations profile
   * collected since browser process startup.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-getBrowserSamplingProfile)
   */
  public suspend fun getBrowserSamplingProfile(): GetBrowserSamplingProfileResponse =
      session.request("Memory.getBrowserSamplingProfile", Unit)

  /**
   * Retrieve native memory allocations profile collected since last
   * `startSampling` call.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Memory/#method-getSamplingProfile)
   */
  public suspend fun getSamplingProfile(): GetSamplingProfileResponse =
      session.request("Memory.getSamplingProfile", Unit)
}
