@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.network

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.debugger.SearchMatch
import org.hildan.chrome.devtools.domains.emulation.UserAgentMetadata
import org.hildan.chrome.devtools.domains.io.StreamHandle
import org.hildan.chrome.devtools.domains.network.events.NetworkEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [NetworkDomain.setAcceptedEncodings] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAcceptedEncodingsRequest(
  /**
   * List of accepted content encodings.
   */
  public val encodings: List<ContentEncoding>,
)

/**
 * A dummy response object for the [NetworkDomain.setAcceptedEncodings] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetAcceptedEncodingsResponse

/**
 * A dummy response object for the [NetworkDomain.clearAcceptedEncodingsOverride] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ClearAcceptedEncodingsOverrideResponse

/**
 * Response type for the [NetworkDomain.canClearBrowserCache] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class CanClearBrowserCacheResponse(
  /**
   * True if browser cache can be cleared.
   */
  public val result: Boolean,
)

/**
 * Response type for the [NetworkDomain.canClearBrowserCookies] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class CanClearBrowserCookiesResponse(
  /**
   * True if browser cookies can be cleared.
   */
  public val result: Boolean,
)

/**
 * Response type for the [NetworkDomain.canEmulateNetworkConditions] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class CanEmulateNetworkConditionsResponse(
  /**
   * True if emulation of network conditions is supported.
   */
  public val result: Boolean,
)

/**
 * A dummy response object for the [NetworkDomain.clearBrowserCache] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ClearBrowserCacheResponse

/**
 * A dummy response object for the [NetworkDomain.clearBrowserCookies] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ClearBrowserCookiesResponse

/**
 * Request object containing input parameters for the [NetworkDomain.continueInterceptedRequest]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class ContinueInterceptedRequestRequest(
  public val interceptionId: InterceptionId,
  /**
   * If set this causes the request to fail with the given reason. Passing `Aborted` for requests
   * marked with `isNavigationRequest` also cancels the navigation. Must not be set in response
   * to an authChallenge.
   */
  public val errorReason: ErrorReason? = null,
  /**
   * If set the requests completes using with the provided base64 encoded raw response, including
   * HTTP status line and headers etc... Must not be set in response to an authChallenge. (Encoded
   * as a base64 string when passed over JSON)
   */
  public val rawResponse: String? = null,
  /**
   * If set the request url will be modified in a way that's not observable by page. Must not be
   * set in response to an authChallenge.
   */
  public val url: String? = null,
  /**
   * If set this allows the request method to be overridden. Must not be set in response to an
   * authChallenge.
   */
  public val method: String? = null,
  /**
   * If set this allows postData to be set. Must not be set in response to an authChallenge.
   */
  public val postData: String? = null,
  /**
   * If set this allows the request headers to be changed. Must not be set in response to an
   * authChallenge.
   */
  public val headers: Headers? = null,
  /**
   * Response to a requestIntercepted with an authChallenge. Must not be set otherwise.
   */
  public val authChallengeResponse: AuthChallengeResponse? = null,
) {
  public class Builder(
    public val interceptionId: InterceptionId,
  ) {
    /**
     * If set this causes the request to fail with the given reason. Passing `Aborted` for requests
     * marked with `isNavigationRequest` also cancels the navigation. Must not be set in response
     * to an authChallenge.
     */
    public var errorReason: ErrorReason? = null

    /**
     * If set the requests completes using with the provided base64 encoded raw response, including
     * HTTP status line and headers etc... Must not be set in response to an authChallenge. (Encoded
     * as a base64 string when passed over JSON)
     */
    public var rawResponse: String? = null

    /**
     * If set the request url will be modified in a way that's not observable by page. Must not be
     * set in response to an authChallenge.
     */
    public var url: String? = null

    /**
     * If set this allows the request method to be overridden. Must not be set in response to an
     * authChallenge.
     */
    public var method: String? = null

    /**
     * If set this allows postData to be set. Must not be set in response to an authChallenge.
     */
    public var postData: String? = null

    /**
     * If set this allows the request headers to be changed. Must not be set in response to an
     * authChallenge.
     */
    public var headers: Headers? = null

    /**
     * Response to a requestIntercepted with an authChallenge. Must not be set otherwise.
     */
    public var authChallengeResponse: AuthChallengeResponse? = null

    public fun build(): ContinueInterceptedRequestRequest =
        ContinueInterceptedRequestRequest(interceptionId, errorReason, rawResponse, url, method,
        postData, headers, authChallengeResponse)
  }
}

/**
 * A dummy response object for the [NetworkDomain.continueInterceptedRequest] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public object ContinueInterceptedRequestResponse

/**
 * Request object containing input parameters for the [NetworkDomain.deleteCookies] command.
 */
@Serializable
public data class DeleteCookiesRequest(
  /**
   * Name of the cookies to remove.
   */
  public val name: String,
  /**
   * If specified, deletes all the cookies with the given name where domain and path match
   * provided URL.
   */
  public val url: String? = null,
  /**
   * If specified, deletes only cookies with the exact domain.
   */
  public val domain: String? = null,
  /**
   * If specified, deletes only cookies with the exact path.
   */
  public val path: String? = null,
) {
  public class Builder(
    /**
     * Name of the cookies to remove.
     */
    public val name: String,
  ) {
    /**
     * If specified, deletes all the cookies with the given name where domain and path match
     * provided URL.
     */
    public var url: String? = null

    /**
     * If specified, deletes only cookies with the exact domain.
     */
    public var domain: String? = null

    /**
     * If specified, deletes only cookies with the exact path.
     */
    public var path: String? = null

    public fun build(): DeleteCookiesRequest = DeleteCookiesRequest(name, url, domain, path)
  }
}

/**
 * A dummy response object for the [NetworkDomain.deleteCookies] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DeleteCookiesResponse

/**
 * A dummy response object for the [NetworkDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [NetworkDomain.emulateNetworkConditions]
 * command.
 */
@Serializable
public data class EmulateNetworkConditionsRequest(
  /**
   * True to emulate internet disconnection.
   */
  public val offline: Boolean,
  /**
   * Minimum latency from request sent to response headers received (ms).
   */
  public val latency: Double,
  /**
   * Maximal aggregated download throughput (bytes/sec). -1 disables download throttling.
   */
  public val downloadThroughput: Double,
  /**
   * Maximal aggregated upload throughput (bytes/sec).  -1 disables upload throttling.
   */
  public val uploadThroughput: Double,
  /**
   * Connection type if known.
   */
  public val connectionType: ConnectionType? = null,
) {
  public class Builder(
    /**
     * True to emulate internet disconnection.
     */
    public val offline: Boolean,
    /**
     * Minimum latency from request sent to response headers received (ms).
     */
    public val latency: Double,
    /**
     * Maximal aggregated download throughput (bytes/sec). -1 disables download throttling.
     */
    public val downloadThroughput: Double,
    /**
     * Maximal aggregated upload throughput (bytes/sec).  -1 disables upload throttling.
     */
    public val uploadThroughput: Double,
  ) {
    /**
     * Connection type if known.
     */
    public var connectionType: ConnectionType? = null

    public fun build(): EmulateNetworkConditionsRequest = EmulateNetworkConditionsRequest(offline,
        latency, downloadThroughput, uploadThroughput, connectionType)
  }
}

/**
 * A dummy response object for the [NetworkDomain.emulateNetworkConditions] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object EmulateNetworkConditionsResponse

/**
 * Request object containing input parameters for the [NetworkDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * Buffer size in bytes to use when preserving network payloads (XHRs, etc).
   */
  @ExperimentalChromeApi
  public val maxTotalBufferSize: Int? = null,
  /**
   * Per-resource buffer size in bytes to use when preserving network payloads (XHRs, etc).
   */
  @ExperimentalChromeApi
  public val maxResourceBufferSize: Int? = null,
  /**
   * Longest post body size (in bytes) that would be included in requestWillBeSent notification
   */
  public val maxPostDataSize: Int? = null,
) {
  public class Builder() {
    /**
     * Buffer size in bytes to use when preserving network payloads (XHRs, etc).
     */
    @ExperimentalChromeApi
    public var maxTotalBufferSize: Int? = null

    /**
     * Per-resource buffer size in bytes to use when preserving network payloads (XHRs, etc).
     */
    @ExperimentalChromeApi
    public var maxResourceBufferSize: Int? = null

    /**
     * Longest post body size (in bytes) that would be included in requestWillBeSent notification
     */
    public var maxPostDataSize: Int? = null

    public fun build(): EnableRequest = EnableRequest(maxTotalBufferSize, maxResourceBufferSize,
        maxPostDataSize)
  }
}

/**
 * A dummy response object for the [NetworkDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Response type for the [NetworkDomain.getAllCookies] command.
 */
@Serializable
public data class GetAllCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>,
)

/**
 * Request object containing input parameters for the [NetworkDomain.getCertificate] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetCertificateRequest(
  /**
   * Origin to get certificate for.
   */
  public val origin: String,
)

/**
 * Response type for the [NetworkDomain.getCertificate] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetCertificateResponse(
  public val tableNames: List<String>,
)

/**
 * Request object containing input parameters for the [NetworkDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesRequest(
  /**
   * The list of URLs for which applicable cookies will be fetched.
   * If not specified, it's assumed to be set to the list containing
   * the URLs of the page and all of its subframes.
   */
  public val urls: List<String>? = null,
) {
  public class Builder() {
    /**
     * The list of URLs for which applicable cookies will be fetched.
     * If not specified, it's assumed to be set to the list containing
     * the URLs of the page and all of its subframes.
     */
    public var urls: List<String>? = null

    public fun build(): GetCookiesRequest = GetCookiesRequest(urls)
  }
}

/**
 * Response type for the [NetworkDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>,
)

/**
 * Request object containing input parameters for the [NetworkDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyRequest(
  /**
   * Identifier of the network request to get content for.
   */
  public val requestId: RequestId,
)

/**
 * Response type for the [NetworkDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyResponse(
  /**
   * Response body.
   */
  public val body: String,
  /**
   * True, if content was sent as base64.
   */
  public val base64Encoded: Boolean,
)

/**
 * Request object containing input parameters for the [NetworkDomain.getRequestPostData] command.
 */
@Serializable
public data class GetRequestPostDataRequest(
  /**
   * Identifier of the network request to get content for.
   */
  public val requestId: RequestId,
)

/**
 * Response type for the [NetworkDomain.getRequestPostData] command.
 */
@Serializable
public data class GetRequestPostDataResponse(
  /**
   * Request body string, omitting files from multipart requests
   */
  public val postData: String,
)

/**
 * Request object containing input parameters for the [NetworkDomain.getResponseBodyForInterception]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResponseBodyForInterceptionRequest(
  /**
   * Identifier for the intercepted request to get body for.
   */
  public val interceptionId: InterceptionId,
)

/**
 * Response type for the [NetworkDomain.getResponseBodyForInterception] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResponseBodyForInterceptionResponse(
  /**
   * Response body.
   */
  public val body: String,
  /**
   * True, if content was sent as base64.
   */
  public val base64Encoded: Boolean,
)

/**
 * Request object containing input parameters for the
 * [NetworkDomain.takeResponseBodyForInterceptionAsStream] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeResponseBodyForInterceptionAsStreamRequest(
  public val interceptionId: InterceptionId,
)

/**
 * Response type for the [NetworkDomain.takeResponseBodyForInterceptionAsStream] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeResponseBodyForInterceptionAsStreamResponse(
  public val stream: StreamHandle,
)

/**
 * Request object containing input parameters for the [NetworkDomain.replayXHR] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ReplayXHRRequest(
  /**
   * Identifier of XHR to replay.
   */
  public val requestId: RequestId,
)

/**
 * A dummy response object for the [NetworkDomain.replayXHR] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ReplayXHRResponse

/**
 * Request object containing input parameters for the [NetworkDomain.searchInResponseBody] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SearchInResponseBodyRequest(
  /**
   * Identifier of the network response to search.
   */
  public val requestId: RequestId,
  /**
   * String to search for.
   */
  public val query: String,
  /**
   * If true, search is case sensitive.
   */
  public val caseSensitive: Boolean? = null,
  /**
   * If true, treats string parameter as regex.
   */
  public val isRegex: Boolean? = null,
) {
  public class Builder(
    /**
     * Identifier of the network response to search.
     */
    public val requestId: RequestId,
    /**
     * String to search for.
     */
    public val query: String,
  ) {
    /**
     * If true, search is case sensitive.
     */
    public var caseSensitive: Boolean? = null

    /**
     * If true, treats string parameter as regex.
     */
    public var isRegex: Boolean? = null

    public fun build(): SearchInResponseBodyRequest = SearchInResponseBodyRequest(requestId, query,
        caseSensitive, isRegex)
  }
}

/**
 * Response type for the [NetworkDomain.searchInResponseBody] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SearchInResponseBodyResponse(
  /**
   * List of search matches.
   */
  public val result: List<SearchMatch>,
)

/**
 * Request object containing input parameters for the [NetworkDomain.setBlockedURLs] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBlockedURLsRequest(
  /**
   * URL patterns to block. Wildcards ('*') are allowed.
   */
  public val urls: List<String>,
)

/**
 * A dummy response object for the [NetworkDomain.setBlockedURLs] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetBlockedURLsResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setBypassServiceWorker]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBypassServiceWorkerRequest(
  /**
   * Bypass service worker and load from network.
   */
  public val bypass: Boolean,
)

/**
 * A dummy response object for the [NetworkDomain.setBypassServiceWorker] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetBypassServiceWorkerResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setCacheDisabled] command.
 */
@Serializable
public data class SetCacheDisabledRequest(
  /**
   * Cache disabled state.
   */
  public val cacheDisabled: Boolean,
)

/**
 * A dummy response object for the [NetworkDomain.setCacheDisabled] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetCacheDisabledResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setCookie] command.
 */
@Serializable
public data class SetCookieRequest(
  /**
   * Cookie name.
   */
  public val name: String,
  /**
   * Cookie value.
   */
  public val `value`: String,
  /**
   * The request-URI to associate with the setting of the cookie. This value can affect the
   * default domain, path, source port, and source scheme values of the created cookie.
   */
  public val url: String? = null,
  /**
   * Cookie domain.
   */
  public val domain: String? = null,
  /**
   * Cookie path.
   */
  public val path: String? = null,
  /**
   * True if cookie is secure.
   */
  public val secure: Boolean? = null,
  /**
   * True if cookie is http-only.
   */
  public val httpOnly: Boolean? = null,
  /**
   * Cookie SameSite type.
   */
  public val sameSite: CookieSameSite? = null,
  /**
   * Cookie expiration date, session cookie if not set
   */
  public val expires: TimeSinceEpoch? = null,
  /**
   * Cookie Priority type.
   */
  @ExperimentalChromeApi
  public val priority: CookiePriority? = null,
  /**
   * True if cookie is SameParty.
   */
  @ExperimentalChromeApi
  public val sameParty: Boolean? = null,
  /**
   * Cookie source scheme type.
   */
  @ExperimentalChromeApi
  public val sourceScheme: CookieSourceScheme? = null,
  /**
   * Cookie source port. Valid values are {-1, [1, 65535]}, -1 indicates an unspecified port.
   * An unspecified port value allows protocol clients to emulate legacy cookie scope for the port.
   * This is a temporary ability and it will be removed in the future.
   */
  @ExperimentalChromeApi
  public val sourcePort: Int? = null,
  /**
   * Cookie partition key. The site of the top-level URL the browser was visiting at the start
   * of the request to the endpoint that set the cookie.
   * If not set, the cookie will be set as not partitioned.
   */
  @ExperimentalChromeApi
  public val partitionKey: String? = null,
) {
  public class Builder(
    /**
     * Cookie name.
     */
    public val name: String,
    /**
     * Cookie value.
     */
    public val `value`: String,
  ) {
    /**
     * The request-URI to associate with the setting of the cookie. This value can affect the
     * default domain, path, source port, and source scheme values of the created cookie.
     */
    public var url: String? = null

    /**
     * Cookie domain.
     */
    public var domain: String? = null

    /**
     * Cookie path.
     */
    public var path: String? = null

    /**
     * True if cookie is secure.
     */
    public var secure: Boolean? = null

    /**
     * True if cookie is http-only.
     */
    public var httpOnly: Boolean? = null

    /**
     * Cookie SameSite type.
     */
    public var sameSite: CookieSameSite? = null

    /**
     * Cookie expiration date, session cookie if not set
     */
    public var expires: TimeSinceEpoch? = null

    /**
     * Cookie Priority type.
     */
    @ExperimentalChromeApi
    public var priority: CookiePriority? = null

    /**
     * True if cookie is SameParty.
     */
    @ExperimentalChromeApi
    public var sameParty: Boolean? = null

    /**
     * Cookie source scheme type.
     */
    @ExperimentalChromeApi
    public var sourceScheme: CookieSourceScheme? = null

    /**
     * Cookie source port. Valid values are {-1, [1, 65535]}, -1 indicates an unspecified port.
     * An unspecified port value allows protocol clients to emulate legacy cookie scope for the
     * port.
     * This is a temporary ability and it will be removed in the future.
     */
    @ExperimentalChromeApi
    public var sourcePort: Int? = null

    /**
     * Cookie partition key. The site of the top-level URL the browser was visiting at the start
     * of the request to the endpoint that set the cookie.
     * If not set, the cookie will be set as not partitioned.
     */
    @ExperimentalChromeApi
    public var partitionKey: String? = null

    public fun build(): SetCookieRequest = SetCookieRequest(name, value, url, domain, path, secure,
        httpOnly, sameSite, expires, priority, sameParty, sourceScheme, sourcePort, partitionKey)
  }
}

/**
 * Response type for the [NetworkDomain.setCookie] command.
 */
@Serializable
public data class SetCookieResponse(
  /**
   * Always set to true. If an error occurs, the response indicates protocol error.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val success: Boolean,
)

/**
 * Request object containing input parameters for the [NetworkDomain.setCookies] command.
 */
@Serializable
public data class SetCookiesRequest(
  /**
   * Cookies to be set.
   */
  public val cookies: List<CookieParam>,
)

/**
 * A dummy response object for the [NetworkDomain.setCookies] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object SetCookiesResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setExtraHTTPHeaders] command.
 */
@Serializable
public data class SetExtraHTTPHeadersRequest(
  /**
   * Map with extra HTTP headers.
   */
  public val headers: Headers,
)

/**
 * A dummy response object for the [NetworkDomain.setExtraHTTPHeaders] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetExtraHTTPHeadersResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setAttachDebugStack] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAttachDebugStackRequest(
  /**
   * Whether to attach a page script stack for debugging purpose.
   */
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [NetworkDomain.setAttachDebugStack] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetAttachDebugStackResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setRequestInterception]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetRequestInterceptionRequest(
  /**
   * Requests matching any of these patterns will be forwarded and wait for the corresponding
   * continueInterceptedRequest call.
   */
  public val patterns: List<RequestPattern>,
)

/**
 * A dummy response object for the [NetworkDomain.setRequestInterception] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public object SetRequestInterceptionResponse

/**
 * Request object containing input parameters for the [NetworkDomain.setUserAgentOverride] command.
 */
@Serializable
public data class SetUserAgentOverrideRequest(
  /**
   * User agent to use.
   */
  public val userAgent: String,
  /**
   * Browser langugage to emulate.
   */
  public val acceptLanguage: String? = null,
  /**
   * The platform navigator.platform should return.
   */
  public val platform: String? = null,
  /**
   * To be sent in Sec-CH-UA-* headers and returned in navigator.userAgentData
   */
  @ExperimentalChromeApi
  public val userAgentMetadata: UserAgentMetadata? = null,
) {
  public class Builder(
    /**
     * User agent to use.
     */
    public val userAgent: String,
  ) {
    /**
     * Browser langugage to emulate.
     */
    public var acceptLanguage: String? = null

    /**
     * The platform navigator.platform should return.
     */
    public var platform: String? = null

    /**
     * To be sent in Sec-CH-UA-* headers and returned in navigator.userAgentData
     */
    @ExperimentalChromeApi
    public var userAgentMetadata: UserAgentMetadata? = null

    public fun build(): SetUserAgentOverrideRequest = SetUserAgentOverrideRequest(userAgent,
        acceptLanguage, platform, userAgentMetadata)
  }
}

/**
 * A dummy response object for the [NetworkDomain.setUserAgentOverride] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetUserAgentOverrideResponse

/**
 * Request object containing input parameters for the [NetworkDomain.getSecurityIsolationStatus]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSecurityIsolationStatusRequest(
  /**
   * If no frameId is provided, the status of the target is provided.
   */
  public val frameId: FrameId? = null,
) {
  public class Builder() {
    /**
     * If no frameId is provided, the status of the target is provided.
     */
    public var frameId: FrameId? = null

    public fun build(): GetSecurityIsolationStatusRequest =
        GetSecurityIsolationStatusRequest(frameId)
  }
}

/**
 * Response type for the [NetworkDomain.getSecurityIsolationStatus] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSecurityIsolationStatusResponse(
  public val status: SecurityIsolationStatus,
)

/**
 * Request object containing input parameters for the [NetworkDomain.enableReportingApi] command.
 */
@Serializable
@ExperimentalChromeApi
public data class EnableReportingApiRequest(
  /**
   * Whether to enable or disable events for the Reporting API
   */
  public val enable: Boolean,
)

/**
 * A dummy response object for the [NetworkDomain.enableReportingApi] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object EnableReportingApiResponse

/**
 * Request object containing input parameters for the [NetworkDomain.loadNetworkResource] command.
 */
@Serializable
@ExperimentalChromeApi
public data class LoadNetworkResourceRequest(
  /**
   * Frame id to get the resource for. Mandatory for frame targets, and
   * should be omitted for worker targets.
   */
  public val frameId: FrameId? = null,
  /**
   * URL of the resource to get content for.
   */
  public val url: String,
  /**
   * Options for the request.
   */
  public val options: LoadNetworkResourceOptions,
) {
  public class Builder(
    /**
     * URL of the resource to get content for.
     */
    public val url: String,
    /**
     * Options for the request.
     */
    public val options: LoadNetworkResourceOptions,
  ) {
    /**
     * Frame id to get the resource for. Mandatory for frame targets, and
     * should be omitted for worker targets.
     */
    public var frameId: FrameId? = null

    public fun build(): LoadNetworkResourceRequest = LoadNetworkResourceRequest(frameId, url,
        options)
  }
}

/**
 * Response type for the [NetworkDomain.loadNetworkResource] command.
 */
@Serializable
@ExperimentalChromeApi
public data class LoadNetworkResourceResponse(
  public val resource: LoadNetworkResourcePageResult,
)

/**
 * Network domain allows tracking network activities of the page. It exposes information about http,
 * file, data and other requests and responses, their headers, bodies, timing, etc.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network)
 */
public class NetworkDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out NetworkEvent>> =
      mapOf(
      "Network.dataReceived" to serializer<NetworkEvent.DataReceived>(),
      "Network.eventSourceMessageReceived" to serializer<NetworkEvent.EventSourceMessageReceived>(),
      "Network.loadingFailed" to serializer<NetworkEvent.LoadingFailed>(),
      "Network.loadingFinished" to serializer<NetworkEvent.LoadingFinished>(),
      "Network.requestIntercepted" to serializer<NetworkEvent.RequestIntercepted>(),
      "Network.requestServedFromCache" to serializer<NetworkEvent.RequestServedFromCache>(),
      "Network.requestWillBeSent" to serializer<NetworkEvent.RequestWillBeSent>(),
      "Network.resourceChangedPriority" to serializer<NetworkEvent.ResourceChangedPriority>(),
      "Network.signedExchangeReceived" to serializer<NetworkEvent.SignedExchangeReceived>(),
      "Network.responseReceived" to serializer<NetworkEvent.ResponseReceived>(),
      "Network.webSocketClosed" to serializer<NetworkEvent.WebSocketClosed>(),
      "Network.webSocketCreated" to serializer<NetworkEvent.WebSocketCreated>(),
      "Network.webSocketFrameError" to serializer<NetworkEvent.WebSocketFrameError>(),
      "Network.webSocketFrameReceived" to serializer<NetworkEvent.WebSocketFrameReceived>(),
      "Network.webSocketFrameSent" to serializer<NetworkEvent.WebSocketFrameSent>(),
      "Network.webSocketHandshakeResponseReceived" to
          serializer<NetworkEvent.WebSocketHandshakeResponseReceived>(),
      "Network.webSocketWillSendHandshakeRequest" to
          serializer<NetworkEvent.WebSocketWillSendHandshakeRequest>(),
      "Network.webTransportCreated" to serializer<NetworkEvent.WebTransportCreated>(),
      "Network.webTransportConnectionEstablished" to
          serializer<NetworkEvent.WebTransportConnectionEstablished>(),
      "Network.webTransportClosed" to serializer<NetworkEvent.WebTransportClosed>(),
      "Network.requestWillBeSentExtraInfo" to serializer<NetworkEvent.RequestWillBeSentExtraInfo>(),
      "Network.responseReceivedExtraInfo" to serializer<NetworkEvent.ResponseReceivedExtraInfo>(),
      "Network.trustTokenOperationDone" to serializer<NetworkEvent.TrustTokenOperationDone>(),
      "Network.subresourceWebBundleMetadataReceived" to
          serializer<NetworkEvent.SubresourceWebBundleMetadataReceived>(),
      "Network.subresourceWebBundleMetadataError" to
          serializer<NetworkEvent.SubresourceWebBundleMetadataError>(),
      "Network.subresourceWebBundleInnerResponseParsed" to
          serializer<NetworkEvent.SubresourceWebBundleInnerResponseParsed>(),
      "Network.subresourceWebBundleInnerResponseError" to
          serializer<NetworkEvent.SubresourceWebBundleInnerResponseError>(),
      "Network.reportingApiReportAdded" to serializer<NetworkEvent.ReportingApiReportAdded>(),
      "Network.reportingApiReportUpdated" to serializer<NetworkEvent.ReportingApiReportUpdated>(),
      "Network.reportingApiEndpointsChangedForOrigin" to
          serializer<NetworkEvent.ReportingApiEndpointsChangedForOrigin>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<NetworkEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fired when data chunk was received over the network.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-dataReceived)
   */
  public fun dataReceivedEvents(): Flow<NetworkEvent.DataReceived> =
      session.typedEvents("Network.dataReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("dataReceivedEvents()"),
  )
  public fun dataReceived(): Flow<NetworkEvent.DataReceived> = dataReceivedEvents()

  /**
   * Fired when EventSource message is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-eventSourceMessageReceived)
   */
  public fun eventSourceMessageReceivedEvents(): Flow<NetworkEvent.EventSourceMessageReceived> =
      session.typedEvents("Network.eventSourceMessageReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("eventSourceMessageReceivedEvents()"),
  )
  public fun eventSourceMessageReceived(): Flow<NetworkEvent.EventSourceMessageReceived> =
      eventSourceMessageReceivedEvents()

  /**
   * Fired when HTTP request has failed to load.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-loadingFailed)
   */
  public fun loadingFailedEvents(): Flow<NetworkEvent.LoadingFailed> =
      session.typedEvents("Network.loadingFailed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("loadingFailedEvents()"),
  )
  public fun loadingFailed(): Flow<NetworkEvent.LoadingFailed> = loadingFailedEvents()

  /**
   * Fired when HTTP request has finished loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-loadingFinished)
   */
  public fun loadingFinishedEvents(): Flow<NetworkEvent.LoadingFinished> =
      session.typedEvents("Network.loadingFinished")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("loadingFinishedEvents()"),
  )
  public fun loadingFinished(): Flow<NetworkEvent.LoadingFinished> = loadingFinishedEvents()

  /**
   * Details of an intercepted HTTP request, which must be either allowed, blocked, modified or
   * mocked.
   * Deprecated, use Fetch.requestPaused instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestIntercepted)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public fun requestInterceptedEvents(): Flow<NetworkEvent.RequestIntercepted> =
      session.typedEvents("Network.requestIntercepted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("requestInterceptedEvents()"),
  )
  public fun requestIntercepted(): Flow<NetworkEvent.RequestIntercepted> =
      requestInterceptedEvents()

  /**
   * Fired if request ended up loading from cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestServedFromCache)
   */
  public fun requestServedFromCacheEvents(): Flow<NetworkEvent.RequestServedFromCache> =
      session.typedEvents("Network.requestServedFromCache")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("requestServedFromCacheEvents()"),
  )
  public fun requestServedFromCache(): Flow<NetworkEvent.RequestServedFromCache> =
      requestServedFromCacheEvents()

  /**
   * Fired when page is about to send HTTP request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestWillBeSent)
   */
  public fun requestWillBeSentEvents(): Flow<NetworkEvent.RequestWillBeSent> =
      session.typedEvents("Network.requestWillBeSent")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("requestWillBeSentEvents()"),
  )
  public fun requestWillBeSent(): Flow<NetworkEvent.RequestWillBeSent> = requestWillBeSentEvents()

  /**
   * Fired when resource loading priority is changed
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-resourceChangedPriority)
   */
  @ExperimentalChromeApi
  public fun resourceChangedPriorityEvents(): Flow<NetworkEvent.ResourceChangedPriority> =
      session.typedEvents("Network.resourceChangedPriority")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("resourceChangedPriorityEvents()"),
  )
  public fun resourceChangedPriority(): Flow<NetworkEvent.ResourceChangedPriority> =
      resourceChangedPriorityEvents()

  /**
   * Fired when a signed exchange was received over the network
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-signedExchangeReceived)
   */
  @ExperimentalChromeApi
  public fun signedExchangeReceivedEvents(): Flow<NetworkEvent.SignedExchangeReceived> =
      session.typedEvents("Network.signedExchangeReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("signedExchangeReceivedEvents()"),
  )
  public fun signedExchangeReceived(): Flow<NetworkEvent.SignedExchangeReceived> =
      signedExchangeReceivedEvents()

  /**
   * Fired when HTTP response is available.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-responseReceived)
   */
  public fun responseReceivedEvents(): Flow<NetworkEvent.ResponseReceived> =
      session.typedEvents("Network.responseReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("responseReceivedEvents()"),
  )
  public fun responseReceived(): Flow<NetworkEvent.ResponseReceived> = responseReceivedEvents()

  /**
   * Fired when WebSocket is closed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketClosed)
   */
  public fun webSocketClosedEvents(): Flow<NetworkEvent.WebSocketClosed> =
      session.typedEvents("Network.webSocketClosed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketClosedEvents()"),
  )
  public fun webSocketClosed(): Flow<NetworkEvent.WebSocketClosed> = webSocketClosedEvents()

  /**
   * Fired upon WebSocket creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketCreated)
   */
  public fun webSocketCreatedEvents(): Flow<NetworkEvent.WebSocketCreated> =
      session.typedEvents("Network.webSocketCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketCreatedEvents()"),
  )
  public fun webSocketCreated(): Flow<NetworkEvent.WebSocketCreated> = webSocketCreatedEvents()

  /**
   * Fired when WebSocket message error occurs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketFrameError)
   */
  public fun webSocketFrameErrorEvents(): Flow<NetworkEvent.WebSocketFrameError> =
      session.typedEvents("Network.webSocketFrameError")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketFrameErrorEvents()"),
  )
  public fun webSocketFrameError(): Flow<NetworkEvent.WebSocketFrameError> =
      webSocketFrameErrorEvents()

  /**
   * Fired when WebSocket message is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketFrameReceived)
   */
  public fun webSocketFrameReceivedEvents(): Flow<NetworkEvent.WebSocketFrameReceived> =
      session.typedEvents("Network.webSocketFrameReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketFrameReceivedEvents()"),
  )
  public fun webSocketFrameReceived(): Flow<NetworkEvent.WebSocketFrameReceived> =
      webSocketFrameReceivedEvents()

  /**
   * Fired when WebSocket message is sent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketFrameSent)
   */
  public fun webSocketFrameSentEvents(): Flow<NetworkEvent.WebSocketFrameSent> =
      session.typedEvents("Network.webSocketFrameSent")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketFrameSentEvents()"),
  )
  public fun webSocketFrameSent(): Flow<NetworkEvent.WebSocketFrameSent> =
      webSocketFrameSentEvents()

  /**
   * Fired when WebSocket handshake response becomes available.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketHandshakeResponseReceived)
   */
  public fun webSocketHandshakeResponseReceivedEvents():
      Flow<NetworkEvent.WebSocketHandshakeResponseReceived> =
      session.typedEvents("Network.webSocketHandshakeResponseReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketHandshakeResponseReceivedEvents()"),
  )
  public fun webSocketHandshakeResponseReceived():
      Flow<NetworkEvent.WebSocketHandshakeResponseReceived> =
      webSocketHandshakeResponseReceivedEvents()

  /**
   * Fired when WebSocket is about to initiate handshake.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketWillSendHandshakeRequest)
   */
  public fun webSocketWillSendHandshakeRequestEvents():
      Flow<NetworkEvent.WebSocketWillSendHandshakeRequest> =
      session.typedEvents("Network.webSocketWillSendHandshakeRequest")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webSocketWillSendHandshakeRequestEvents()"),
  )
  public fun webSocketWillSendHandshakeRequest():
      Flow<NetworkEvent.WebSocketWillSendHandshakeRequest> =
      webSocketWillSendHandshakeRequestEvents()

  /**
   * Fired upon WebTransport creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webTransportCreated)
   */
  public fun webTransportCreatedEvents(): Flow<NetworkEvent.WebTransportCreated> =
      session.typedEvents("Network.webTransportCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webTransportCreatedEvents()"),
  )
  public fun webTransportCreated(): Flow<NetworkEvent.WebTransportCreated> =
      webTransportCreatedEvents()

  /**
   * Fired when WebTransport handshake is finished.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webTransportConnectionEstablished)
   */
  public fun webTransportConnectionEstablishedEvents():
      Flow<NetworkEvent.WebTransportConnectionEstablished> =
      session.typedEvents("Network.webTransportConnectionEstablished")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webTransportConnectionEstablishedEvents()"),
  )
  public fun webTransportConnectionEstablished():
      Flow<NetworkEvent.WebTransportConnectionEstablished> =
      webTransportConnectionEstablishedEvents()

  /**
   * Fired when WebTransport is disposed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webTransportClosed)
   */
  public fun webTransportClosedEvents(): Flow<NetworkEvent.WebTransportClosed> =
      session.typedEvents("Network.webTransportClosed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("webTransportClosedEvents()"),
  )
  public fun webTransportClosed(): Flow<NetworkEvent.WebTransportClosed> =
      webTransportClosedEvents()

  /**
   * Fired when additional information about a requestWillBeSent event is available from the
   * network stack. Not every requestWillBeSent event will have an additional
   * requestWillBeSentExtraInfo fired for it, and there is no guarantee whether requestWillBeSent
   * or requestWillBeSentExtraInfo will be fired first for the same request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestWillBeSentExtraInfo)
   */
  @ExperimentalChromeApi
  public fun requestWillBeSentExtraInfoEvents(): Flow<NetworkEvent.RequestWillBeSentExtraInfo> =
      session.typedEvents("Network.requestWillBeSentExtraInfo")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("requestWillBeSentExtraInfoEvents()"),
  )
  public fun requestWillBeSentExtraInfo(): Flow<NetworkEvent.RequestWillBeSentExtraInfo> =
      requestWillBeSentExtraInfoEvents()

  /**
   * Fired when additional information about a responseReceived event is available from the network
   * stack. Not every responseReceived event will have an additional responseReceivedExtraInfo for
   * it, and responseReceivedExtraInfo may be fired before or after responseReceived.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-responseReceivedExtraInfo)
   */
  @ExperimentalChromeApi
  public fun responseReceivedExtraInfoEvents(): Flow<NetworkEvent.ResponseReceivedExtraInfo> =
      session.typedEvents("Network.responseReceivedExtraInfo")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("responseReceivedExtraInfoEvents()"),
  )
  public fun responseReceivedExtraInfo(): Flow<NetworkEvent.ResponseReceivedExtraInfo> =
      responseReceivedExtraInfoEvents()

  /**
   * Fired exactly once for each Trust Token operation. Depending on
   * the type of the operation and whether the operation succeeded or
   * failed, the event is fired before the corresponding request was sent
   * or after the response was received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-trustTokenOperationDone)
   */
  @ExperimentalChromeApi
  public fun trustTokenOperationDoneEvents(): Flow<NetworkEvent.TrustTokenOperationDone> =
      session.typedEvents("Network.trustTokenOperationDone")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("trustTokenOperationDoneEvents()"),
  )
  public fun trustTokenOperationDone(): Flow<NetworkEvent.TrustTokenOperationDone> =
      trustTokenOperationDoneEvents()

  /**
   * Fired once when parsing the .wbn file has succeeded.
   * The event contains the information about the web bundle contents.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleMetadataReceived)
   */
  @ExperimentalChromeApi
  public fun subresourceWebBundleMetadataReceivedEvents():
      Flow<NetworkEvent.SubresourceWebBundleMetadataReceived> =
      session.typedEvents("Network.subresourceWebBundleMetadataReceived")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("subresourceWebBundleMetadataReceivedEvents()"),
  )
  public fun subresourceWebBundleMetadataReceived():
      Flow<NetworkEvent.SubresourceWebBundleMetadataReceived> =
      subresourceWebBundleMetadataReceivedEvents()

  /**
   * Fired once when parsing the .wbn file has failed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleMetadataError)
   */
  @ExperimentalChromeApi
  public fun subresourceWebBundleMetadataErrorEvents():
      Flow<NetworkEvent.SubresourceWebBundleMetadataError> =
      session.typedEvents("Network.subresourceWebBundleMetadataError")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("subresourceWebBundleMetadataErrorEvents()"),
  )
  public fun subresourceWebBundleMetadataError():
      Flow<NetworkEvent.SubresourceWebBundleMetadataError> =
      subresourceWebBundleMetadataErrorEvents()

  /**
   * Fired when handling requests for resources within a .wbn file.
   * Note: this will only be fired for resources that are requested by the webpage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleInnerResponseParsed)
   */
  @ExperimentalChromeApi
  public fun subresourceWebBundleInnerResponseParsedEvents():
      Flow<NetworkEvent.SubresourceWebBundleInnerResponseParsed> =
      session.typedEvents("Network.subresourceWebBundleInnerResponseParsed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("subresourceWebBundleInnerResponseParsedEvents()"),
  )
  public fun subresourceWebBundleInnerResponseParsed():
      Flow<NetworkEvent.SubresourceWebBundleInnerResponseParsed> =
      subresourceWebBundleInnerResponseParsedEvents()

  /**
   * Fired when request for resources within a .wbn file failed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleInnerResponseError)
   */
  @ExperimentalChromeApi
  public fun subresourceWebBundleInnerResponseErrorEvents():
      Flow<NetworkEvent.SubresourceWebBundleInnerResponseError> =
      session.typedEvents("Network.subresourceWebBundleInnerResponseError")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("subresourceWebBundleInnerResponseErrorEvents()"),
  )
  public fun subresourceWebBundleInnerResponseError():
      Flow<NetworkEvent.SubresourceWebBundleInnerResponseError> =
      subresourceWebBundleInnerResponseErrorEvents()

  /**
   * Is sent whenever a new report is added.
   * And after 'enableReportingApi' for all existing reports.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-reportingApiReportAdded)
   */
  @ExperimentalChromeApi
  public fun reportingApiReportAddedEvents(): Flow<NetworkEvent.ReportingApiReportAdded> =
      session.typedEvents("Network.reportingApiReportAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("reportingApiReportAddedEvents()"),
  )
  public fun reportingApiReportAdded(): Flow<NetworkEvent.ReportingApiReportAdded> =
      reportingApiReportAddedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-reportingApiReportUpdated)
   */
  @ExperimentalChromeApi
  public fun reportingApiReportUpdatedEvents(): Flow<NetworkEvent.ReportingApiReportUpdated> =
      session.typedEvents("Network.reportingApiReportUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("reportingApiReportUpdatedEvents()"),
  )
  public fun reportingApiReportUpdated(): Flow<NetworkEvent.ReportingApiReportUpdated> =
      reportingApiReportUpdatedEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-reportingApiEndpointsChangedForOrigin)
   */
  @ExperimentalChromeApi
  public fun reportingApiEndpointsChangedForOriginEvents():
      Flow<NetworkEvent.ReportingApiEndpointsChangedForOrigin> =
      session.typedEvents("Network.reportingApiEndpointsChangedForOrigin")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("reportingApiEndpointsChangedForOriginEvents()"),
  )
  public fun reportingApiEndpointsChangedForOrigin():
      Flow<NetworkEvent.ReportingApiEndpointsChangedForOrigin> =
      reportingApiEndpointsChangedForOriginEvents()

  /**
   * Sets a list of content encodings that will be accepted. Empty list means no encoding is
   * accepted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setAcceptedEncodings)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setAcceptedEncodings(input: SetAcceptedEncodingsRequest):
      SetAcceptedEncodingsResponse = session.request("Network.setAcceptedEncodings", input)

  /**
   * Sets a list of content encodings that will be accepted. Empty list means no encoding is
   * accepted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setAcceptedEncodings)
   *
   * @param encodings List of accepted content encodings.
   */
  @ExperimentalChromeApi
  public suspend fun setAcceptedEncodings(encodings: List<ContentEncoding>):
      SetAcceptedEncodingsResponse {
    val input = SetAcceptedEncodingsRequest(encodings)
    return setAcceptedEncodings(input)
  }

  /**
   * Clears accepted encodings set by setAcceptedEncodings
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-clearAcceptedEncodingsOverride)
   */
  @ExperimentalChromeApi
  public suspend fun clearAcceptedEncodingsOverride(): ClearAcceptedEncodingsOverrideResponse =
      session.request("Network.clearAcceptedEncodingsOverride", Unit)

  /**
   * Tells whether clearing browser cache is supported.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-canClearBrowserCache)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun canClearBrowserCache(): CanClearBrowserCacheResponse =
      session.request("Network.canClearBrowserCache", Unit)

  /**
   * Tells whether clearing browser cookies is supported.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-canClearBrowserCookies)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun canClearBrowserCookies(): CanClearBrowserCookiesResponse =
      session.request("Network.canClearBrowserCookies", Unit)

  /**
   * Tells whether emulation of network conditions is supported.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-canEmulateNetworkConditions)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun canEmulateNetworkConditions(): CanEmulateNetworkConditionsResponse =
      session.request("Network.canEmulateNetworkConditions", Unit)

  /**
   * Clears browser cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-clearBrowserCache)
   */
  public suspend fun clearBrowserCache(): ClearBrowserCacheResponse =
      session.request("Network.clearBrowserCache", Unit)

  /**
   * Clears browser cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-clearBrowserCookies)
   */
  public suspend fun clearBrowserCookies(): ClearBrowserCookiesResponse =
      session.request("Network.clearBrowserCookies", Unit)

  /**
   * Response to Network.requestIntercepted which either modifies the request to continue with any
   * modifications, or blocks it, or completes it with the provided response bytes. If a network
   * fetch occurs as a result which encounters a redirect an additional Network.requestIntercepted
   * event will be sent with the same InterceptionId.
   * Deprecated, use Fetch.continueRequest, Fetch.fulfillRequest and Fetch.failRequest instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-continueInterceptedRequest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun continueInterceptedRequest(input: ContinueInterceptedRequestRequest):
      ContinueInterceptedRequestResponse = session.request("Network.continueInterceptedRequest",
      input)

  /**
   * Response to Network.requestIntercepted which either modifies the request to continue with any
   * modifications, or blocks it, or completes it with the provided response bytes. If a network
   * fetch occurs as a result which encounters a redirect an additional Network.requestIntercepted
   * event will be sent with the same InterceptionId.
   * Deprecated, use Fetch.continueRequest, Fetch.fulfillRequest and Fetch.failRequest instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-continueInterceptedRequest)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun continueInterceptedRequest(interceptionId: InterceptionId,
      optionalArgs: ContinueInterceptedRequestRequest.Builder.() -> Unit = {}):
      ContinueInterceptedRequestResponse {
    val builder = ContinueInterceptedRequestRequest.Builder(interceptionId)
    val input = builder.apply(optionalArgs).build()
    return continueInterceptedRequest(input)
  }

  /**
   * Deletes browser cookies with matching name and url or domain/path pair.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-deleteCookies)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun deleteCookies(input: DeleteCookiesRequest): DeleteCookiesResponse =
      session.request("Network.deleteCookies", input)

  /**
   * Deletes browser cookies with matching name and url or domain/path pair.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-deleteCookies)
   *
   * @param name Name of the cookies to remove.
   */
  @JvmOverloads
  public suspend inline fun deleteCookies(name: String,
      optionalArgs: DeleteCookiesRequest.Builder.() -> Unit = {}): DeleteCookiesResponse {
    val builder = DeleteCookiesRequest.Builder(name)
    val input = builder.apply(optionalArgs).build()
    return deleteCookies(input)
  }

  /**
   * Disables network tracking, prevents network events from being sent to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Network.disable", Unit)

  /**
   * Activates emulation of network conditions.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-emulateNetworkConditions)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun emulateNetworkConditions(input: EmulateNetworkConditionsRequest):
      EmulateNetworkConditionsResponse = session.request("Network.emulateNetworkConditions", input)

  /**
   * Activates emulation of network conditions.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-emulateNetworkConditions)
   *
   * @param offline True to emulate internet disconnection.
   * @param latency Minimum latency from request sent to response headers received (ms).
   * @param downloadThroughput Maximal aggregated download throughput (bytes/sec). -1 disables
   * download throttling.
   * @param uploadThroughput Maximal aggregated upload throughput (bytes/sec).  -1 disables upload
   * throttling.
   */
  @JvmOverloads
  public suspend inline fun emulateNetworkConditions(
    offline: Boolean,
    latency: Double,
    downloadThroughput: Double,
    uploadThroughput: Double,
    optionalArgs: EmulateNetworkConditionsRequest.Builder.() -> Unit = {},
  ): EmulateNetworkConditionsResponse {
    val builder = EmulateNetworkConditionsRequest.Builder(offline, latency, downloadThroughput,
        uploadThroughput)
    val input = builder.apply(optionalArgs).build()
    return emulateNetworkConditions(input)
  }

  /**
   * Enables network tracking, network events will now be delivered to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse =
      session.request("Network.enable", input)

  /**
   * Enables network tracking, network events will now be delivered to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Returns all browser cookies. Depending on the backend support, will return detailed cookie
   * information in the `cookies` field.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getAllCookies)
   */
  public suspend fun getAllCookies(): GetAllCookiesResponse =
      session.request("Network.getAllCookies", Unit)

  /**
   * Returns the DER-encoded certificate.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getCertificate)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getCertificate(input: GetCertificateRequest): GetCertificateResponse =
      session.request("Network.getCertificate", input)

  /**
   * Returns the DER-encoded certificate.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getCertificate)
   *
   * @param origin Origin to get certificate for.
   */
  @ExperimentalChromeApi
  public suspend fun getCertificate(origin: String): GetCertificateResponse {
    val input = GetCertificateRequest(origin)
    return getCertificate(input)
  }

  /**
   * Returns all browser cookies for the current URL. Depending on the backend support, will return
   * detailed cookie information in the `cookies` field.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getCookies)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getCookies(input: GetCookiesRequest): GetCookiesResponse =
      session.request("Network.getCookies", input)

  /**
   * Returns all browser cookies for the current URL. Depending on the backend support, will return
   * detailed cookie information in the `cookies` field.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getCookies)
   */
  @JvmOverloads
  public suspend inline fun getCookies(optionalArgs: GetCookiesRequest.Builder.() -> Unit = {}):
      GetCookiesResponse {
    val builder = GetCookiesRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getCookies(input)
  }

  /**
   * Returns content served for the given request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getResponseBody)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getResponseBody(input: GetResponseBodyRequest): GetResponseBodyResponse =
      session.request("Network.getResponseBody", input)

  /**
   * Returns content served for the given request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getResponseBody)
   *
   * @param requestId Identifier of the network request to get content for.
   */
  public suspend fun getResponseBody(requestId: RequestId): GetResponseBodyResponse {
    val input = GetResponseBodyRequest(requestId)
    return getResponseBody(input)
  }

  /**
   * Returns post data sent with the request. Returns an error when no data was sent with the
   * request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getRequestPostData)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getRequestPostData(input: GetRequestPostDataRequest):
      GetRequestPostDataResponse = session.request("Network.getRequestPostData", input)

  /**
   * Returns post data sent with the request. Returns an error when no data was sent with the
   * request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getRequestPostData)
   *
   * @param requestId Identifier of the network request to get content for.
   */
  public suspend fun getRequestPostData(requestId: RequestId): GetRequestPostDataResponse {
    val input = GetRequestPostDataRequest(requestId)
    return getRequestPostData(input)
  }

  /**
   * Returns content served for the given currently intercepted request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getResponseBodyForInterception)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getResponseBodyForInterception(input: GetResponseBodyForInterceptionRequest):
      GetResponseBodyForInterceptionResponse =
      session.request("Network.getResponseBodyForInterception", input)

  /**
   * Returns content served for the given currently intercepted request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getResponseBodyForInterception)
   *
   * @param interceptionId Identifier for the intercepted request to get body for.
   */
  @ExperimentalChromeApi
  public suspend fun getResponseBodyForInterception(interceptionId: InterceptionId):
      GetResponseBodyForInterceptionResponse {
    val input = GetResponseBodyForInterceptionRequest(interceptionId)
    return getResponseBodyForInterception(input)
  }

  /**
   * Returns a handle to the stream representing the response body. Note that after this command,
   * the intercepted request can't be continued as is -- you either need to cancel it or to provide
   * the response body. The stream only supports sequential read, IO.read will fail if the position
   * is specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-takeResponseBodyForInterceptionAsStream)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend
      fun takeResponseBodyForInterceptionAsStream(input: TakeResponseBodyForInterceptionAsStreamRequest):
      TakeResponseBodyForInterceptionAsStreamResponse =
      session.request("Network.takeResponseBodyForInterceptionAsStream", input)

  /**
   * Returns a handle to the stream representing the response body. Note that after this command,
   * the intercepted request can't be continued as is -- you either need to cancel it or to provide
   * the response body. The stream only supports sequential read, IO.read will fail if the position
   * is specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-takeResponseBodyForInterceptionAsStream)
   */
  @ExperimentalChromeApi
  public suspend fun takeResponseBodyForInterceptionAsStream(interceptionId: InterceptionId):
      TakeResponseBodyForInterceptionAsStreamResponse {
    val input = TakeResponseBodyForInterceptionAsStreamRequest(interceptionId)
    return takeResponseBodyForInterceptionAsStream(input)
  }

  /**
   * This method sends a new XMLHttpRequest which is identical to the original one. The following
   * parameters should be identical: method, url, async, request body, extra headers,
   * withCredentials
   * attribute, user, password.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-replayXHR)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun replayXHR(input: ReplayXHRRequest): ReplayXHRResponse =
      session.request("Network.replayXHR", input)

  /**
   * This method sends a new XMLHttpRequest which is identical to the original one. The following
   * parameters should be identical: method, url, async, request body, extra headers,
   * withCredentials
   * attribute, user, password.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-replayXHR)
   *
   * @param requestId Identifier of XHR to replay.
   */
  @ExperimentalChromeApi
  public suspend fun replayXHR(requestId: RequestId): ReplayXHRResponse {
    val input = ReplayXHRRequest(requestId)
    return replayXHR(input)
  }

  /**
   * Searches for given string in response content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-searchInResponseBody)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun searchInResponseBody(input: SearchInResponseBodyRequest):
      SearchInResponseBodyResponse = session.request("Network.searchInResponseBody", input)

  /**
   * Searches for given string in response content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-searchInResponseBody)
   *
   * @param requestId Identifier of the network response to search.
   * @param query String to search for.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun searchInResponseBody(
    requestId: RequestId,
    query: String,
    optionalArgs: SearchInResponseBodyRequest.Builder.() -> Unit = {},
  ): SearchInResponseBodyResponse {
    val builder = SearchInResponseBodyRequest.Builder(requestId, query)
    val input = builder.apply(optionalArgs).build()
    return searchInResponseBody(input)
  }

  /**
   * Blocks URLs from loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setBlockedURLs)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setBlockedURLs(input: SetBlockedURLsRequest): SetBlockedURLsResponse =
      session.request("Network.setBlockedURLs", input)

  /**
   * Blocks URLs from loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setBlockedURLs)
   *
   * @param urls URL patterns to block. Wildcards ('*') are allowed.
   */
  @ExperimentalChromeApi
  public suspend fun setBlockedURLs(urls: List<String>): SetBlockedURLsResponse {
    val input = SetBlockedURLsRequest(urls)
    return setBlockedURLs(input)
  }

  /**
   * Toggles ignoring of service worker for each request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setBypassServiceWorker)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setBypassServiceWorker(input: SetBypassServiceWorkerRequest):
      SetBypassServiceWorkerResponse = session.request("Network.setBypassServiceWorker", input)

  /**
   * Toggles ignoring of service worker for each request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setBypassServiceWorker)
   *
   * @param bypass Bypass service worker and load from network.
   */
  @ExperimentalChromeApi
  public suspend fun setBypassServiceWorker(bypass: Boolean): SetBypassServiceWorkerResponse {
    val input = SetBypassServiceWorkerRequest(bypass)
    return setBypassServiceWorker(input)
  }

  /**
   * Toggles ignoring cache for each request. If `true`, cache will not be used.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setCacheDisabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setCacheDisabled(input: SetCacheDisabledRequest): SetCacheDisabledResponse =
      session.request("Network.setCacheDisabled", input)

  /**
   * Toggles ignoring cache for each request. If `true`, cache will not be used.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setCacheDisabled)
   *
   * @param cacheDisabled Cache disabled state.
   */
  public suspend fun setCacheDisabled(cacheDisabled: Boolean): SetCacheDisabledResponse {
    val input = SetCacheDisabledRequest(cacheDisabled)
    return setCacheDisabled(input)
  }

  /**
   * Sets a cookie with the given cookie data; may overwrite equivalent cookies if they exist.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setCookie)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setCookie(input: SetCookieRequest): SetCookieResponse =
      session.request("Network.setCookie", input)

  /**
   * Sets a cookie with the given cookie data; may overwrite equivalent cookies if they exist.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setCookie)
   *
   * @param name Cookie name.
   * @param value Cookie value.
   */
  @JvmOverloads
  public suspend inline fun setCookie(
    name: String,
    `value`: String,
    optionalArgs: SetCookieRequest.Builder.() -> Unit = {},
  ): SetCookieResponse {
    val builder = SetCookieRequest.Builder(name, value)
    val input = builder.apply(optionalArgs).build()
    return setCookie(input)
  }

  /**
   * Sets given cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setCookies)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setCookies(input: SetCookiesRequest): SetCookiesResponse =
      session.request("Network.setCookies", input)

  /**
   * Sets given cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setCookies)
   *
   * @param cookies Cookies to be set.
   */
  public suspend fun setCookies(cookies: List<CookieParam>): SetCookiesResponse {
    val input = SetCookiesRequest(cookies)
    return setCookies(input)
  }

  /**
   * Specifies whether to always send extra HTTP headers with the requests from this page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setExtraHTTPHeaders)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setExtraHTTPHeaders(input: SetExtraHTTPHeadersRequest):
      SetExtraHTTPHeadersResponse = session.request("Network.setExtraHTTPHeaders", input)

  /**
   * Specifies whether to always send extra HTTP headers with the requests from this page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setExtraHTTPHeaders)
   *
   * @param headers Map with extra HTTP headers.
   */
  public suspend fun setExtraHTTPHeaders(headers: Headers): SetExtraHTTPHeadersResponse {
    val input = SetExtraHTTPHeadersRequest(headers)
    return setExtraHTTPHeaders(input)
  }

  /**
   * Specifies whether to attach a page script stack id in requests
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setAttachDebugStack)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setAttachDebugStack(input: SetAttachDebugStackRequest):
      SetAttachDebugStackResponse = session.request("Network.setAttachDebugStack", input)

  /**
   * Specifies whether to attach a page script stack id in requests
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setAttachDebugStack)
   *
   * @param enabled Whether to attach a page script stack for debugging purpose.
   */
  @ExperimentalChromeApi
  public suspend fun setAttachDebugStack(enabled: Boolean): SetAttachDebugStackResponse {
    val input = SetAttachDebugStackRequest(enabled)
    return setAttachDebugStack(input)
  }

  /**
   * Sets the requests to intercept that match the provided patterns and optionally resource types.
   * Deprecated, please use Fetch.enable instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setRequestInterception)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setRequestInterception(input: SetRequestInterceptionRequest):
      SetRequestInterceptionResponse = session.request("Network.setRequestInterception", input)

  /**
   * Sets the requests to intercept that match the provided patterns and optionally resource types.
   * Deprecated, please use Fetch.enable instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setRequestInterception)
   *
   * @param patterns Requests matching any of these patterns will be forwarded and wait for the
   * corresponding
   * continueInterceptedRequest call.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setRequestInterception(patterns: List<RequestPattern>):
      SetRequestInterceptionResponse {
    val input = SetRequestInterceptionRequest(patterns)
    return setRequestInterception(input)
  }

  /**
   * Allows overriding user agent with the given string.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setUserAgentOverride)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setUserAgentOverride(input: SetUserAgentOverrideRequest):
      SetUserAgentOverrideResponse = session.request("Network.setUserAgentOverride", input)

  /**
   * Allows overriding user agent with the given string.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-setUserAgentOverride)
   *
   * @param userAgent User agent to use.
   */
  @JvmOverloads
  public suspend inline fun setUserAgentOverride(userAgent: String,
      optionalArgs: SetUserAgentOverrideRequest.Builder.() -> Unit = {}):
      SetUserAgentOverrideResponse {
    val builder = SetUserAgentOverrideRequest.Builder(userAgent)
    val input = builder.apply(optionalArgs).build()
    return setUserAgentOverride(input)
  }

  /**
   * Returns information about the COEP/COOP isolation status.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getSecurityIsolationStatus)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getSecurityIsolationStatus(input: GetSecurityIsolationStatusRequest):
      GetSecurityIsolationStatusResponse = session.request("Network.getSecurityIsolationStatus",
      input)

  /**
   * Returns information about the COEP/COOP isolation status.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-getSecurityIsolationStatus)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun getSecurityIsolationStatus(optionalArgs: GetSecurityIsolationStatusRequest.Builder.() -> Unit
      = {}): GetSecurityIsolationStatusResponse {
    val builder = GetSecurityIsolationStatusRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getSecurityIsolationStatus(input)
  }

  /**
   * Enables tracking for the Reporting API, events generated by the Reporting API will now be
   * delivered to the client.
   * Enabling triggers 'reportingApiReportAdded' for all existing reports.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-enableReportingApi)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun enableReportingApi(input: EnableReportingApiRequest):
      EnableReportingApiResponse = session.request("Network.enableReportingApi", input)

  /**
   * Enables tracking for the Reporting API, events generated by the Reporting API will now be
   * delivered to the client.
   * Enabling triggers 'reportingApiReportAdded' for all existing reports.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-enableReportingApi)
   *
   * @param enable Whether to enable or disable events for the Reporting API
   */
  @ExperimentalChromeApi
  public suspend fun enableReportingApi(enable: Boolean): EnableReportingApiResponse {
    val input = EnableReportingApiRequest(enable)
    return enableReportingApi(input)
  }

  /**
   * Fetches the resource and returns the content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-loadNetworkResource)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun loadNetworkResource(input: LoadNetworkResourceRequest):
      LoadNetworkResourceResponse = session.request("Network.loadNetworkResource", input)

  /**
   * Fetches the resource and returns the content.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#method-loadNetworkResource)
   *
   * @param url URL of the resource to get content for.
   * @param options Options for the request.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun loadNetworkResource(
    url: String,
    options: LoadNetworkResourceOptions,
    optionalArgs: LoadNetworkResourceRequest.Builder.() -> Unit = {},
  ): LoadNetworkResourceResponse {
    val builder = LoadNetworkResourceRequest.Builder(url, options)
    val input = builder.apply(optionalArgs).build()
    return loadNetworkResource(input)
  }
}
