@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.network.events

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.AuthChallenge
import org.hildan.chrome.devtools.domains.network.BlockedCookieWithReason
import org.hildan.chrome.devtools.domains.network.BlockedReason
import org.hildan.chrome.devtools.domains.network.BlockedSetCookieWithReason
import org.hildan.chrome.devtools.domains.network.ClientSecurityState
import org.hildan.chrome.devtools.domains.network.ConnectTiming
import org.hildan.chrome.devtools.domains.network.CorsErrorStatus
import org.hildan.chrome.devtools.domains.network.ErrorReason
import org.hildan.chrome.devtools.domains.network.Headers
import org.hildan.chrome.devtools.domains.network.IPAddressSpace
import org.hildan.chrome.devtools.domains.network.Initiator
import org.hildan.chrome.devtools.domains.network.InterceptionId
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.MonotonicTime
import org.hildan.chrome.devtools.domains.network.ReportingApiEndpoint
import org.hildan.chrome.devtools.domains.network.ReportingApiReport
import org.hildan.chrome.devtools.domains.network.Request
import org.hildan.chrome.devtools.domains.network.RequestId
import org.hildan.chrome.devtools.domains.network.ResourcePriority
import org.hildan.chrome.devtools.domains.network.ResourceType
import org.hildan.chrome.devtools.domains.network.Response
import org.hildan.chrome.devtools.domains.network.SignedExchangeInfo
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.network.TrustTokenOperationDoneStatus
import org.hildan.chrome.devtools.domains.network.TrustTokenOperationType
import org.hildan.chrome.devtools.domains.network.WebSocketFrame
import org.hildan.chrome.devtools.domains.network.WebSocketRequest
import org.hildan.chrome.devtools.domains.network.WebSocketResponse
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class NetworkEvent {
  /**
   * Fired when data chunk was received over the network.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-dataReceived)
   */
  @Serializable
  public data class DataReceived(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Data chunk length.
     */
    public val dataLength: Int,
    /**
     * Actual bytes received (might be less than dataLength for compressed encodings).
     */
    public val encodedDataLength: Int,
  ) : NetworkEvent()

  /**
   * Fired when EventSource message is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-eventSourceMessageReceived)
   */
  @Serializable
  public data class EventSourceMessageReceived(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Message type.
     */
    public val eventName: String,
    /**
     * Message identifier.
     */
    public val eventId: String,
    /**
     * Message content.
     */
    public val `data`: String,
  ) : NetworkEvent()

  /**
   * Fired when HTTP request has failed to load.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-loadingFailed)
   */
  @Serializable
  public data class LoadingFailed(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Resource type.
     */
    public val type: ResourceType,
    /**
     * User friendly error message.
     */
    public val errorText: String,
    /**
     * True if loading was canceled.
     */
    public val canceled: Boolean? = null,
    /**
     * The reason why loading was blocked, if any.
     */
    public val blockedReason: BlockedReason? = null,
    /**
     * The reason why loading was blocked by CORS, if any.
     */
    public val corsErrorStatus: CorsErrorStatus? = null,
  ) : NetworkEvent()

  /**
   * Fired when HTTP request has finished loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-loadingFinished)
   */
  @Serializable
  public data class LoadingFinished(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Total number of bytes received for this request.
     */
    public val encodedDataLength: Double,
    /**
     * Set when 1) response was blocked by Cross-Origin Read Blocking and also
     * 2) this needs to be reported to the DevTools console.
     */
    public val shouldReportCorbBlocking: Boolean? = null,
  ) : NetworkEvent()

  /**
   * Details of an intercepted HTTP request, which must be either allowed, blocked, modified or
   * mocked.
   * Deprecated, use Fetch.requestPaused instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestIntercepted)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  @Serializable
  public data class RequestIntercepted(
    /**
     * Each request the page makes will have a unique id, however if any redirects are encountered
     * while processing that fetch, they will be reported with the same id as the original fetch.
     * Likewise if HTTP authentication is needed then the same fetch id will be used.
     */
    public val interceptionId: InterceptionId,
    public val request: Request,
    /**
     * The id of the frame that initiated the request.
     */
    public val frameId: FrameId,
    /**
     * How the requested resource will be used.
     */
    public val resourceType: ResourceType,
    /**
     * Whether this is a navigation request, which can abort the navigation completely.
     */
    public val isNavigationRequest: Boolean,
    /**
     * Set if the request is a navigation that will result in a download.
     * Only present after response is received from the server (i.e. HeadersReceived stage).
     */
    public val isDownload: Boolean? = null,
    /**
     * Redirect location, only sent if a redirect was intercepted.
     */
    public val redirectUrl: String? = null,
    /**
     * Details of the Authorization Challenge encountered. If this is set then
     * continueInterceptedRequest must contain an authChallengeResponse.
     */
    public val authChallenge: AuthChallenge? = null,
    /**
     * Response error if intercepted at response stage or if redirect occurred while intercepting
     * request.
     */
    public val responseErrorReason: ErrorReason? = null,
    /**
     * Response code if intercepted at response stage or if redirect occurred while intercepting
     * request or auth retry occurred.
     */
    public val responseStatusCode: Int? = null,
    /**
     * Response headers if intercepted at the response stage or if redirect occurred while
     * intercepting request or auth retry occurred.
     */
    public val responseHeaders: Headers? = null,
    /**
     * If the intercepted request had a corresponding requestWillBeSent event fired for it, then
     * this requestId will be the same as the requestId present in the requestWillBeSent event.
     */
    public val requestId: RequestId? = null,
  ) : NetworkEvent()

  /**
   * Fired if request ended up loading from cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestServedFromCache)
   */
  @Serializable
  public data class RequestServedFromCache(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
  ) : NetworkEvent()

  /**
   * Fired when page is about to send HTTP request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestWillBeSent)
   */
  @Serializable
  public data class RequestWillBeSent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Loader identifier. Empty string if the request is fetched from worker.
     */
    public val loaderId: LoaderId,
    /**
     * URL of the document this request is loaded for.
     */
    public val documentURL: String,
    /**
     * Request data.
     */
    public val request: Request,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Timestamp.
     */
    public val wallTime: TimeSinceEpoch,
    /**
     * Request initiator.
     */
    public val initiator: Initiator,
    /**
     * In the case that redirectResponse is populated, this flag indicates whether
     * requestWillBeSentExtraInfo and responseReceivedExtraInfo events will be or were emitted
     * for the request which was just redirected.
     */
    @ExperimentalChromeApi
    public val redirectHasExtraInfo: Boolean? = null,
    /**
     * Redirect response data.
     */
    public val redirectResponse: Response? = null,
    /**
     * Type of this resource.
     */
    public val type: ResourceType? = null,
    /**
     * Frame identifier.
     */
    public val frameId: FrameId? = null,
    /**
     * Whether the request is initiated by a user gesture. Defaults to false.
     */
    public val hasUserGesture: Boolean? = null,
  ) : NetworkEvent()

  /**
   * Fired when resource loading priority is changed
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-resourceChangedPriority)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ResourceChangedPriority(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * New priority
     */
    public val newPriority: ResourcePriority,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
  ) : NetworkEvent()

  /**
   * Fired when a signed exchange was received over the network
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-signedExchangeReceived)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class SignedExchangeReceived(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Information about the signed exchange response.
     */
    public val info: SignedExchangeInfo,
  ) : NetworkEvent()

  /**
   * Fired when HTTP response is available.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-responseReceived)
   */
  @Serializable
  public data class ResponseReceived(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Loader identifier. Empty string if the request is fetched from worker.
     */
    public val loaderId: LoaderId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Resource type.
     */
    public val type: ResourceType,
    /**
     * Response data.
     */
    public val response: Response,
    /**
     * Indicates whether requestWillBeSentExtraInfo and responseReceivedExtraInfo events will be
     * or were emitted for this request.
     */
    @ExperimentalChromeApi
    public val hasExtraInfo: Boolean? = null,
    /**
     * Frame identifier.
     */
    public val frameId: FrameId? = null,
  ) : NetworkEvent()

  /**
   * Fired when WebSocket is closed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketClosed)
   */
  @Serializable
  public data class WebSocketClosed(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
  ) : NetworkEvent()

  /**
   * Fired upon WebSocket creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketCreated)
   */
  @Serializable
  public data class WebSocketCreated(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * WebSocket request URL.
     */
    public val url: String,
    /**
     * Request initiator.
     */
    public val initiator: Initiator? = null,
  ) : NetworkEvent()

  /**
   * Fired when WebSocket message error occurs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketFrameError)
   */
  @Serializable
  public data class WebSocketFrameError(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket error message.
     */
    public val errorMessage: String,
  ) : NetworkEvent()

  /**
   * Fired when WebSocket message is received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketFrameReceived)
   */
  @Serializable
  public data class WebSocketFrameReceived(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket response data.
     */
    public val response: WebSocketFrame,
  ) : NetworkEvent()

  /**
   * Fired when WebSocket message is sent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketFrameSent)
   */
  @Serializable
  public data class WebSocketFrameSent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket response data.
     */
    public val response: WebSocketFrame,
  ) : NetworkEvent()

  /**
   * Fired when WebSocket handshake response becomes available.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketHandshakeResponseReceived)
   */
  @Serializable
  public data class WebSocketHandshakeResponseReceived(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket response data.
     */
    public val response: WebSocketResponse,
  ) : NetworkEvent()

  /**
   * Fired when WebSocket is about to initiate handshake.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webSocketWillSendHandshakeRequest)
   */
  @Serializable
  public data class WebSocketWillSendHandshakeRequest(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * UTC Timestamp.
     */
    public val wallTime: TimeSinceEpoch,
    /**
     * WebSocket request data.
     */
    public val request: WebSocketRequest,
  ) : NetworkEvent()

  /**
   * Fired upon WebTransport creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webTransportCreated)
   */
  @Serializable
  public data class WebTransportCreated(
    /**
     * WebTransport identifier.
     */
    public val transportId: RequestId,
    /**
     * WebTransport request URL.
     */
    public val url: String,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Request initiator.
     */
    public val initiator: Initiator? = null,
  ) : NetworkEvent()

  /**
   * Fired when WebTransport handshake is finished.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webTransportConnectionEstablished)
   */
  @Serializable
  public data class WebTransportConnectionEstablished(
    /**
     * WebTransport identifier.
     */
    public val transportId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
  ) : NetworkEvent()

  /**
   * Fired when WebTransport is disposed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-webTransportClosed)
   */
  @Serializable
  public data class WebTransportClosed(
    /**
     * WebTransport identifier.
     */
    public val transportId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
  ) : NetworkEvent()

  /**
   * Fired when additional information about a requestWillBeSent event is available from the
   * network stack. Not every requestWillBeSent event will have an additional
   * requestWillBeSentExtraInfo fired for it, and there is no guarantee whether requestWillBeSent
   * or requestWillBeSentExtraInfo will be fired first for the same request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-requestWillBeSentExtraInfo)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class RequestWillBeSentExtraInfo(
    /**
     * Request identifier. Used to match this information to an existing requestWillBeSent event.
     */
    public val requestId: RequestId,
    /**
     * A list of cookies potentially associated to the requested URL. This includes both cookies
     * sent with
     * the request and the ones not sent; the latter are distinguished by having blockedReason field
     * set.
     */
    public val associatedCookies: List<BlockedCookieWithReason>,
    /**
     * Raw request headers as they will be sent over the wire.
     */
    public val headers: Headers,
    /**
     * Connection timing information for the request.
     */
    @ExperimentalChromeApi
    public val connectTiming: ConnectTiming? = null,
    /**
     * The client security state set for the request.
     */
    public val clientSecurityState: ClientSecurityState? = null,
    /**
     * Whether the site has partitioned cookies stored in a partition different than the current
     * one.
     */
    public val siteHasCookieInOtherPartition: Boolean? = null,
  ) : NetworkEvent()

  /**
   * Fired when additional information about a responseReceived event is available from the network
   * stack. Not every responseReceived event will have an additional responseReceivedExtraInfo for
   * it, and responseReceivedExtraInfo may be fired before or after responseReceived.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-responseReceivedExtraInfo)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ResponseReceivedExtraInfo(
    /**
     * Request identifier. Used to match this information to another responseReceived event.
     */
    public val requestId: RequestId,
    /**
     * A list of cookies which were not stored from the response along with the corresponding
     * reasons for blocking. The cookies here may not be valid due to syntax errors, which
     * are represented by the invalid cookie line string instead of a proper cookie.
     */
    public val blockedCookies: List<BlockedSetCookieWithReason>,
    /**
     * Raw response headers as they were received over the wire.
     */
    public val headers: Headers,
    /**
     * The IP address space of the resource. The address space can only be determined once the
     * transport
     * established the connection, so we can't send it in `requestWillBeSentExtraInfo`.
     */
    public val resourceIPAddressSpace: IPAddressSpace,
    /**
     * The status code of the response. This is useful in cases the request failed and no
     * responseReceived
     * event is triggered, which is the case for, e.g., CORS errors. This is also the correct status
     * code
     * for cached requests, where the status in responseReceived is a 200 and this will be 304.
     */
    public val statusCode: Int,
    /**
     * Raw response header text as it was received over the wire. The raw text may not always be
     * available, such as in the case of HTTP/2 or QUIC.
     */
    public val headersText: String? = null,
  ) : NetworkEvent()

  /**
   * Fired exactly once for each Trust Token operation. Depending on
   * the type of the operation and whether the operation succeeded or
   * failed, the event is fired before the corresponding request was sent
   * or after the response was received.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-trustTokenOperationDone)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class TrustTokenOperationDone(
    /**
     * Detailed success or error status of the operation.
     * 'AlreadyExists' also signifies a successful operation, as the result
     * of the operation already exists und thus, the operation was abort
     * preemptively (e.g. a cache hit).
     */
    public val status: TrustTokenOperationDoneStatus,
    public val type: TrustTokenOperationType,
    public val requestId: RequestId,
    /**
     * Top level origin. The context in which the operation was attempted.
     */
    public val topLevelOrigin: String? = null,
    /**
     * Origin of the issuer in case of a "Issuance" or "Redemption" operation.
     */
    public val issuerOrigin: String? = null,
    /**
     * The number of obtained Trust Tokens on a successful "Issuance" operation.
     */
    public val issuedTokenCount: Int? = null,
  ) : NetworkEvent()

  /**
   * Fired once when parsing the .wbn file has succeeded.
   * The event contains the information about the web bundle contents.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleMetadataReceived)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class SubresourceWebBundleMetadataReceived(
    /**
     * Request identifier. Used to match this information to another event.
     */
    public val requestId: RequestId,
    /**
     * A list of URLs of resources in the subresource Web Bundle.
     */
    public val urls: List<String>,
  ) : NetworkEvent()

  /**
   * Fired once when parsing the .wbn file has failed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleMetadataError)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class SubresourceWebBundleMetadataError(
    /**
     * Request identifier. Used to match this information to another event.
     */
    public val requestId: RequestId,
    /**
     * Error message
     */
    public val errorMessage: String,
  ) : NetworkEvent()

  /**
   * Fired when handling requests for resources within a .wbn file.
   * Note: this will only be fired for resources that are requested by the webpage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleInnerResponseParsed)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class SubresourceWebBundleInnerResponseParsed(
    /**
     * Request identifier of the subresource request
     */
    public val innerRequestId: RequestId,
    /**
     * URL of the subresource resource.
     */
    public val innerRequestURL: String,
    /**
     * Bundle request identifier. Used to match this information to another event.
     * This made be absent in case when the instrumentation was enabled only
     * after webbundle was parsed.
     */
    public val bundleRequestId: RequestId? = null,
  ) : NetworkEvent()

  /**
   * Fired when request for resources within a .wbn file failed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-subresourceWebBundleInnerResponseError)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class SubresourceWebBundleInnerResponseError(
    /**
     * Request identifier of the subresource request
     */
    public val innerRequestId: RequestId,
    /**
     * URL of the subresource resource.
     */
    public val innerRequestURL: String,
    /**
     * Error message
     */
    public val errorMessage: String,
    /**
     * Bundle request identifier. Used to match this information to another event.
     * This made be absent in case when the instrumentation was enabled only
     * after webbundle was parsed.
     */
    public val bundleRequestId: RequestId? = null,
  ) : NetworkEvent()

  /**
   * Is sent whenever a new report is added.
   * And after 'enableReportingApi' for all existing reports.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-reportingApiReportAdded)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ReportingApiReportAdded(
    public val report: ReportingApiReport,
  ) : NetworkEvent()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-reportingApiReportUpdated)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ReportingApiReportUpdated(
    public val report: ReportingApiReport,
  ) : NetworkEvent()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#event-reportingApiEndpointsChangedForOrigin)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ReportingApiEndpointsChangedForOrigin(
    /**
     * Origin of the document(s) which configured the endpoints.
     */
    public val origin: String,
    public val endpoints: List<ReportingApiEndpoint>,
  ) : NetworkEvent()
}
