@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.overlay

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.dom.Quad
import org.hildan.chrome.devtools.domains.dom.RGBA
import org.hildan.chrome.devtools.domains.overlay.events.OverlayEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [OverlayDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [OverlayDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [OverlayDomain.getHighlightObjectForTest]
 * command.
 */
@Serializable
public data class GetHighlightObjectForTestRequest(
  /**
   * Id of the node to get highlight object for.
   */
  public val nodeId: NodeId,
  /**
   * Whether to include distance info.
   */
  public val includeDistance: Boolean? = null,
  /**
   * Whether to include style info.
   */
  public val includeStyle: Boolean? = null,
  /**
   * The color format to get config with (default: hex).
   */
  public val colorFormat: ColorFormat? = null,
  /**
   * Whether to show accessibility info (default: true).
   */
  public val showAccessibilityInfo: Boolean? = null,
) {
  public class Builder(
    /**
     * Id of the node to get highlight object for.
     */
    public val nodeId: NodeId,
  ) {
    /**
     * Whether to include distance info.
     */
    public var includeDistance: Boolean? = null

    /**
     * Whether to include style info.
     */
    public var includeStyle: Boolean? = null

    /**
     * The color format to get config with (default: hex).
     */
    public var colorFormat: ColorFormat? = null

    /**
     * Whether to show accessibility info (default: true).
     */
    public var showAccessibilityInfo: Boolean? = null

    public fun build(): GetHighlightObjectForTestRequest = GetHighlightObjectForTestRequest(nodeId,
        includeDistance, includeStyle, colorFormat, showAccessibilityInfo)
  }
}

/**
 * Response type for the [OverlayDomain.getHighlightObjectForTest] command.
 */
@Serializable
public data class GetHighlightObjectForTestResponse(
  /**
   * Highlight data for the node.
   */
  public val highlight: JsonObject,
)

/**
 * Request object containing input parameters for the [OverlayDomain.getGridHighlightObjectsForTest]
 * command.
 */
@Serializable
public data class GetGridHighlightObjectsForTestRequest(
  /**
   * Ids of the node to get highlight object for.
   */
  public val nodeIds: List<NodeId>,
)

/**
 * Response type for the [OverlayDomain.getGridHighlightObjectsForTest] command.
 */
@Serializable
public data class GetGridHighlightObjectsForTestResponse(
  /**
   * Grid Highlight data for the node ids provided.
   */
  public val highlights: JsonObject,
)

/**
 * Request object containing input parameters for the
 * [OverlayDomain.getSourceOrderHighlightObjectForTest] command.
 */
@Serializable
public data class GetSourceOrderHighlightObjectForTestRequest(
  /**
   * Id of the node to highlight.
   */
  public val nodeId: NodeId,
)

/**
 * Response type for the [OverlayDomain.getSourceOrderHighlightObjectForTest] command.
 */
@Serializable
public data class GetSourceOrderHighlightObjectForTestResponse(
  /**
   * Source order highlight data for the node id provided.
   */
  public val highlight: JsonObject,
)

/**
 * A dummy response object for the [OverlayDomain.hideHighlight] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object HideHighlightResponse

/**
 * Request object containing input parameters for the [OverlayDomain.highlightFrame] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class HighlightFrameRequest(
  /**
   * Identifier of the frame to highlight.
   */
  public val frameId: FrameId,
  /**
   * The content box highlight fill color (default: transparent).
   */
  public val contentColor: RGBA? = null,
  /**
   * The content box highlight outline color (default: transparent).
   */
  public val contentOutlineColor: RGBA? = null,
) {
  public class Builder(
    /**
     * Identifier of the frame to highlight.
     */
    public val frameId: FrameId,
  ) {
    /**
     * The content box highlight fill color (default: transparent).
     */
    public var contentColor: RGBA? = null

    /**
     * The content box highlight outline color (default: transparent).
     */
    public var contentOutlineColor: RGBA? = null

    public fun build(): HighlightFrameRequest = HighlightFrameRequest(frameId, contentColor,
        contentOutlineColor)
  }
}

/**
 * A dummy response object for the [OverlayDomain.highlightFrame] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object HighlightFrameResponse

/**
 * Request object containing input parameters for the [OverlayDomain.highlightNode] command.
 */
@Serializable
public data class HighlightNodeRequest(
  /**
   * A descriptor for the highlight appearance.
   */
  public val highlightConfig: HighlightConfig,
  /**
   * Identifier of the node to highlight.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to highlight.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node to be highlighted.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Selectors to highlight relevant nodes.
   */
  public val selector: String? = null,
) {
  public class Builder(
    /**
     * A descriptor for the highlight appearance.
     */
    public val highlightConfig: HighlightConfig,
  ) {
    /**
     * Identifier of the node to highlight.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node to highlight.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node to be highlighted.
     */
    public var objectId: RemoteObjectId? = null

    /**
     * Selectors to highlight relevant nodes.
     */
    public var selector: String? = null

    public fun build(): HighlightNodeRequest = HighlightNodeRequest(highlightConfig, nodeId,
        backendNodeId, objectId, selector)
  }
}

/**
 * A dummy response object for the [OverlayDomain.highlightNode] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object HighlightNodeResponse

/**
 * Request object containing input parameters for the [OverlayDomain.highlightQuad] command.
 */
@Serializable
public data class HighlightQuadRequest(
  /**
   * Quad to highlight
   */
  public val quad: Quad,
  /**
   * The highlight fill color (default: transparent).
   */
  public val color: RGBA? = null,
  /**
   * The highlight outline color (default: transparent).
   */
  public val outlineColor: RGBA? = null,
) {
  public class Builder(
    /**
     * Quad to highlight
     */
    public val quad: Quad,
  ) {
    /**
     * The highlight fill color (default: transparent).
     */
    public var color: RGBA? = null

    /**
     * The highlight outline color (default: transparent).
     */
    public var outlineColor: RGBA? = null

    public fun build(): HighlightQuadRequest = HighlightQuadRequest(quad, color, outlineColor)
  }
}

/**
 * A dummy response object for the [OverlayDomain.highlightQuad] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object HighlightQuadResponse

/**
 * Request object containing input parameters for the [OverlayDomain.highlightRect] command.
 */
@Serializable
public data class HighlightRectRequest(
  /**
   * X coordinate
   */
  public val x: Int,
  /**
   * Y coordinate
   */
  public val y: Int,
  /**
   * Rectangle width
   */
  public val width: Int,
  /**
   * Rectangle height
   */
  public val height: Int,
  /**
   * The highlight fill color (default: transparent).
   */
  public val color: RGBA? = null,
  /**
   * The highlight outline color (default: transparent).
   */
  public val outlineColor: RGBA? = null,
) {
  public class Builder(
    /**
     * X coordinate
     */
    public val x: Int,
    /**
     * Y coordinate
     */
    public val y: Int,
    /**
     * Rectangle width
     */
    public val width: Int,
    /**
     * Rectangle height
     */
    public val height: Int,
  ) {
    /**
     * The highlight fill color (default: transparent).
     */
    public var color: RGBA? = null

    /**
     * The highlight outline color (default: transparent).
     */
    public var outlineColor: RGBA? = null

    public fun build(): HighlightRectRequest = HighlightRectRequest(x, y, width, height, color,
        outlineColor)
  }
}

/**
 * A dummy response object for the [OverlayDomain.highlightRect] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object HighlightRectResponse

/**
 * Request object containing input parameters for the [OverlayDomain.highlightSourceOrder] command.
 */
@Serializable
public data class HighlightSourceOrderRequest(
  /**
   * A descriptor for the appearance of the overlay drawing.
   */
  public val sourceOrderConfig: SourceOrderConfig,
  /**
   * Identifier of the node to highlight.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to highlight.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node to be highlighted.
   */
  public val objectId: RemoteObjectId? = null,
) {
  public class Builder(
    /**
     * A descriptor for the appearance of the overlay drawing.
     */
    public val sourceOrderConfig: SourceOrderConfig,
  ) {
    /**
     * Identifier of the node to highlight.
     */
    public var nodeId: NodeId? = null

    /**
     * Identifier of the backend node to highlight.
     */
    public var backendNodeId: BackendNodeId? = null

    /**
     * JavaScript object id of the node to be highlighted.
     */
    public var objectId: RemoteObjectId? = null

    public fun build(): HighlightSourceOrderRequest = HighlightSourceOrderRequest(sourceOrderConfig,
        nodeId, backendNodeId, objectId)
  }
}

/**
 * A dummy response object for the [OverlayDomain.highlightSourceOrder] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object HighlightSourceOrderResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setInspectMode] command.
 */
@Serializable
public data class SetInspectModeRequest(
  /**
   * Set an inspection mode.
   */
  public val mode: InspectMode,
  /**
   * A descriptor for the highlight appearance of hovered-over nodes. May be omitted if `enabled
   * == false`.
   */
  public val highlightConfig: HighlightConfig? = null,
) {
  public class Builder(
    /**
     * Set an inspection mode.
     */
    public val mode: InspectMode,
  ) {
    /**
     * A descriptor for the highlight appearance of hovered-over nodes. May be omitted if `enabled
     * == false`.
     */
    public var highlightConfig: HighlightConfig? = null

    public fun build(): SetInspectModeRequest = SetInspectModeRequest(mode, highlightConfig)
  }
}

/**
 * A dummy response object for the [OverlayDomain.setInspectMode] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetInspectModeResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowAdHighlights] command.
 */
@Serializable
public data class SetShowAdHighlightsRequest(
  /**
   * True for showing ad highlights
   */
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowAdHighlights] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowAdHighlightsResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setPausedInDebuggerMessage]
 * command.
 */
@Serializable
public data class SetPausedInDebuggerMessageRequest(
  /**
   * The message to display, also triggers resume and step over controls.
   */
  public val message: String? = null,
) {
  public class Builder() {
    /**
     * The message to display, also triggers resume and step over controls.
     */
    public var message: String? = null

    public fun build(): SetPausedInDebuggerMessageRequest =
        SetPausedInDebuggerMessageRequest(message)
  }
}

/**
 * A dummy response object for the [OverlayDomain.setPausedInDebuggerMessage] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetPausedInDebuggerMessageResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowDebugBorders] command.
 */
@Serializable
public data class SetShowDebugBordersRequest(
  /**
   * True for showing debug borders
   */
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowDebugBorders] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowDebugBordersResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowFPSCounter] command.
 */
@Serializable
public data class SetShowFPSCounterRequest(
  /**
   * True for showing the FPS counter
   */
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowFPSCounter] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowFPSCounterResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowGridOverlays] command.
 */
@Serializable
public data class SetShowGridOverlaysRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val gridNodeHighlightConfigs: List<GridNodeHighlightConfig>,
)

/**
 * A dummy response object for the [OverlayDomain.setShowGridOverlays] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowGridOverlaysResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowFlexOverlays] command.
 */
@Serializable
public data class SetShowFlexOverlaysRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val flexNodeHighlightConfigs: List<FlexNodeHighlightConfig>,
)

/**
 * A dummy response object for the [OverlayDomain.setShowFlexOverlays] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowFlexOverlaysResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowScrollSnapOverlays]
 * command.
 */
@Serializable
public data class SetShowScrollSnapOverlaysRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val scrollSnapHighlightConfigs: List<ScrollSnapHighlightConfig>,
)

/**
 * A dummy response object for the [OverlayDomain.setShowScrollSnapOverlays] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetShowScrollSnapOverlaysResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowContainerQueryOverlays]
 * command.
 */
@Serializable
public data class SetShowContainerQueryOverlaysRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val containerQueryHighlightConfigs: List<ContainerQueryHighlightConfig>,
)

/**
 * A dummy response object for the [OverlayDomain.setShowContainerQueryOverlays] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetShowContainerQueryOverlaysResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowPaintRects] command.
 */
@Serializable
public data class SetShowPaintRectsRequest(
  /**
   * True for showing paint rectangles
   */
  public val result: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowPaintRects] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowPaintRectsResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowLayoutShiftRegions]
 * command.
 */
@Serializable
public data class SetShowLayoutShiftRegionsRequest(
  /**
   * True for showing layout shift regions
   */
  public val result: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowLayoutShiftRegions] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetShowLayoutShiftRegionsResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowScrollBottleneckRects]
 * command.
 */
@Serializable
public data class SetShowScrollBottleneckRectsRequest(
  /**
   * True for showing scroll bottleneck rects
   */
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowScrollBottleneckRects] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetShowScrollBottleneckRectsResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowHitTestBorders] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class SetShowHitTestBordersRequest(
  /**
   * True for showing hit-test borders
   */
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowHitTestBorders] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object SetShowHitTestBordersResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowWebVitals] command.
 */
@Serializable
public data class SetShowWebVitalsRequest(
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowWebVitals] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetShowWebVitalsResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowViewportSizeOnResize]
 * command.
 */
@Serializable
public data class SetShowViewportSizeOnResizeRequest(
  /**
   * Whether to paint size or not.
   */
  public val show: Boolean,
)

/**
 * A dummy response object for the [OverlayDomain.setShowViewportSizeOnResize] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetShowViewportSizeOnResizeResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowHinge] command.
 */
@Serializable
public data class SetShowHingeRequest(
  /**
   * hinge data, null means hideHinge
   */
  public val hingeConfig: HingeConfig? = null,
) {
  public class Builder() {
    /**
     * hinge data, null means hideHinge
     */
    public var hingeConfig: HingeConfig? = null

    public fun build(): SetShowHingeRequest = SetShowHingeRequest(hingeConfig)
  }
}

/**
 * A dummy response object for the [OverlayDomain.setShowHinge] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object SetShowHingeResponse

/**
 * Request object containing input parameters for the [OverlayDomain.setShowIsolatedElements]
 * command.
 */
@Serializable
public data class SetShowIsolatedElementsRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val isolatedElementHighlightConfigs: List<IsolatedElementHighlightConfig>,
)

/**
 * A dummy response object for the [OverlayDomain.setShowIsolatedElements] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetShowIsolatedElementsResponse

/**
 * This domain provides various functionality related to drawing atop the inspected page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay)
 */
@ExperimentalChromeApi
public class OverlayDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out OverlayEvent>> =
      mapOf(
      "Overlay.inspectNodeRequested" to serializer<OverlayEvent.InspectNodeRequested>(),
      "Overlay.nodeHighlightRequested" to serializer<OverlayEvent.NodeHighlightRequested>(),
      "Overlay.screenshotRequested" to serializer<OverlayEvent.ScreenshotRequested>(),
      "Overlay.inspectModeCanceled" to serializer<OverlayEvent.InspectModeCanceled>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<OverlayEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Fired when the node should be inspected. This happens after call to `setInspectMode` or when
   * user manually inspects an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-inspectNodeRequested)
   */
  public fun inspectNodeRequestedEvents(): Flow<OverlayEvent.InspectNodeRequested> =
      session.typedEvents("Overlay.inspectNodeRequested")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("inspectNodeRequestedEvents()"),
  )
  public fun inspectNodeRequested(): Flow<OverlayEvent.InspectNodeRequested> =
      inspectNodeRequestedEvents()

  /**
   * Fired when the node should be highlighted. This happens after call to `setInspectMode`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-nodeHighlightRequested)
   */
  public fun nodeHighlightRequestedEvents(): Flow<OverlayEvent.NodeHighlightRequested> =
      session.typedEvents("Overlay.nodeHighlightRequested")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("nodeHighlightRequestedEvents()"),
  )
  public fun nodeHighlightRequested(): Flow<OverlayEvent.NodeHighlightRequested> =
      nodeHighlightRequestedEvents()

  /**
   * Fired when user asks to capture screenshot of some area on the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-screenshotRequested)
   */
  public fun screenshotRequestedEvents(): Flow<OverlayEvent.ScreenshotRequested> =
      session.typedEvents("Overlay.screenshotRequested")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("screenshotRequestedEvents()"),
  )
  public fun screenshotRequested(): Flow<OverlayEvent.ScreenshotRequested> =
      screenshotRequestedEvents()

  /**
   * Fired when user cancels the inspect mode.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-inspectModeCanceled)
   */
  public fun inspectModeCanceledEvents(): Flow<OverlayEvent.InspectModeCanceled> =
      session.typedEvents("Overlay.inspectModeCanceled")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("inspectModeCanceledEvents()"),
  )
  public fun inspectModeCanceled(): Flow<OverlayEvent.InspectModeCanceled> =
      inspectModeCanceledEvents()

  /**
   * Disables domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Overlay.disable", Unit)

  /**
   * Enables domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Overlay.enable", Unit)

  /**
   * For testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getHighlightObjectForTest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getHighlightObjectForTest(input: GetHighlightObjectForTestRequest):
      GetHighlightObjectForTestResponse = session.request("Overlay.getHighlightObjectForTest",
      input)

  /**
   * For testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getHighlightObjectForTest)
   *
   * @param nodeId Id of the node to get highlight object for.
   */
  @JvmOverloads
  public suspend inline fun getHighlightObjectForTest(nodeId: NodeId,
      optionalArgs: GetHighlightObjectForTestRequest.Builder.() -> Unit = {}):
      GetHighlightObjectForTestResponse {
    val builder = GetHighlightObjectForTestRequest.Builder(nodeId)
    val input = builder.apply(optionalArgs).build()
    return getHighlightObjectForTest(input)
  }

  /**
   * For Persistent Grid testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getGridHighlightObjectsForTest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getGridHighlightObjectsForTest(input: GetGridHighlightObjectsForTestRequest):
      GetGridHighlightObjectsForTestResponse =
      session.request("Overlay.getGridHighlightObjectsForTest", input)

  /**
   * For Persistent Grid testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getGridHighlightObjectsForTest)
   *
   * @param nodeIds Ids of the node to get highlight object for.
   */
  public suspend fun getGridHighlightObjectsForTest(nodeIds: List<NodeId>):
      GetGridHighlightObjectsForTestResponse {
    val input = GetGridHighlightObjectsForTestRequest(nodeIds)
    return getGridHighlightObjectsForTest(input)
  }

  /**
   * For Source Order Viewer testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getSourceOrderHighlightObjectForTest)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend
      fun getSourceOrderHighlightObjectForTest(input: GetSourceOrderHighlightObjectForTestRequest):
      GetSourceOrderHighlightObjectForTestResponse =
      session.request("Overlay.getSourceOrderHighlightObjectForTest", input)

  /**
   * For Source Order Viewer testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getSourceOrderHighlightObjectForTest)
   *
   * @param nodeId Id of the node to highlight.
   */
  public suspend fun getSourceOrderHighlightObjectForTest(nodeId: NodeId):
      GetSourceOrderHighlightObjectForTestResponse {
    val input = GetSourceOrderHighlightObjectForTestRequest(nodeId)
    return getSourceOrderHighlightObjectForTest(input)
  }

  /**
   * Hides any highlight.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-hideHighlight)
   */
  public suspend fun hideHighlight(): HideHighlightResponse =
      session.request("Overlay.hideHighlight", Unit)

  /**
   * Highlights owner element of the frame with given id.
   * Deprecated: Doesn't work reliablity and cannot be fixed due to process
   * separatation (the owner node might be in a different process). Determine
   * the owner node in the client and use highlightNode.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightFrame)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun highlightFrame(input: HighlightFrameRequest): HighlightFrameResponse =
      session.request("Overlay.highlightFrame", input)

  /**
   * Highlights owner element of the frame with given id.
   * Deprecated: Doesn't work reliablity and cannot be fixed due to process
   * separatation (the owner node might be in a different process). Determine
   * the owner node in the client and use highlightNode.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightFrame)
   *
   * @param frameId Identifier of the frame to highlight.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @JvmOverloads
  public suspend inline fun highlightFrame(frameId: FrameId,
      optionalArgs: HighlightFrameRequest.Builder.() -> Unit = {}): HighlightFrameResponse {
    val builder = HighlightFrameRequest.Builder(frameId)
    val input = builder.apply(optionalArgs).build()
    return highlightFrame(input)
  }

  /**
   * Highlights DOM node with given id or with the given JavaScript object wrapper. Either nodeId or
   * objectId must be specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightNode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun highlightNode(input: HighlightNodeRequest): HighlightNodeResponse =
      session.request("Overlay.highlightNode", input)

  /**
   * Highlights DOM node with given id or with the given JavaScript object wrapper. Either nodeId or
   * objectId must be specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightNode)
   *
   * @param highlightConfig A descriptor for the highlight appearance.
   */
  @JvmOverloads
  public suspend inline fun highlightNode(highlightConfig: HighlightConfig,
      optionalArgs: HighlightNodeRequest.Builder.() -> Unit = {}): HighlightNodeResponse {
    val builder = HighlightNodeRequest.Builder(highlightConfig)
    val input = builder.apply(optionalArgs).build()
    return highlightNode(input)
  }

  /**
   * Highlights given quad. Coordinates are absolute with respect to the main frame viewport.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightQuad)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun highlightQuad(input: HighlightQuadRequest): HighlightQuadResponse =
      session.request("Overlay.highlightQuad", input)

  /**
   * Highlights given quad. Coordinates are absolute with respect to the main frame viewport.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightQuad)
   *
   * @param quad Quad to highlight
   */
  @JvmOverloads
  public suspend inline fun highlightQuad(quad: Quad,
      optionalArgs: HighlightQuadRequest.Builder.() -> Unit = {}): HighlightQuadResponse {
    val builder = HighlightQuadRequest.Builder(quad)
    val input = builder.apply(optionalArgs).build()
    return highlightQuad(input)
  }

  /**
   * Highlights given rectangle. Coordinates are absolute with respect to the main frame viewport.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightRect)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun highlightRect(input: HighlightRectRequest): HighlightRectResponse =
      session.request("Overlay.highlightRect", input)

  /**
   * Highlights given rectangle. Coordinates are absolute with respect to the main frame viewport.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightRect)
   *
   * @param x X coordinate
   * @param y Y coordinate
   * @param width Rectangle width
   * @param height Rectangle height
   */
  @JvmOverloads
  public suspend inline fun highlightRect(
    x: Int,
    y: Int,
    width: Int,
    height: Int,
    optionalArgs: HighlightRectRequest.Builder.() -> Unit = {},
  ): HighlightRectResponse {
    val builder = HighlightRectRequest.Builder(x, y, width, height)
    val input = builder.apply(optionalArgs).build()
    return highlightRect(input)
  }

  /**
   * Highlights the source order of the children of the DOM node with given id or with the given
   * JavaScript object wrapper. Either nodeId or objectId must be specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightSourceOrder)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun highlightSourceOrder(input: HighlightSourceOrderRequest):
      HighlightSourceOrderResponse = session.request("Overlay.highlightSourceOrder", input)

  /**
   * Highlights the source order of the children of the DOM node with given id or with the given
   * JavaScript object wrapper. Either nodeId or objectId must be specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightSourceOrder)
   *
   * @param sourceOrderConfig A descriptor for the appearance of the overlay drawing.
   */
  @JvmOverloads
  public suspend inline fun highlightSourceOrder(sourceOrderConfig: SourceOrderConfig,
      optionalArgs: HighlightSourceOrderRequest.Builder.() -> Unit = {}):
      HighlightSourceOrderResponse {
    val builder = HighlightSourceOrderRequest.Builder(sourceOrderConfig)
    val input = builder.apply(optionalArgs).build()
    return highlightSourceOrder(input)
  }

  /**
   * Enters the 'inspect' mode. In this mode, elements that user is hovering over are highlighted.
   * Backend then generates 'inspectNodeRequested' event upon element selection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setInspectMode)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setInspectMode(input: SetInspectModeRequest): SetInspectModeResponse =
      session.request("Overlay.setInspectMode", input)

  /**
   * Enters the 'inspect' mode. In this mode, elements that user is hovering over are highlighted.
   * Backend then generates 'inspectNodeRequested' event upon element selection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setInspectMode)
   *
   * @param mode Set an inspection mode.
   */
  @JvmOverloads
  public suspend inline fun setInspectMode(mode: InspectMode,
      optionalArgs: SetInspectModeRequest.Builder.() -> Unit = {}): SetInspectModeResponse {
    val builder = SetInspectModeRequest.Builder(mode)
    val input = builder.apply(optionalArgs).build()
    return setInspectMode(input)
  }

  /**
   * Highlights owner element of all frames detected to be ads.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowAdHighlights)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowAdHighlights(input: SetShowAdHighlightsRequest):
      SetShowAdHighlightsResponse = session.request("Overlay.setShowAdHighlights", input)

  /**
   * Highlights owner element of all frames detected to be ads.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowAdHighlights)
   *
   * @param show True for showing ad highlights
   */
  public suspend fun setShowAdHighlights(show: Boolean): SetShowAdHighlightsResponse {
    val input = SetShowAdHighlightsRequest(show)
    return setShowAdHighlights(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setPausedInDebuggerMessage)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setPausedInDebuggerMessage(input: SetPausedInDebuggerMessageRequest):
      SetPausedInDebuggerMessageResponse = session.request("Overlay.setPausedInDebuggerMessage",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setPausedInDebuggerMessage)
   */
  @JvmOverloads
  public suspend inline
      fun setPausedInDebuggerMessage(optionalArgs: SetPausedInDebuggerMessageRequest.Builder.() -> Unit
      = {}): SetPausedInDebuggerMessageResponse {
    val builder = SetPausedInDebuggerMessageRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setPausedInDebuggerMessage(input)
  }

  /**
   * Requests that backend shows debug borders on layers
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowDebugBorders)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowDebugBorders(input: SetShowDebugBordersRequest):
      SetShowDebugBordersResponse = session.request("Overlay.setShowDebugBorders", input)

  /**
   * Requests that backend shows debug borders on layers
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowDebugBorders)
   *
   * @param show True for showing debug borders
   */
  public suspend fun setShowDebugBorders(show: Boolean): SetShowDebugBordersResponse {
    val input = SetShowDebugBordersRequest(show)
    return setShowDebugBorders(input)
  }

  /**
   * Requests that backend shows the FPS counter
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowFPSCounter)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowFPSCounter(input: SetShowFPSCounterRequest): SetShowFPSCounterResponse =
      session.request("Overlay.setShowFPSCounter", input)

  /**
   * Requests that backend shows the FPS counter
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowFPSCounter)
   *
   * @param show True for showing the FPS counter
   */
  public suspend fun setShowFPSCounter(show: Boolean): SetShowFPSCounterResponse {
    val input = SetShowFPSCounterRequest(show)
    return setShowFPSCounter(input)
  }

  /**
   * Highlight multiple elements with the CSS Grid overlay.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowGridOverlays)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowGridOverlays(input: SetShowGridOverlaysRequest):
      SetShowGridOverlaysResponse = session.request("Overlay.setShowGridOverlays", input)

  /**
   * Highlight multiple elements with the CSS Grid overlay.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowGridOverlays)
   *
   * @param gridNodeHighlightConfigs An array of node identifiers and descriptors for the highlight
   * appearance.
   */
  public suspend fun setShowGridOverlays(gridNodeHighlightConfigs: List<GridNodeHighlightConfig>):
      SetShowGridOverlaysResponse {
    val input = SetShowGridOverlaysRequest(gridNodeHighlightConfigs)
    return setShowGridOverlays(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowFlexOverlays)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowFlexOverlays(input: SetShowFlexOverlaysRequest):
      SetShowFlexOverlaysResponse = session.request("Overlay.setShowFlexOverlays", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowFlexOverlays)
   *
   * @param flexNodeHighlightConfigs An array of node identifiers and descriptors for the highlight
   * appearance.
   */
  public suspend fun setShowFlexOverlays(flexNodeHighlightConfigs: List<FlexNodeHighlightConfig>):
      SetShowFlexOverlaysResponse {
    val input = SetShowFlexOverlaysRequest(flexNodeHighlightConfigs)
    return setShowFlexOverlays(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowScrollSnapOverlays)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowScrollSnapOverlays(input: SetShowScrollSnapOverlaysRequest):
      SetShowScrollSnapOverlaysResponse = session.request("Overlay.setShowScrollSnapOverlays",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowScrollSnapOverlays)
   *
   * @param scrollSnapHighlightConfigs An array of node identifiers and descriptors for the
   * highlight appearance.
   */
  public suspend
      fun setShowScrollSnapOverlays(scrollSnapHighlightConfigs: List<ScrollSnapHighlightConfig>):
      SetShowScrollSnapOverlaysResponse {
    val input = SetShowScrollSnapOverlaysRequest(scrollSnapHighlightConfigs)
    return setShowScrollSnapOverlays(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowContainerQueryOverlays)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowContainerQueryOverlays(input: SetShowContainerQueryOverlaysRequest):
      SetShowContainerQueryOverlaysResponse =
      session.request("Overlay.setShowContainerQueryOverlays", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowContainerQueryOverlays)
   *
   * @param containerQueryHighlightConfigs An array of node identifiers and descriptors for the
   * highlight appearance.
   */
  public suspend
      fun setShowContainerQueryOverlays(containerQueryHighlightConfigs: List<ContainerQueryHighlightConfig>):
      SetShowContainerQueryOverlaysResponse {
    val input = SetShowContainerQueryOverlaysRequest(containerQueryHighlightConfigs)
    return setShowContainerQueryOverlays(input)
  }

  /**
   * Requests that backend shows paint rectangles
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowPaintRects)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowPaintRects(input: SetShowPaintRectsRequest): SetShowPaintRectsResponse =
      session.request("Overlay.setShowPaintRects", input)

  /**
   * Requests that backend shows paint rectangles
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowPaintRects)
   *
   * @param result True for showing paint rectangles
   */
  public suspend fun setShowPaintRects(result: Boolean): SetShowPaintRectsResponse {
    val input = SetShowPaintRectsRequest(result)
    return setShowPaintRects(input)
  }

  /**
   * Requests that backend shows layout shift regions
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowLayoutShiftRegions)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowLayoutShiftRegions(input: SetShowLayoutShiftRegionsRequest):
      SetShowLayoutShiftRegionsResponse = session.request("Overlay.setShowLayoutShiftRegions",
      input)

  /**
   * Requests that backend shows layout shift regions
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowLayoutShiftRegions)
   *
   * @param result True for showing layout shift regions
   */
  public suspend fun setShowLayoutShiftRegions(result: Boolean): SetShowLayoutShiftRegionsResponse {
    val input = SetShowLayoutShiftRegionsRequest(result)
    return setShowLayoutShiftRegions(input)
  }

  /**
   * Requests that backend shows scroll bottleneck rects
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowScrollBottleneckRects)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowScrollBottleneckRects(input: SetShowScrollBottleneckRectsRequest):
      SetShowScrollBottleneckRectsResponse = session.request("Overlay.setShowScrollBottleneckRects",
      input)

  /**
   * Requests that backend shows scroll bottleneck rects
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowScrollBottleneckRects)
   *
   * @param show True for showing scroll bottleneck rects
   */
  public suspend fun setShowScrollBottleneckRects(show: Boolean):
      SetShowScrollBottleneckRectsResponse {
    val input = SetShowScrollBottleneckRectsRequest(show)
    return setShowScrollBottleneckRects(input)
  }

  /**
   * Deprecated, no longer has any effect.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowHitTestBorders)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun setShowHitTestBorders(input: SetShowHitTestBordersRequest):
      SetShowHitTestBordersResponse = session.request("Overlay.setShowHitTestBorders", input)

  /**
   * Deprecated, no longer has any effect.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowHitTestBorders)
   *
   * @param show True for showing hit-test borders
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun setShowHitTestBorders(show: Boolean): SetShowHitTestBordersResponse {
    val input = SetShowHitTestBordersRequest(show)
    return setShowHitTestBorders(input)
  }

  /**
   * Request that backend shows an overlay with web vital metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowWebVitals)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowWebVitals(input: SetShowWebVitalsRequest): SetShowWebVitalsResponse =
      session.request("Overlay.setShowWebVitals", input)

  /**
   * Request that backend shows an overlay with web vital metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowWebVitals)
   */
  public suspend fun setShowWebVitals(show: Boolean): SetShowWebVitalsResponse {
    val input = SetShowWebVitalsRequest(show)
    return setShowWebVitals(input)
  }

  /**
   * Paints viewport size upon main frame resize.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowViewportSizeOnResize)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowViewportSizeOnResize(input: SetShowViewportSizeOnResizeRequest):
      SetShowViewportSizeOnResizeResponse = session.request("Overlay.setShowViewportSizeOnResize",
      input)

  /**
   * Paints viewport size upon main frame resize.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowViewportSizeOnResize)
   *
   * @param show Whether to paint size or not.
   */
  public suspend fun setShowViewportSizeOnResize(show: Boolean):
      SetShowViewportSizeOnResizeResponse {
    val input = SetShowViewportSizeOnResizeRequest(show)
    return setShowViewportSizeOnResize(input)
  }

  /**
   * Add a dual screen device hinge
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowHinge)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowHinge(input: SetShowHingeRequest): SetShowHingeResponse =
      session.request("Overlay.setShowHinge", input)

  /**
   * Add a dual screen device hinge
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowHinge)
   */
  @JvmOverloads
  public suspend inline fun setShowHinge(optionalArgs: SetShowHingeRequest.Builder.() -> Unit = {}):
      SetShowHingeResponse {
    val builder = SetShowHingeRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return setShowHinge(input)
  }

  /**
   * Show elements in isolation mode with overlays.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowIsolatedElements)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setShowIsolatedElements(input: SetShowIsolatedElementsRequest):
      SetShowIsolatedElementsResponse = session.request("Overlay.setShowIsolatedElements", input)

  /**
   * Show elements in isolation mode with overlays.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowIsolatedElements)
   *
   * @param isolatedElementHighlightConfigs An array of node identifiers and descriptors for the
   * highlight appearance.
   */
  public suspend
      fun setShowIsolatedElements(isolatedElementHighlightConfigs: List<IsolatedElementHighlightConfig>):
      SetShowIsolatedElementsResponse {
    val input = SetShowIsolatedElementsRequest(isolatedElementHighlightConfigs)
    return setShowIsolatedElements(input)
  }
}
