@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.page

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.ResourceType
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.runtime.ScriptId
import org.hildan.chrome.devtools.domains.runtime.UniqueDebuggerId
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Unique frame identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameId)
 */
public typealias FrameId = String

/**
 * Indicates whether a frame has been identified as an ad.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AdFrameType)
 */
@ExperimentalChromeApi
@Serializable
public enum class AdFrameType {
  none,
  child,
  root,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AdFrameExplanation)
 */
@ExperimentalChromeApi
@Serializable
public enum class AdFrameExplanation {
  ParentIsAd,
  CreatedByAdScript,
  MatchedBlockingRule,
}

/**
 * Indicates whether a frame has been identified as an ad and why.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AdFrameStatus)
 */
@ExperimentalChromeApi
@Serializable
public data class AdFrameStatus(
  public val adFrameType: AdFrameType,
  public val explanations: List<AdFrameExplanation>? = null,
)

/**
 * Identifies the bottom-most script which caused the frame to be labelled
 * as an ad.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AdScriptId)
 */
@ExperimentalChromeApi
@Serializable
public data class AdScriptId(
  /**
   * Script Id of the bottom-most script which caused the frame to be labelled
   * as an ad.
   */
  public val scriptId: ScriptId,
  /**
   * Id of adScriptId's debugger.
   */
  public val debuggerId: UniqueDebuggerId,
)

/**
 * Indicates whether the frame is a secure context and why it is the case.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-SecureContextType)
 */
@ExperimentalChromeApi
@Serializable
public enum class SecureContextType {
  Secure,
  SecureLocalhost,
  InsecureScheme,
  InsecureAncestor,
}

/**
 * Indicates whether the frame is cross-origin isolated and why it is the case.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-CrossOriginIsolatedContextType)
 */
@ExperimentalChromeApi
@Serializable
public enum class CrossOriginIsolatedContextType {
  Isolated,
  NotIsolated,
  NotIsolatedFeatureDisabled,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-GatedAPIFeatures)
 */
@ExperimentalChromeApi
@Serializable
public enum class GatedAPIFeatures {
  SharedArrayBuffers,
  SharedArrayBuffersTransferAllowed,
  PerformanceMeasureMemory,
  PerformanceProfile,
}

/**
 * All Permissions Policy features. This enum should match the one defined
 * in third_party/blink/renderer/core/permissions_policy/permissions_policy_features.json5.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-PermissionsPolicyFeature)
 */
@ExperimentalChromeApi
@Serializable
public enum class PermissionsPolicyFeature {
  accelerometer,
  `ambient-light-sensor`,
  `attribution-reporting`,
  autoplay,
  bluetooth,
  `browsing-topics`,
  camera,
  `ch-dpr`,
  `ch-device-memory`,
  `ch-downlink`,
  `ch-ect`,
  `ch-prefers-color-scheme`,
  `ch-prefers-reduced-motion`,
  `ch-rtt`,
  `ch-save-data`,
  `ch-ua`,
  `ch-ua-arch`,
  `ch-ua-bitness`,
  `ch-ua-platform`,
  `ch-ua-model`,
  `ch-ua-mobile`,
  `ch-ua-full`,
  `ch-ua-full-version`,
  `ch-ua-full-version-list`,
  `ch-ua-platform-version`,
  `ch-ua-reduced`,
  `ch-ua-wow64`,
  `ch-viewport-height`,
  `ch-viewport-width`,
  `ch-width`,
  `clipboard-read`,
  `clipboard-write`,
  `compute-pressure`,
  `cross-origin-isolated`,
  `direct-sockets`,
  `display-capture`,
  `document-domain`,
  `encrypted-media`,
  `execution-while-out-of-viewport`,
  `execution-while-not-rendered`,
  `focus-without-user-activation`,
  fullscreen,
  frobulate,
  gamepad,
  geolocation,
  gyroscope,
  hid,
  `identity-credentials-get`,
  `idle-detection`,
  `interest-cohort`,
  `join-ad-interest-group`,
  `keyboard-map`,
  `local-fonts`,
  magnetometer,
  microphone,
  midi,
  `otp-credentials`,
  payment,
  `picture-in-picture`,
  `publickey-credentials-get`,
  `run-ad-auction`,
  `screen-wake-lock`,
  serial,
  `shared-autofill`,
  `shared-storage`,
  `storage-access`,
  `sync-xhr`,
  `trust-token-redemption`,
  unload,
  usb,
  `vertical-scroll`,
  `web-share`,
  `window-placement`,
  `xr-spatial-tracking`,
}

/**
 * Reason for a permissions policy feature to be disabled.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-PermissionsPolicyBlockReason)
 */
@ExperimentalChromeApi
@Serializable
public enum class PermissionsPolicyBlockReason {
  Header,
  IframeAttribute,
  InFencedFrameTree,
  InIsolatedApp,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-PermissionsPolicyBlockLocator)
 */
@ExperimentalChromeApi
@Serializable
public data class PermissionsPolicyBlockLocator(
  public val frameId: FrameId,
  public val blockReason: PermissionsPolicyBlockReason,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-PermissionsPolicyFeatureState)
 */
@ExperimentalChromeApi
@Serializable
public data class PermissionsPolicyFeatureState(
  public val feature: PermissionsPolicyFeature,
  public val allowed: Boolean,
  public val locator: PermissionsPolicyBlockLocator? = null,
)

/**
 * Origin Trial(https://www.chromium.org/blink/origin-trials) support.
 * Status for an Origin Trial token.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-OriginTrialTokenStatus)
 */
@ExperimentalChromeApi
@Serializable
public enum class OriginTrialTokenStatus {
  Success,
  NotSupported,
  Insecure,
  Expired,
  WrongOrigin,
  InvalidSignature,
  Malformed,
  WrongVersion,
  FeatureDisabled,
  TokenDisabled,
  FeatureDisabledForUser,
  UnknownTrial,
}

/**
 * Status for an Origin Trial.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-OriginTrialStatus)
 */
@ExperimentalChromeApi
@Serializable
public enum class OriginTrialStatus {
  Enabled,
  ValidTokenNotProvided,
  OSNotSupported,
  TrialNotAllowed,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-OriginTrialUsageRestriction)
 */
@ExperimentalChromeApi
@Serializable
public enum class OriginTrialUsageRestriction {
  None,
  Subset,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-OriginTrialToken)
 */
@ExperimentalChromeApi
@Serializable
public data class OriginTrialToken(
  public val origin: String,
  public val matchSubDomains: Boolean,
  public val trialName: String,
  public val expiryTime: TimeSinceEpoch,
  public val isThirdParty: Boolean,
  public val usageRestriction: OriginTrialUsageRestriction,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-OriginTrialTokenWithStatus)
 */
@ExperimentalChromeApi
@Serializable
public data class OriginTrialTokenWithStatus(
  public val rawTokenText: String,
  /**
   * `parsedToken` is present only when the token is extractable and
   * parsable.
   */
  public val parsedToken: OriginTrialToken? = null,
  public val status: OriginTrialTokenStatus,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-OriginTrial)
 */
@ExperimentalChromeApi
@Serializable
public data class OriginTrial(
  public val trialName: String,
  public val status: OriginTrialStatus,
  public val tokensWithStatus: List<OriginTrialTokenWithStatus>,
)

/**
 * Information about the Frame on the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-Frame)
 */
@Serializable
public data class Frame(
  /**
   * Frame unique identifier.
   */
  public val id: FrameId,
  /**
   * Parent frame identifier.
   */
  public val parentId: FrameId? = null,
  /**
   * Identifier of the loader associated with this frame.
   */
  public val loaderId: LoaderId,
  /**
   * Frame's name as specified in the tag.
   */
  public val name: String? = null,
  /**
   * Frame document's URL without fragment.
   */
  public val url: String,
  /**
   * Frame document's URL fragment including the '#'.
   */
  @ExperimentalChromeApi
  public val urlFragment: String? = null,
  /**
   * Frame document's registered domain, taking the public suffixes list into account.
   * Extracted from the Frame's url.
   * Example URLs: http://www.google.com/file.html -> "google.com"
   *               http://a.b.co.uk/file.html      -> "b.co.uk"
   */
  @ExperimentalChromeApi
  public val domainAndRegistry: String? = null,
  /**
   * Frame document's security origin.
   */
  public val securityOrigin: String,
  /**
   * Frame document's mimeType as determined by the browser.
   */
  public val mimeType: String,
  /**
   * If the frame failed to load, this contains the URL that could not be loaded. Note that unlike
   * url above, this URL may contain a fragment.
   */
  @ExperimentalChromeApi
  public val unreachableUrl: String? = null,
  /**
   * Indicates whether this frame was tagged as an ad and why.
   */
  @ExperimentalChromeApi
  public val adFrameStatus: AdFrameStatus? = null,
  /**
   * Indicates whether the main document is a secure context and explains why that is the case.
   */
  @ExperimentalChromeApi
  public val secureContextType: SecureContextType? = null,
  /**
   * Indicates whether this is a cross origin isolated context.
   */
  @ExperimentalChromeApi
  public val crossOriginIsolatedContextType: CrossOriginIsolatedContextType? = null,
  /**
   * Indicated which gated APIs / features are available.
   */
  @ExperimentalChromeApi
  public val gatedAPIFeatures: List<GatedAPIFeatures>? = null,
)

/**
 * Information about the Resource on the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameResource)
 */
@ExperimentalChromeApi
@Serializable
public data class FrameResource(
  /**
   * Resource URL.
   */
  public val url: String,
  /**
   * Type of this resource.
   */
  public val type: ResourceType,
  /**
   * Resource mimeType as determined by the browser.
   */
  public val mimeType: String,
  /**
   * last-modified timestamp as reported by server.
   */
  public val lastModified: TimeSinceEpoch? = null,
  /**
   * Resource content size.
   */
  public val contentSize: Double? = null,
  /**
   * True if the resource failed to load.
   */
  public val failed: Boolean? = null,
  /**
   * True if the resource was canceled during loading.
   */
  public val canceled: Boolean? = null,
)

/**
 * Information about the Frame hierarchy along with their cached resources.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameResourceTree)
 */
@ExperimentalChromeApi
@Serializable
public data class FrameResourceTree(
  /**
   * Frame information for this tree item.
   */
  public val frame: Frame,
  /**
   * Child frames.
   */
  public val childFrames: List<FrameResourceTree>? = null,
  /**
   * Information about frame resources.
   */
  public val resources: List<FrameResource>,
)

/**
 * Information about the Frame hierarchy.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameTree)
 */
@Serializable
public data class FrameTree(
  /**
   * Frame information for this tree item.
   */
  public val frame: Frame,
  /**
   * Child frames.
   */
  public val childFrames: List<FrameTree>? = null,
)

/**
 * Unique script identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScriptIdentifier)
 */
public typealias ScriptIdentifier = String

/**
 * Transition type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-TransitionType)
 */
@Serializable
public enum class TransitionType {
  link,
  typed,
  address_bar,
  auto_bookmark,
  auto_subframe,
  manual_subframe,
  generated,
  auto_toplevel,
  form_submit,
  reload,
  keyword,
  keyword_generated,
  other,
}

/**
 * Navigation history entry.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-NavigationEntry)
 */
@Serializable
public data class NavigationEntry(
  /**
   * Unique id of the navigation history entry.
   */
  public val id: Int,
  /**
   * URL of the navigation history entry.
   */
  public val url: String,
  /**
   * URL that the user typed in the url bar.
   */
  public val userTypedURL: String,
  /**
   * Title of the navigation history entry.
   */
  public val title: String,
  /**
   * Transition type.
   */
  public val transitionType: TransitionType,
)

/**
 * Screencast frame metadata.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScreencastFrameMetadata)
 */
@ExperimentalChromeApi
@Serializable
public data class ScreencastFrameMetadata(
  /**
   * Top offset in DIP.
   */
  public val offsetTop: Double,
  /**
   * Page scale factor.
   */
  public val pageScaleFactor: Double,
  /**
   * Device screen width in DIP.
   */
  public val deviceWidth: Double,
  /**
   * Device screen height in DIP.
   */
  public val deviceHeight: Double,
  /**
   * Position of horizontal scroll in CSS pixels.
   */
  public val scrollOffsetX: Double,
  /**
   * Position of vertical scroll in CSS pixels.
   */
  public val scrollOffsetY: Double,
  /**
   * Frame swap timestamp.
   */
  public val timestamp: TimeSinceEpoch? = null,
)

/**
 * Javascript dialog type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-DialogType)
 */
@Serializable
public enum class DialogType {
  alert,
  confirm,
  prompt,
  beforeunload,
}

/**
 * Error while paring app manifest.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AppManifestError)
 */
@Serializable
public data class AppManifestError(
  /**
   * Error message.
   */
  public val message: String,
  /**
   * If criticial, this is a non-recoverable parse error.
   */
  public val critical: Int,
  /**
   * Error line.
   */
  public val line: Int,
  /**
   * Error column.
   */
  public val column: Int,
)

/**
 * Parsed app manifest properties.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AppManifestParsedProperties)
 */
@ExperimentalChromeApi
@Serializable
public data class AppManifestParsedProperties(
  /**
   * Computed scope value
   */
  public val scope: String,
)

/**
 * Layout viewport position and dimensions.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-LayoutViewport)
 */
@Serializable
public data class LayoutViewport(
  /**
   * Horizontal offset relative to the document (CSS pixels).
   */
  public val pageX: Int,
  /**
   * Vertical offset relative to the document (CSS pixels).
   */
  public val pageY: Int,
  /**
   * Width (CSS pixels), excludes scrollbar if present.
   */
  public val clientWidth: Int,
  /**
   * Height (CSS pixels), excludes scrollbar if present.
   */
  public val clientHeight: Int,
)

/**
 * Visual viewport position, dimensions, and scale.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-VisualViewport)
 */
@Serializable
public data class VisualViewport(
  /**
   * Horizontal offset relative to the layout viewport (CSS pixels).
   */
  public val offsetX: Double,
  /**
   * Vertical offset relative to the layout viewport (CSS pixels).
   */
  public val offsetY: Double,
  /**
   * Horizontal offset relative to the document (CSS pixels).
   */
  public val pageX: Double,
  /**
   * Vertical offset relative to the document (CSS pixels).
   */
  public val pageY: Double,
  /**
   * Width (CSS pixels), excludes scrollbar if present.
   */
  public val clientWidth: Double,
  /**
   * Height (CSS pixels), excludes scrollbar if present.
   */
  public val clientHeight: Double,
  /**
   * Scale relative to the ideal viewport (size at width=device-width).
   */
  public val scale: Double,
  /**
   * Page zoom factor (CSS to device independent pixels ratio).
   */
  public val zoom: Double? = null,
)

/**
 * Viewport for capturing screenshot.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-Viewport)
 */
@Serializable
public data class Viewport(
  /**
   * X offset in device independent pixels (dip).
   */
  public val x: Double,
  /**
   * Y offset in device independent pixels (dip).
   */
  public val y: Double,
  /**
   * Rectangle width in device independent pixels (dip).
   */
  public val width: Double,
  /**
   * Rectangle height in device independent pixels (dip).
   */
  public val height: Double,
  /**
   * Page scale factor.
   */
  public val scale: Double,
)

/**
 * Generic font families collection.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FontFamilies)
 */
@ExperimentalChromeApi
@Serializable
public data class FontFamilies(
  /**
   * The standard font-family.
   */
  public val standard: String? = null,
  /**
   * The fixed font-family.
   */
  public val fixed: String? = null,
  /**
   * The serif font-family.
   */
  public val serif: String? = null,
  /**
   * The sansSerif font-family.
   */
  public val sansSerif: String? = null,
  /**
   * The cursive font-family.
   */
  public val cursive: String? = null,
  /**
   * The fantasy font-family.
   */
  public val fantasy: String? = null,
  /**
   * The math font-family.
   */
  public val math: String? = null,
)

/**
 * Font families collection for a script.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScriptFontFamilies)
 */
@ExperimentalChromeApi
@Serializable
public data class ScriptFontFamilies(
  /**
   * Name of the script which these font families are defined for.
   */
  public val script: String,
  /**
   * Generic font families collection for the script.
   */
  public val fontFamilies: FontFamilies,
)

/**
 * Default font sizes.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FontSizes)
 */
@ExperimentalChromeApi
@Serializable
public data class FontSizes(
  /**
   * Default standard font size.
   */
  public val standard: Int? = null,
  /**
   * Default fixed font size.
   */
  public val fixed: Int? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ClientNavigationReason)
 */
@ExperimentalChromeApi
@Serializable
public enum class ClientNavigationReason {
  formSubmissionGet,
  formSubmissionPost,
  httpHeaderRefresh,
  scriptInitiated,
  metaTagRefresh,
  pageBlockInterstitial,
  reload,
  anchorClick,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ClientNavigationDisposition)
 */
@ExperimentalChromeApi
@Serializable
public enum class ClientNavigationDisposition {
  currentTab,
  newTab,
  newWindow,
  download,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-InstallabilityErrorArgument)
 */
@ExperimentalChromeApi
@Serializable
public data class InstallabilityErrorArgument(
  /**
   * Argument name (e.g. name:'minimum-icon-size-in-pixels').
   */
  public val name: String,
  /**
   * Argument value (e.g. value:'64').
   */
  public val `value`: String,
)

/**
 * The installability error
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-InstallabilityError)
 */
@ExperimentalChromeApi
@Serializable
public data class InstallabilityError(
  /**
   * The error id (e.g. 'manifest-missing-suitable-icon').
   */
  public val errorId: String,
  /**
   * The list of error arguments (e.g. {name:'minimum-icon-size-in-pixels', value:'64'}).
   */
  public val errorArguments: List<InstallabilityErrorArgument>,
)

/**
 * The referring-policy used for the navigation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ReferrerPolicy)
 */
@ExperimentalChromeApi
@Serializable
public enum class ReferrerPolicy {
  noReferrer,
  noReferrerWhenDowngrade,
  origin,
  originWhenCrossOrigin,
  sameOrigin,
  strictOrigin,
  strictOriginWhenCrossOrigin,
  unsafeUrl,
}

/**
 * Per-script compilation cache parameters for `Page.produceCompilationCache`
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-CompilationCacheParams)
 */
@ExperimentalChromeApi
@Serializable
public data class CompilationCacheParams(
  /**
   * The URL of the script to produce a compilation cache entry for.
   */
  public val url: String,
  /**
   * A hint to the backend whether eager compilation is recommended.
   * (the actual compilation mode used is upon backend discretion).
   */
  public val eager: Boolean? = null,
)

/**
 * The type of a frameNavigated event.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-NavigationType)
 */
@ExperimentalChromeApi
@Serializable
public enum class NavigationType {
  Navigation,
  BackForwardCacheRestore,
}

/**
 * List of not restored reasons for back-forward cache.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-BackForwardCacheNotRestoredReason)
 */
@ExperimentalChromeApi
@Serializable
public enum class BackForwardCacheNotRestoredReason {
  NotPrimaryMainFrame,
  BackForwardCacheDisabled,
  RelatedActiveContentsExist,
  HTTPStatusNotOK,
  SchemeNotHTTPOrHTTPS,
  Loading,
  WasGrantedMediaAccess,
  DisableForRenderFrameHostCalled,
  DomainNotAllowed,
  HTTPMethodNotGET,
  SubframeIsNavigating,
  Timeout,
  CacheLimit,
  JavaScriptExecution,
  RendererProcessKilled,
  RendererProcessCrashed,
  SchedulerTrackedFeatureUsed,
  ConflictingBrowsingInstance,
  CacheFlushed,
  ServiceWorkerVersionActivation,
  SessionRestored,
  ServiceWorkerPostMessage,
  EnteredBackForwardCacheBeforeServiceWorkerHostAdded,
  RenderFrameHostReused_SameSite,
  RenderFrameHostReused_CrossSite,
  ServiceWorkerClaim,
  IgnoreEventAndEvict,
  HaveInnerContents,
  TimeoutPuttingInCache,
  BackForwardCacheDisabledByLowMemory,
  BackForwardCacheDisabledByCommandLine,
  NetworkRequestDatapipeDrainedAsBytesConsumer,
  NetworkRequestRedirected,
  NetworkRequestTimeout,
  NetworkExceedsBufferLimit,
  NavigationCancelledWhileRestoring,
  NotMostRecentNavigationEntry,
  BackForwardCacheDisabledForPrerender,
  UserAgentOverrideDiffers,
  ForegroundCacheLimit,
  BrowsingInstanceNotSwapped,
  BackForwardCacheDisabledForDelegate,
  UnloadHandlerExistsInMainFrame,
  UnloadHandlerExistsInSubFrame,
  ServiceWorkerUnregistration,
  CacheControlNoStore,
  CacheControlNoStoreCookieModified,
  CacheControlNoStoreHTTPOnlyCookieModified,
  NoResponseHead,
  Unknown,
  ActivationNavigationsDisallowedForBug1234857,
  ErrorDocument,
  FencedFramesEmbedder,
  WebSocket,
  WebTransport,
  WebRTC,
  MainResourceHasCacheControlNoStore,
  MainResourceHasCacheControlNoCache,
  SubresourceHasCacheControlNoStore,
  SubresourceHasCacheControlNoCache,
  ContainsPlugins,
  DocumentLoaded,
  DedicatedWorkerOrWorklet,
  OutstandingNetworkRequestOthers,
  OutstandingIndexedDBTransaction,
  RequestedMIDIPermission,
  RequestedAudioCapturePermission,
  RequestedVideoCapturePermission,
  RequestedBackForwardCacheBlockedSensors,
  RequestedBackgroundWorkPermission,
  BroadcastChannel,
  IndexedDBConnection,
  WebXR,
  SharedWorker,
  WebLocks,
  WebHID,
  WebShare,
  RequestedStorageAccessGrant,
  WebNfc,
  OutstandingNetworkRequestFetch,
  OutstandingNetworkRequestXHR,
  AppBanner,
  Printing,
  WebDatabase,
  PictureInPicture,
  Portal,
  SpeechRecognizer,
  IdleManager,
  PaymentManager,
  SpeechSynthesis,
  KeyboardLock,
  WebOTPService,
  OutstandingNetworkRequestDirectSocket,
  InjectedJavascript,
  InjectedStyleSheet,
  KeepaliveRequest,
  Dummy,
  ContentSecurityHandler,
  ContentWebAuthenticationAPI,
  ContentFileChooser,
  ContentSerial,
  ContentFileSystemAccess,
  ContentMediaDevicesDispatcherHost,
  ContentWebBluetooth,
  ContentWebUSB,
  ContentMediaSessionService,
  ContentScreenReader,
  EmbedderPopupBlockerTabHelper,
  EmbedderSafeBrowsingTriggeredPopupBlocker,
  EmbedderSafeBrowsingThreatDetails,
  EmbedderAppBannerManager,
  EmbedderDomDistillerViewerSource,
  EmbedderDomDistillerSelfDeletingRequestDelegate,
  EmbedderOomInterventionTabHelper,
  EmbedderOfflinePage,
  EmbedderChromePasswordManagerClientBindCredentialManager,
  EmbedderPermissionRequestManager,
  EmbedderModalDialog,
  EmbedderExtensions,
  EmbedderExtensionMessaging,
  EmbedderExtensionMessagingForOpenPort,
  EmbedderExtensionSentMessageToCachedFrame,
}

/**
 * Types of not restored reasons for back-forward cache.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-BackForwardCacheNotRestoredReasonType)
 */
@ExperimentalChromeApi
@Serializable
public enum class BackForwardCacheNotRestoredReasonType {
  SupportPending,
  PageSupportNeeded,
  Circumstantial,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-BackForwardCacheNotRestoredExplanation)
 */
@ExperimentalChromeApi
@Serializable
public data class BackForwardCacheNotRestoredExplanation(
  /**
   * Type of the reason
   */
  public val type: BackForwardCacheNotRestoredReasonType,
  /**
   * Not restored reason
   */
  public val reason: BackForwardCacheNotRestoredReason,
  /**
   * Context associated with the reason. The meaning of this context is
   * dependent on the reason:
   * - EmbedderExtensionSentMessageToCachedFrame: the extension ID.
   */
  public val context: String? = null,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-BackForwardCacheNotRestoredExplanationTree)
 */
@ExperimentalChromeApi
@Serializable
public data class BackForwardCacheNotRestoredExplanationTree(
  /**
   * URL of each frame
   */
  public val url: String,
  /**
   * Not restored reasons of each frame
   */
  public val explanations: List<BackForwardCacheNotRestoredExplanation>,
  /**
   * Array of children frame
   */
  public val children: List<BackForwardCacheNotRestoredExplanationTree>,
)

/**
 * List of FinalStatus reasons for Prerender2.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-PrerenderFinalStatus)
 */
@Serializable
public enum class PrerenderFinalStatus {
  Activated,
  Destroyed,
  LowEndDevice,
  InvalidSchemeRedirect,
  InvalidSchemeNavigation,
  InProgressNavigation,
  NavigationRequestBlockedByCsp,
  MainFrameNavigation,
  MojoBinderPolicy,
  RendererProcessCrashed,
  RendererProcessKilled,
  Download,
  TriggerDestroyed,
  NavigationNotCommitted,
  NavigationBadHttpStatus,
  ClientCertRequested,
  NavigationRequestNetworkError,
  MaxNumOfRunningPrerendersExceeded,
  CancelAllHostsForTesting,
  DidFailLoad,
  Stop,
  SslCertificateError,
  LoginAuthRequested,
  UaChangeRequiresReload,
  BlockedByClient,
  AudioOutputDeviceRequested,
  MixedContent,
  TriggerBackgrounded,
  EmbedderTriggeredAndCrossOriginRedirected,
  MemoryLimitExceeded,
  FailToGetMemoryUsage,
  DataSaverEnabled,
  HasEffectiveUrl,
  ActivatedBeforeStarted,
  InactivePageRestriction,
  StartFailed,
  TimeoutBackgrounded,
  CrossSiteRedirect,
  CrossSiteNavigation,
  SameSiteCrossOriginRedirect,
  SameSiteCrossOriginNavigation,
  SameSiteCrossOriginRedirectNotOptIn,
  SameSiteCrossOriginNavigationNotOptIn,
  ActivationNavigationParameterMismatch,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'format' of the command 'Page.captureScreenshot'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScreenshotFormat)
 */
@Serializable
public enum class ScreenshotFormat {
  jpeg,
  png,
  webp,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'format' of the command 'Page.captureSnapshot'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-SnapshotFormat)
 */
@Serializable
public enum class SnapshotFormat {
  mhtml,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'transferMode' of the command 'Page.printToPDF'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-TransferMode)
 */
@Serializable
public enum class TransferMode {
  ReturnAsBase64,
  ReturnAsStream,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'behavior' of the command 'Page.setDownloadBehavior'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-DownloadBehavior)
 */
@Serializable
public enum class DownloadBehavior {
  deny,
  allow,
  default,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'configuration' of the command 'Page.setTouchEmulationEnabled'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-TouchEmulationEnabledConfiguration)
 */
@Serializable
public enum class TouchEmulationEnabledConfiguration {
  mobile,
  desktop,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'format' of the command 'Page.startScreencast'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScreencastFormat)
 */
@Serializable
public enum class ScreencastFormat {
  jpeg,
  png,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'state' of the command 'Page.setWebLifecycleState'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-WebLifecycleState)
 */
@Serializable
public enum class WebLifecycleState {
  frozen,
  active,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'mode' of the command 'Page.setSPCTransactionMode'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-SPCTransactionMode)
 */
@Serializable
public enum class SPCTransactionMode {
  none,
  autoaccept,
  autoreject,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'mode' of the event 'Page.fileChooserOpened'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FileChooserOpenedMode)
 */
@Serializable
public enum class FileChooserOpenedMode {
  selectSingle,
  selectMultiple,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'reason' of the event 'Page.frameDetached'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameDetachedReason)
 */
@Serializable
public enum class FrameDetachedReason {
  remove,
  swap,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'state' of the event 'Page.downloadProgress'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-DownloadProgressState)
 */
@Serializable
public enum class DownloadProgressState {
  inProgress,
  completed,
  canceled,
}
