@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.page.events

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.MonotonicTime
import org.hildan.chrome.devtools.domains.page.BackForwardCacheNotRestoredExplanation
import org.hildan.chrome.devtools.domains.page.BackForwardCacheNotRestoredExplanationTree
import org.hildan.chrome.devtools.domains.page.ClientNavigationDisposition
import org.hildan.chrome.devtools.domains.page.ClientNavigationReason
import org.hildan.chrome.devtools.domains.page.DialogType
import org.hildan.chrome.devtools.domains.page.DownloadProgressState
import org.hildan.chrome.devtools.domains.page.FileChooserOpenedMode
import org.hildan.chrome.devtools.domains.page.Frame
import org.hildan.chrome.devtools.domains.page.FrameDetachedReason
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.page.NavigationType
import org.hildan.chrome.devtools.domains.page.PrerenderFinalStatus
import org.hildan.chrome.devtools.domains.page.ScreencastFrameMetadata
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class PageEvent {
  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-domContentEventFired)
   */
  @Serializable
  public data class DomContentEventFired(
    public val timestamp: MonotonicTime,
  ) : PageEvent()

  /**
   * Emitted only when `page.interceptFileChooser` is enabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-fileChooserOpened)
   */
  @Serializable
  public data class FileChooserOpened(
    /**
     * Id of the frame containing input node.
     */
    @ExperimentalChromeApi
    public val frameId: FrameId? = null,
    /**
     * Input mode.
     */
    public val mode: FileChooserOpenedMode,
    /**
     * Input node id. Only present for file choosers opened via an <input type="file"> element.
     */
    @ExperimentalChromeApi
    public val backendNodeId: BackendNodeId? = null,
  ) : PageEvent()

  /**
   * Fired when frame has been attached to its parent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameAttached)
   */
  @Serializable
  public data class FrameAttached(
    /**
     * Id of the frame that has been attached.
     */
    public val frameId: FrameId,
    /**
     * Parent frame identifier.
     */
    public val parentFrameId: FrameId,
    /**
     * JavaScript stack trace of when frame was attached, only set if frame initiated from script.
     */
    public val stack: StackTrace? = null,
  ) : PageEvent()

  /**
   * Fired when frame no longer has a scheduled navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameClearedScheduledNavigation)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @Serializable
  public data class FrameClearedScheduledNavigation(
    /**
     * Id of the frame that has cleared its scheduled navigation.
     */
    public val frameId: FrameId,
  ) : PageEvent()

  /**
   * Fired when frame has been detached from its parent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameDetached)
   */
  @Serializable
  public data class FrameDetached(
    /**
     * Id of the frame that has been detached.
     */
    public val frameId: FrameId,
    @ExperimentalChromeApi
    public val reason: FrameDetachedReason? = null,
  ) : PageEvent()

  /**
   * Fired once navigation of the frame has completed. Frame is now associated with the new loader.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameNavigated)
   */
  @Serializable
  public data class FrameNavigated(
    /**
     * Frame object.
     */
    public val frame: Frame,
    @ExperimentalChromeApi
    public val type: NavigationType? = null,
  ) : PageEvent()

  /**
   * Fired when opening document to write to.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-documentOpened)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class DocumentOpened(
    /**
     * Frame object.
     */
    public val frame: Frame,
  ) : PageEvent()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameResized)
   */
  @ExperimentalChromeApi
  @Serializable
  public object FrameResized : PageEvent()

  /**
   * Fired when a renderer-initiated navigation is requested.
   * Navigation may still be cancelled after the event is issued.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameRequestedNavigation)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class FrameRequestedNavigation(
    /**
     * Id of the frame that is being navigated.
     */
    public val frameId: FrameId,
    /**
     * The reason for the navigation.
     */
    public val reason: ClientNavigationReason,
    /**
     * The destination URL for the requested navigation.
     */
    public val url: String,
    /**
     * The disposition for the navigation.
     */
    public val disposition: ClientNavigationDisposition,
  ) : PageEvent()

  /**
   * Fired when frame schedules a potential navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameScheduledNavigation)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @Serializable
  public data class FrameScheduledNavigation(
    /**
     * Id of the frame that has scheduled a navigation.
     */
    public val frameId: FrameId,
    /**
     * Delay (in seconds) until the navigation is scheduled to begin. The navigation is not
     * guaranteed to start.
     */
    public val delay: Double,
    /**
     * The reason for the navigation.
     */
    public val reason: ClientNavigationReason,
    /**
     * The destination URL for the scheduled navigation.
     */
    public val url: String,
  ) : PageEvent()

  /**
   * Fired when frame has started loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameStartedLoading)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class FrameStartedLoading(
    /**
     * Id of the frame that has started loading.
     */
    public val frameId: FrameId,
  ) : PageEvent()

  /**
   * Fired when frame has stopped loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameStoppedLoading)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class FrameStoppedLoading(
    /**
     * Id of the frame that has stopped loading.
     */
    public val frameId: FrameId,
  ) : PageEvent()

  /**
   * Fired when page is about to start a download.
   * Deprecated. Use Browser.downloadWillBegin instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-downloadWillBegin)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  @Serializable
  public data class DownloadWillBegin(
    /**
     * Id of the frame that caused download to begin.
     */
    public val frameId: FrameId,
    /**
     * Global unique identifier of the download.
     */
    public val guid: String,
    /**
     * URL of the resource being downloaded.
     */
    public val url: String,
    /**
     * Suggested file name of the resource (the actual name of the file saved on disk may differ).
     */
    public val suggestedFilename: String,
  ) : PageEvent()

  /**
   * Fired when download makes progress. Last call has |done| == true.
   * Deprecated. Use Browser.downloadProgress instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-downloadProgress)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  @Serializable
  public data class DownloadProgress(
    /**
     * Global unique identifier of the download.
     */
    public val guid: String,
    /**
     * Total expected bytes to download.
     */
    public val totalBytes: Double,
    /**
     * Total bytes received.
     */
    public val receivedBytes: Double,
    /**
     * Download status.
     */
    public val state: DownloadProgressState,
  ) : PageEvent()

  /**
   * Fired when interstitial page was hidden
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-interstitialHidden)
   */
  @Serializable
  public object InterstitialHidden : PageEvent()

  /**
   * Fired when interstitial page was shown
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-interstitialShown)
   */
  @Serializable
  public object InterstitialShown : PageEvent()

  /**
   * Fired when a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload) has been
   * closed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-javascriptDialogClosed)
   */
  @Serializable
  public data class JavascriptDialogClosed(
    /**
     * Whether dialog was confirmed.
     */
    public val result: Boolean,
    /**
     * User input in case of prompt.
     */
    public val userInput: String,
  ) : PageEvent()

  /**
   * Fired when a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload) is about
   * to
   * open.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-javascriptDialogOpening)
   */
  @Serializable
  public data class JavascriptDialogOpening(
    /**
     * Frame url.
     */
    public val url: String,
    /**
     * Message that will be displayed by the dialog.
     */
    public val message: String,
    /**
     * Dialog type.
     */
    public val type: DialogType,
    /**
     * True iff browser is capable showing or acting on the given dialog. When browser has no
     * dialog handler for given target, calling alert while Page domain is engaged will stall
     * the page execution. Execution can be resumed via calling Page.handleJavaScriptDialog.
     */
    public val hasBrowserHandler: Boolean,
    /**
     * Default dialog prompt.
     */
    public val defaultPrompt: String? = null,
  ) : PageEvent()

  /**
   * Fired for top level page lifecycle events such as navigation, load, paint, etc.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-lifecycleEvent)
   */
  @Serializable
  public data class LifecycleEvent(
    /**
     * Id of the frame.
     */
    public val frameId: FrameId,
    /**
     * Loader identifier. Empty string if the request is fetched from worker.
     */
    public val loaderId: LoaderId,
    public val name: String,
    public val timestamp: MonotonicTime,
  ) : PageEvent()

  /**
   * Fired for failed bfcache history navigations if BackForwardCache feature is enabled. Do
   * not assume any ordering with the Page.frameNavigated event. This event is fired only for
   * main-frame history navigation where the document changes (non-same-document navigations),
   * when bfcache navigation fails.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-backForwardCacheNotUsed)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class BackForwardCacheNotUsed(
    /**
     * The loader id for the associated navgation.
     */
    public val loaderId: LoaderId,
    /**
     * The frame id of the associated frame.
     */
    public val frameId: FrameId,
    /**
     * Array of reasons why the page could not be cached. This must not be empty.
     */
    public val notRestoredExplanations: List<BackForwardCacheNotRestoredExplanation>,
    /**
     * Tree structure of reasons why the page could not be cached for each frame.
     */
    public val notRestoredExplanationsTree: BackForwardCacheNotRestoredExplanationTree? = null,
  ) : PageEvent()

  /**
   * Fired when a prerender attempt is completed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-prerenderAttemptCompleted)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class PrerenderAttemptCompleted(
    /**
     * The frame id of the frame initiating prerendering.
     */
    public val initiatingFrameId: FrameId,
    public val prerenderingUrl: String,
    public val finalStatus: PrerenderFinalStatus,
    /**
     * This is used to give users more information about the name of the API call
     * that is incompatible with prerender and has caused the cancellation of the attempt
     */
    public val disallowedApiMethod: String? = null,
  ) : PageEvent()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-loadEventFired)
   */
  @Serializable
  public data class LoadEventFired(
    public val timestamp: MonotonicTime,
  ) : PageEvent()

  /**
   * Fired when same-document navigation happens, e.g. due to history API usage or anchor
   * navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-navigatedWithinDocument)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class NavigatedWithinDocument(
    /**
     * Id of the frame.
     */
    public val frameId: FrameId,
    /**
     * Frame's new url.
     */
    public val url: String,
  ) : PageEvent()

  /**
   * Compressed image data requested by the `startScreencast`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-screencastFrame)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ScreencastFrame(
    /**
     * Base64-encoded compressed image. (Encoded as a base64 string when passed over JSON)
     */
    public val `data`: String,
    /**
     * Screencast frame metadata.
     */
    public val metadata: ScreencastFrameMetadata,
    /**
     * Frame number.
     */
    public val sessionId: Int,
  ) : PageEvent()

  /**
   * Fired when the page with currently enabled screencast was shown or hidden `.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-screencastVisibilityChanged)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ScreencastVisibilityChanged(
    /**
     * True if the page is visible.
     */
    public val visible: Boolean,
  ) : PageEvent()

  /**
   * Fired when a new window is going to be opened, via window.open(), link click, form submission,
   * etc.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-windowOpen)
   */
  @Serializable
  public data class WindowOpen(
    /**
     * The URL for the new window.
     */
    public val url: String,
    /**
     * Window name.
     */
    public val windowName: String,
    /**
     * An array of enabled window features.
     */
    public val windowFeatures: List<String>,
    /**
     * Whether or not it was triggered by user gesture.
     */
    public val userGesture: Boolean,
  ) : PageEvent()

  /**
   * Issued for every compilation cache generated. Is only available
   * if Page.setGenerateCompilationCache is enabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-compilationCacheProduced)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class CompilationCacheProduced(
    public val url: String,
    /**
     * Base64-encoded data (Encoded as a base64 string when passed over JSON)
     */
    public val `data`: String,
  ) : PageEvent()
}
