@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.performance

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.performance.events.PerformanceEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [PerformanceDomain.disable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [PerformanceDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * Time domain to use for collecting and reporting duration metrics.
   */
  public val timeDomain: TimeDomain? = null,
) {
  public class Builder() {
    /**
     * Time domain to use for collecting and reporting duration metrics.
     */
    public var timeDomain: TimeDomain? = null

    public fun build(): EnableRequest = EnableRequest(timeDomain)
  }
}

/**
 * A dummy response object for the [PerformanceDomain.enable] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [PerformanceDomain.setTimeDomain] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetTimeDomainRequest(
  /**
   * Time domain
   */
  public val timeDomain: TimeDomain,
)

/**
 * A dummy response object for the [PerformanceDomain.setTimeDomain] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public object SetTimeDomainResponse

/**
 * Response type for the [PerformanceDomain.getMetrics] command.
 */
@Serializable
public data class GetMetricsResponse(
  /**
   * Current values for run-time metrics.
   */
  public val metrics: List<Metric>,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance)
 */
public class PerformanceDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out PerformanceEvent>> =
      mapOf(
      "Performance.metrics" to serializer<PerformanceEvent.Metrics>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<PerformanceEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Current values of the metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#event-metrics)
   */
  public fun metricsEvents(): Flow<PerformanceEvent.Metrics> =
      session.typedEvents("Performance.metrics")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("metricsEvents()"),
  )
  public fun metrics(): Flow<PerformanceEvent.Metrics> = metricsEvents()

  /**
   * Disable collecting and reporting metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Performance.disable", Unit)

  /**
   * Enable collecting and reporting metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse =
      session.request("Performance.enable", input)

  /**
   * Enable collecting and reporting metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Sets time domain to use for collecting and reporting duration metrics.
   * Note that this must be called before enabling metrics collection. Calling
   * this method while metrics collection is enabled returns an error.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-setTimeDomain)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setTimeDomain(input: SetTimeDomainRequest): SetTimeDomainResponse =
      session.request("Performance.setTimeDomain", input)

  /**
   * Sets time domain to use for collecting and reporting duration metrics.
   * Note that this must be called before enabling metrics collection. Calling
   * this method while metrics collection is enabled returns an error.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-setTimeDomain)
   *
   * @param timeDomain Time domain
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setTimeDomain(timeDomain: TimeDomain): SetTimeDomainResponse {
    val input = SetTimeDomainRequest(timeDomain)
    return setTimeDomain(input)
  }

  /**
   * Retrieve current values of run-time metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-getMetrics)
   */
  public suspend fun getMetrics(): GetMetricsResponse = session.request("Performance.getMetrics",
      Unit)
}
