@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.performancetimeline

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.performancetimeline.events.PerformanceTimelineEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [PerformanceTimelineDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * The types of event to report, as specified in
   * https://w3c.github.io/performance-timeline/#dom-performanceentry-entrytype
   * The specified filter overrides any previous filters, passing empty
   * filter disables recording.
   * Note that not all types exposed to the web platform are currently supported.
   */
  public val eventTypes: List<String>,
)

/**
 * A dummy response object for the [PerformanceTimelineDomain.enable] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object EnableResponse

/**
 * Reporting of performance timeline events, as specified in
 * https://w3c.github.io/performance-timeline/#dom-performanceobserver.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline)
 */
@ExperimentalChromeApi
public class PerformanceTimelineDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName:
      Map<String, DeserializationStrategy<out PerformanceTimelineEvent>> = mapOf(
      "PerformanceTimeline.timelineEventAdded" to
          serializer<PerformanceTimelineEvent.TimelineEventAdded>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<PerformanceTimelineEvent> =
      session.typedEvents(deserializersByEventName)

  /**
   * Sent when a performance timeline event is added. See reportPerformanceTimeline method.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#event-timelineEventAdded)
   */
  public fun timelineEventAddedEvents(): Flow<PerformanceTimelineEvent.TimelineEventAdded> =
      session.typedEvents("PerformanceTimeline.timelineEventAdded")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("timelineEventAddedEvents()"),
  )
  public fun timelineEventAdded(): Flow<PerformanceTimelineEvent.TimelineEventAdded> =
      timelineEventAddedEvents()

  /**
   * Previously buffered events would be reported before method returns.
   * See also: timelineEventAdded
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse =
      session.request("PerformanceTimeline.enable", input)

  /**
   * Previously buffered events would be reported before method returns.
   * See also: timelineEventAdded
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#method-enable)
   *
   * @param eventTypes The types of event to report, as specified in
   * https://w3c.github.io/performance-timeline/#dom-performanceentry-entrytype
   * The specified filter overrides any previous filters, passing empty
   * filter disables recording.
   * Note that not all types exposed to the web platform are currently supported.
   */
  public suspend fun enable(eventTypes: List<String>): EnableResponse {
    val input = EnableRequest(eventTypes)
    return enable(input)
  }
}
