@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.profiler

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.profiler.events.ProfilerEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [ProfilerDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [ProfilerDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Response type for the [ProfilerDomain.getBestEffortCoverage] command.
 */
@Serializable
public data class GetBestEffortCoverageResponse(
  /**
   * Coverage data for the current isolate.
   */
  public val result: List<ScriptCoverage>,
)

/**
 * Request object containing input parameters for the [ProfilerDomain.setSamplingInterval] command.
 */
@Serializable
public data class SetSamplingIntervalRequest(
  /**
   * New sampling interval in microseconds.
   */
  public val interval: Int,
)

/**
 * A dummy response object for the [ProfilerDomain.setSamplingInterval] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetSamplingIntervalResponse

/**
 * A dummy response object for the [ProfilerDomain.start] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object StartResponse

/**
 * Request object containing input parameters for the [ProfilerDomain.startPreciseCoverage] command.
 */
@Serializable
public data class StartPreciseCoverageRequest(
  /**
   * Collect accurate call counts beyond simple 'covered' or 'not covered'.
   */
  public val callCount: Boolean? = null,
  /**
   * Collect block-based coverage.
   */
  public val detailed: Boolean? = null,
  /**
   * Allow the backend to send updates on its own initiative
   */
  public val allowTriggeredUpdates: Boolean? = null,
) {
  public class Builder() {
    /**
     * Collect accurate call counts beyond simple 'covered' or 'not covered'.
     */
    public var callCount: Boolean? = null

    /**
     * Collect block-based coverage.
     */
    public var detailed: Boolean? = null

    /**
     * Allow the backend to send updates on its own initiative
     */
    public var allowTriggeredUpdates: Boolean? = null

    public fun build(): StartPreciseCoverageRequest = StartPreciseCoverageRequest(callCount,
        detailed, allowTriggeredUpdates)
  }
}

/**
 * Response type for the [ProfilerDomain.startPreciseCoverage] command.
 */
@Serializable
public data class StartPreciseCoverageResponse(
  /**
   * Monotonically increasing time (in seconds) when the coverage update was taken in the backend.
   */
  public val timestamp: Double,
)

/**
 * Response type for the [ProfilerDomain.stop] command.
 */
@Serializable
public data class StopResponse(
  /**
   * Recorded profile.
   */
  public val profile: Profile,
)

/**
 * A dummy response object for the [ProfilerDomain.stopPreciseCoverage] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object StopPreciseCoverageResponse

/**
 * Response type for the [ProfilerDomain.takePreciseCoverage] command.
 */
@Serializable
public data class TakePreciseCoverageResponse(
  /**
   * Coverage data for the current isolate.
   */
  public val result: List<ScriptCoverage>,
  /**
   * Monotonically increasing time (in seconds) when the coverage update was taken in the backend.
   */
  public val timestamp: Double,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler)
 */
public class ProfilerDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out ProfilerEvent>> =
      mapOf(
      "Profiler.consoleProfileFinished" to serializer<ProfilerEvent.ConsoleProfileFinished>(),
      "Profiler.consoleProfileStarted" to serializer<ProfilerEvent.ConsoleProfileStarted>(),
      "Profiler.preciseCoverageDeltaUpdate" to
          serializer<ProfilerEvent.PreciseCoverageDeltaUpdate>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<ProfilerEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#event-consoleProfileFinished)
   */
  public fun consoleProfileFinishedEvents(): Flow<ProfilerEvent.ConsoleProfileFinished> =
      session.typedEvents("Profiler.consoleProfileFinished")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("consoleProfileFinishedEvents()"),
  )
  public fun consoleProfileFinished(): Flow<ProfilerEvent.ConsoleProfileFinished> =
      consoleProfileFinishedEvents()

  /**
   * Sent when new profile recording is started using console.profile() call.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#event-consoleProfileStarted)
   */
  public fun consoleProfileStartedEvents(): Flow<ProfilerEvent.ConsoleProfileStarted> =
      session.typedEvents("Profiler.consoleProfileStarted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("consoleProfileStartedEvents()"),
  )
  public fun consoleProfileStarted(): Flow<ProfilerEvent.ConsoleProfileStarted> =
      consoleProfileStartedEvents()

  /**
   * Reports coverage delta since the last poll (either from an event like this, or from
   * `takePreciseCoverage` for the current isolate. May only be sent if precise code
   * coverage has been started. This event can be trigged by the embedder to, for example,
   * trigger collection of coverage data immediately at a certain point in time.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#event-preciseCoverageDeltaUpdate)
   */
  @ExperimentalChromeApi
  public fun preciseCoverageDeltaUpdateEvents(): Flow<ProfilerEvent.PreciseCoverageDeltaUpdate> =
      session.typedEvents("Profiler.preciseCoverageDeltaUpdate")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("preciseCoverageDeltaUpdateEvents()"),
  )
  public fun preciseCoverageDeltaUpdate(): Flow<ProfilerEvent.PreciseCoverageDeltaUpdate> =
      preciseCoverageDeltaUpdateEvents()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Profiler.disable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Profiler.enable", Unit)

  /**
   * Collect coverage data for the current isolate. The coverage data may be incomplete due to
   * garbage collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-getBestEffortCoverage)
   */
  public suspend fun getBestEffortCoverage(): GetBestEffortCoverageResponse =
      session.request("Profiler.getBestEffortCoverage", Unit)

  /**
   * Changes CPU profiler sampling interval. Must be called before CPU profiles recording started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-setSamplingInterval)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setSamplingInterval(input: SetSamplingIntervalRequest):
      SetSamplingIntervalResponse = session.request("Profiler.setSamplingInterval", input)

  /**
   * Changes CPU profiler sampling interval. Must be called before CPU profiles recording started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-setSamplingInterval)
   *
   * @param interval New sampling interval in microseconds.
   */
  public suspend fun setSamplingInterval(interval: Int): SetSamplingIntervalResponse {
    val input = SetSamplingIntervalRequest(interval)
    return setSamplingInterval(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-start)
   */
  public suspend fun start(): StartResponse = session.request("Profiler.start", Unit)

  /**
   * Enable precise code coverage. Coverage data for JavaScript executed before enabling precise
   * code
   * coverage may be incomplete. Enabling prevents running optimized code and resets execution
   * counters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-startPreciseCoverage)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun startPreciseCoverage(input: StartPreciseCoverageRequest):
      StartPreciseCoverageResponse = session.request("Profiler.startPreciseCoverage", input)

  /**
   * Enable precise code coverage. Coverage data for JavaScript executed before enabling precise
   * code
   * coverage may be incomplete. Enabling prevents running optimized code and resets execution
   * counters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-startPreciseCoverage)
   */
  @JvmOverloads
  public suspend inline
      fun startPreciseCoverage(optionalArgs: StartPreciseCoverageRequest.Builder.() -> Unit = {}):
      StartPreciseCoverageResponse {
    val builder = StartPreciseCoverageRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return startPreciseCoverage(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-stop)
   */
  public suspend fun stop(): StopResponse = session.request("Profiler.stop", Unit)

  /**
   * Disable precise code coverage. Disabling releases unnecessary execution count records and
   * allows
   * executing optimized code.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-stopPreciseCoverage)
   */
  public suspend fun stopPreciseCoverage(): StopPreciseCoverageResponse =
      session.request("Profiler.stopPreciseCoverage", Unit)

  /**
   * Collect coverage data for the current isolate, and resets execution counters. Precise code
   * coverage needs to have started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-takePreciseCoverage)
   */
  public suspend fun takePreciseCoverage(): TakePreciseCoverageResponse =
      session.request("Profiler.takePreciseCoverage", Unit)
}
