@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.profiler

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.runtime.CallFrame
import org.hildan.chrome.devtools.domains.runtime.ScriptId

/**
 * Profile node. Holds callsite information, execution statistics and child nodes.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#type-ProfileNode)
 */
@Serializable
public data class ProfileNode(
  /**
   * Unique id of the node.
   */
  public val id: Int,
  /**
   * Function location.
   */
  public val callFrame: CallFrame,
  /**
   * Number of samples where this node was on top of the call stack.
   */
  public val hitCount: Int? = null,
  /**
   * Child node ids.
   */
  public val children: List<Int>? = null,
  /**
   * The reason of being not optimized. The function may be deoptimized or marked as don't
   * optimize.
   */
  public val deoptReason: String? = null,
  /**
   * An array of source position ticks.
   */
  public val positionTicks: List<PositionTickInfo>? = null,
)

/**
 * Profile.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#type-Profile)
 */
@Serializable
public data class Profile(
  /**
   * The list of profile nodes. First item is the root node.
   */
  public val nodes: List<ProfileNode>,
  /**
   * Profiling start timestamp in microseconds.
   */
  public val startTime: Double,
  /**
   * Profiling end timestamp in microseconds.
   */
  public val endTime: Double,
  /**
   * Ids of samples top nodes.
   */
  public val samples: List<Int>? = null,
  /**
   * Time intervals between adjacent samples in microseconds. The first delta is relative to the
   * profile startTime.
   */
  public val timeDeltas: List<Int>? = null,
)

/**
 * Specifies a number of samples attributed to a certain source position.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#type-PositionTickInfo)
 */
@Serializable
public data class PositionTickInfo(
  /**
   * Source line number (1-based).
   */
  public val line: Int,
  /**
   * Number of samples attributed to the source line.
   */
  public val ticks: Int,
)

/**
 * Coverage data for a source range.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#type-CoverageRange)
 */
@Serializable
public data class CoverageRange(
  /**
   * JavaScript script source offset for the range start.
   */
  public val startOffset: Int,
  /**
   * JavaScript script source offset for the range end.
   */
  public val endOffset: Int,
  /**
   * Collected execution count of the source range.
   */
  public val count: Int,
)

/**
 * Coverage data for a JavaScript function.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#type-FunctionCoverage)
 */
@Serializable
public data class FunctionCoverage(
  /**
   * JavaScript function name.
   */
  public val functionName: String,
  /**
   * Source ranges inside the function with coverage data.
   */
  public val ranges: List<CoverageRange>,
  /**
   * Whether coverage data for this function has block granularity.
   */
  public val isBlockCoverage: Boolean,
)

/**
 * Coverage data for a JavaScript script.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#type-ScriptCoverage)
 */
@Serializable
public data class ScriptCoverage(
  /**
   * JavaScript script id.
   */
  public val scriptId: ScriptId,
  /**
   * JavaScript script name or url.
   */
  public val url: String,
  /**
   * Functions contained in the script that has coverage data.
   */
  public val functions: List<FunctionCoverage>,
)
