@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.runtime

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.runtime.events.RuntimeEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [RuntimeDomain.awaitPromise] command.
 */
@Serializable
public data class AwaitPromiseRequest(
  /**
   * Identifier of the promise.
   */
  public val promiseObjectId: RemoteObjectId,
  /**
   * Whether the result is expected to be a JSON object that should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  public val generatePreview: Boolean? = null,
) {
  public class Builder(
    /**
     * Identifier of the promise.
     */
    public val promiseObjectId: RemoteObjectId,
  ) {
    /**
     * Whether the result is expected to be a JSON object that should be sent by value.
     */
    public var returnByValue: Boolean? = null

    /**
     * Whether preview should be generated for the result.
     */
    public var generatePreview: Boolean? = null

    public fun build(): AwaitPromiseRequest = AwaitPromiseRequest(promiseObjectId, returnByValue,
        generatePreview)
  }
}

/**
 * Response type for the [RuntimeDomain.awaitPromise] command.
 */
@Serializable
public data class AwaitPromiseResponse(
  /**
   * Promise result. Will contain rejected value if promise was rejected.
   */
  public val result: RemoteObject,
  /**
   * Exception details if stack strace is available.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.callFunctionOn] command.
 */
@Serializable
public data class CallFunctionOnRequest(
  /**
   * Declaration of the function to call.
   */
  public val functionDeclaration: String,
  /**
   * Identifier of the object to call function on. Either objectId or executionContextId should
   * be specified.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Call arguments. All call arguments must belong to the same JavaScript world as the target
   * object.
   */
  public val arguments: List<CallArgument>? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Whether the result is expected to be a JSON object which should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * Whether execution should be treated as initiated by user in the UI.
   */
  public val userGesture: Boolean? = null,
  /**
   * Whether execution should `await` for resulting value and return once awaited promise is
   * resolved.
   */
  public val awaitPromise: Boolean? = null,
  /**
   * Specifies execution context which global object will be used to call function on. Either
   * executionContextId or objectId should be specified.
   */
  public val executionContextId: ExecutionContextId? = null,
  /**
   * Symbolic group name that can be used to release multiple objects. If objectGroup is not
   * specified and objectId is, objectGroup will be inherited from object.
   */
  public val objectGroup: String? = null,
  /**
   * Whether to throw an exception if side effect cannot be ruled out during evaluation.
   */
  @ExperimentalChromeApi
  public val throwOnSideEffect: Boolean? = null,
  /**
   * Whether the result should contain `webDriverValue`, serialized according to
   * https://w3c.github.io/webdriver-bidi. This is mutually exclusive with `returnByValue`, but
   * resulting `objectId` is still provided.
   */
  @ExperimentalChromeApi
  public val generateWebDriverValue: Boolean? = null,
) {
  public class Builder(
    /**
     * Declaration of the function to call.
     */
    public val functionDeclaration: String,
  ) {
    /**
     * Identifier of the object to call function on. Either objectId or executionContextId should
     * be specified.
     */
    public var objectId: RemoteObjectId? = null

    /**
     * Call arguments. All call arguments must belong to the same JavaScript world as the target
     * object.
     */
    public var arguments: List<CallArgument>? = null

    /**
     * In silent mode exceptions thrown during evaluation are not reported and do not pause
     * execution. Overrides `setPauseOnException` state.
     */
    public var silent: Boolean? = null

    /**
     * Whether the result is expected to be a JSON object which should be sent by value.
     */
    public var returnByValue: Boolean? = null

    /**
     * Whether preview should be generated for the result.
     */
    @ExperimentalChromeApi
    public var generatePreview: Boolean? = null

    /**
     * Whether execution should be treated as initiated by user in the UI.
     */
    public var userGesture: Boolean? = null

    /**
     * Whether execution should `await` for resulting value and return once awaited promise is
     * resolved.
     */
    public var awaitPromise: Boolean? = null

    /**
     * Specifies execution context which global object will be used to call function on. Either
     * executionContextId or objectId should be specified.
     */
    public var executionContextId: ExecutionContextId? = null

    /**
     * Symbolic group name that can be used to release multiple objects. If objectGroup is not
     * specified and objectId is, objectGroup will be inherited from object.
     */
    public var objectGroup: String? = null

    /**
     * Whether to throw an exception if side effect cannot be ruled out during evaluation.
     */
    @ExperimentalChromeApi
    public var throwOnSideEffect: Boolean? = null

    /**
     * Whether the result should contain `webDriverValue`, serialized according to
     * https://w3c.github.io/webdriver-bidi. This is mutually exclusive with `returnByValue`, but
     * resulting `objectId` is still provided.
     */
    @ExperimentalChromeApi
    public var generateWebDriverValue: Boolean? = null

    public fun build(): CallFunctionOnRequest = CallFunctionOnRequest(functionDeclaration, objectId,
        arguments, silent, returnByValue, generatePreview, userGesture, awaitPromise,
        executionContextId, objectGroup, throwOnSideEffect, generateWebDriverValue)
  }
}

/**
 * Response type for the [RuntimeDomain.callFunctionOn] command.
 */
@Serializable
public data class CallFunctionOnResponse(
  /**
   * Call result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.compileScript] command.
 */
@Serializable
public data class CompileScriptRequest(
  /**
   * Expression to compile.
   */
  public val expression: String,
  /**
   * Source url to be set for the script.
   */
  public val sourceURL: String,
  /**
   * Specifies whether the compiled script should be persisted.
   */
  public val persistScript: Boolean,
  /**
   * Specifies in which execution context to perform script run. If the parameter is omitted the
   * evaluation will be performed in the context of the inspected page.
   */
  public val executionContextId: ExecutionContextId? = null,
) {
  public class Builder(
    /**
     * Expression to compile.
     */
    public val expression: String,
    /**
     * Source url to be set for the script.
     */
    public val sourceURL: String,
    /**
     * Specifies whether the compiled script should be persisted.
     */
    public val persistScript: Boolean,
  ) {
    /**
     * Specifies in which execution context to perform script run. If the parameter is omitted the
     * evaluation will be performed in the context of the inspected page.
     */
    public var executionContextId: ExecutionContextId? = null

    public fun build(): CompileScriptRequest = CompileScriptRequest(expression, sourceURL,
        persistScript, executionContextId)
  }
}

/**
 * Response type for the [RuntimeDomain.compileScript] command.
 */
@Serializable
public data class CompileScriptResponse(
  /**
   * Id of the script.
   */
  public val scriptId: ScriptId? = null,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * A dummy response object for the [RuntimeDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [RuntimeDomain.discardConsoleEntries] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object DiscardConsoleEntriesResponse

/**
 * A dummy response object for the [RuntimeDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.evaluate] command.
 */
@Serializable
public data class EvaluateRequest(
  /**
   * Expression to evaluate.
   */
  public val expression: String,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
  /**
   * Determines whether Command Line API should be available during the evaluation.
   */
  public val includeCommandLineAPI: Boolean? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Specifies in which execution context to perform evaluation. If the parameter is omitted the
   * evaluation will be performed in the context of the inspected page.
   * This is mutually exclusive with `uniqueContextId`, which offers an
   * alternative way to identify the execution context that is more reliable
   * in a multi-process environment.
   */
  public val contextId: ExecutionContextId? = null,
  /**
   * Whether the result is expected to be a JSON object that should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * Whether execution should be treated as initiated by user in the UI.
   */
  public val userGesture: Boolean? = null,
  /**
   * Whether execution should `await` for resulting value and return once awaited promise is
   * resolved.
   */
  public val awaitPromise: Boolean? = null,
  /**
   * Whether to throw an exception if side effect cannot be ruled out during evaluation.
   * This implies `disableBreaks` below.
   */
  @ExperimentalChromeApi
  public val throwOnSideEffect: Boolean? = null,
  /**
   * Terminate execution after timing out (number of milliseconds).
   */
  @ExperimentalChromeApi
  public val timeout: TimeDelta? = null,
  /**
   * Disable breakpoints during execution.
   */
  @ExperimentalChromeApi
  public val disableBreaks: Boolean? = null,
  /**
   * Setting this flag to true enables `let` re-declaration and top-level `await`.
   * Note that `let` variables can only be re-declared if they originate from
   * `replMode` themselves.
   */
  @ExperimentalChromeApi
  public val replMode: Boolean? = null,
  /**
   * The Content Security Policy (CSP) for the target might block 'unsafe-eval'
   * which includes eval(), Function(), setTimeout() and setInterval()
   * when called with non-callable arguments. This flag bypasses CSP for this
   * evaluation and allows unsafe-eval. Defaults to true.
   */
  @ExperimentalChromeApi
  public val allowUnsafeEvalBlockedByCSP: Boolean? = null,
  /**
   * An alternative way to specify the execution context to evaluate in.
   * Compared to contextId that may be reused across processes, this is guaranteed to be
   * system-unique, so it can be used to prevent accidental evaluation of the expression
   * in context different than intended (e.g. as a result of navigation across process
   * boundaries).
   * This is mutually exclusive with `contextId`.
   */
  @ExperimentalChromeApi
  public val uniqueContextId: String? = null,
  /**
   * Whether the result should be serialized according to https://w3c.github.io/webdriver-bidi.
   */
  @ExperimentalChromeApi
  public val generateWebDriverValue: Boolean? = null,
) {
  public class Builder(
    /**
     * Expression to evaluate.
     */
    public val expression: String,
  ) {
    /**
     * Symbolic group name that can be used to release multiple objects.
     */
    public var objectGroup: String? = null

    /**
     * Determines whether Command Line API should be available during the evaluation.
     */
    public var includeCommandLineAPI: Boolean? = null

    /**
     * In silent mode exceptions thrown during evaluation are not reported and do not pause
     * execution. Overrides `setPauseOnException` state.
     */
    public var silent: Boolean? = null

    /**
     * Specifies in which execution context to perform evaluation. If the parameter is omitted the
     * evaluation will be performed in the context of the inspected page.
     * This is mutually exclusive with `uniqueContextId`, which offers an
     * alternative way to identify the execution context that is more reliable
     * in a multi-process environment.
     */
    public var contextId: ExecutionContextId? = null

    /**
     * Whether the result is expected to be a JSON object that should be sent by value.
     */
    public var returnByValue: Boolean? = null

    /**
     * Whether preview should be generated for the result.
     */
    @ExperimentalChromeApi
    public var generatePreview: Boolean? = null

    /**
     * Whether execution should be treated as initiated by user in the UI.
     */
    public var userGesture: Boolean? = null

    /**
     * Whether execution should `await` for resulting value and return once awaited promise is
     * resolved.
     */
    public var awaitPromise: Boolean? = null

    /**
     * Whether to throw an exception if side effect cannot be ruled out during evaluation.
     * This implies `disableBreaks` below.
     */
    @ExperimentalChromeApi
    public var throwOnSideEffect: Boolean? = null

    /**
     * Terminate execution after timing out (number of milliseconds).
     */
    @ExperimentalChromeApi
    public var timeout: TimeDelta? = null

    /**
     * Disable breakpoints during execution.
     */
    @ExperimentalChromeApi
    public var disableBreaks: Boolean? = null

    /**
     * Setting this flag to true enables `let` re-declaration and top-level `await`.
     * Note that `let` variables can only be re-declared if they originate from
     * `replMode` themselves.
     */
    @ExperimentalChromeApi
    public var replMode: Boolean? = null

    /**
     * The Content Security Policy (CSP) for the target might block 'unsafe-eval'
     * which includes eval(), Function(), setTimeout() and setInterval()
     * when called with non-callable arguments. This flag bypasses CSP for this
     * evaluation and allows unsafe-eval. Defaults to true.
     */
    @ExperimentalChromeApi
    public var allowUnsafeEvalBlockedByCSP: Boolean? = null

    /**
     * An alternative way to specify the execution context to evaluate in.
     * Compared to contextId that may be reused across processes, this is guaranteed to be
     * system-unique, so it can be used to prevent accidental evaluation of the expression
     * in context different than intended (e.g. as a result of navigation across process
     * boundaries).
     * This is mutually exclusive with `contextId`.
     */
    @ExperimentalChromeApi
    public var uniqueContextId: String? = null

    /**
     * Whether the result should be serialized according to https://w3c.github.io/webdriver-bidi.
     */
    @ExperimentalChromeApi
    public var generateWebDriverValue: Boolean? = null

    public fun build(): EvaluateRequest = EvaluateRequest(expression, objectGroup,
        includeCommandLineAPI, silent, contextId, returnByValue, generatePreview, userGesture,
        awaitPromise, throwOnSideEffect, timeout, disableBreaks, replMode,
        allowUnsafeEvalBlockedByCSP, uniqueContextId, generateWebDriverValue)
  }
}

/**
 * Response type for the [RuntimeDomain.evaluate] command.
 */
@Serializable
public data class EvaluateResponse(
  /**
   * Evaluation result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Response type for the [RuntimeDomain.getIsolateId] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetIsolateIdResponse(
  /**
   * The isolate id.
   */
  public val id: String,
)

/**
 * Response type for the [RuntimeDomain.getHeapUsage] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHeapUsageResponse(
  /**
   * Used heap size in bytes.
   */
  public val usedSize: Double,
  /**
   * Allocated heap size in bytes.
   */
  public val totalSize: Double,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.getProperties] command.
 */
@Serializable
public data class GetPropertiesRequest(
  /**
   * Identifier of the object to return properties for.
   */
  public val objectId: RemoteObjectId,
  /**
   * If true, returns properties belonging only to the element itself, not to its prototype
   * chain.
   */
  public val ownProperties: Boolean? = null,
  /**
   * If true, returns accessor properties (with getter/setter) only; internal properties are not
   * returned either.
   */
  @ExperimentalChromeApi
  public val accessorPropertiesOnly: Boolean? = null,
  /**
   * Whether preview should be generated for the results.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * If true, returns non-indexed properties only.
   */
  @ExperimentalChromeApi
  public val nonIndexedPropertiesOnly: Boolean? = null,
) {
  public class Builder(
    /**
     * Identifier of the object to return properties for.
     */
    public val objectId: RemoteObjectId,
  ) {
    /**
     * If true, returns properties belonging only to the element itself, not to its prototype
     * chain.
     */
    public var ownProperties: Boolean? = null

    /**
     * If true, returns accessor properties (with getter/setter) only; internal properties are not
     * returned either.
     */
    @ExperimentalChromeApi
    public var accessorPropertiesOnly: Boolean? = null

    /**
     * Whether preview should be generated for the results.
     */
    @ExperimentalChromeApi
    public var generatePreview: Boolean? = null

    /**
     * If true, returns non-indexed properties only.
     */
    @ExperimentalChromeApi
    public var nonIndexedPropertiesOnly: Boolean? = null

    public fun build(): GetPropertiesRequest = GetPropertiesRequest(objectId, ownProperties,
        accessorPropertiesOnly, generatePreview, nonIndexedPropertiesOnly)
  }
}

/**
 * Response type for the [RuntimeDomain.getProperties] command.
 */
@Serializable
public data class GetPropertiesResponse(
  /**
   * Object properties.
   */
  public val result: List<PropertyDescriptor>,
  /**
   * Internal object properties (only of the element itself).
   */
  public val internalProperties: List<InternalPropertyDescriptor>? = null,
  /**
   * Object private properties.
   */
  @ExperimentalChromeApi
  public val privateProperties: List<PrivatePropertyDescriptor>? = null,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.globalLexicalScopeNames]
 * command.
 */
@Serializable
public data class GlobalLexicalScopeNamesRequest(
  /**
   * Specifies in which execution context to lookup global scope variables.
   */
  public val executionContextId: ExecutionContextId? = null,
) {
  public class Builder() {
    /**
     * Specifies in which execution context to lookup global scope variables.
     */
    public var executionContextId: ExecutionContextId? = null

    public fun build(): GlobalLexicalScopeNamesRequest =
        GlobalLexicalScopeNamesRequest(executionContextId)
  }
}

/**
 * Response type for the [RuntimeDomain.globalLexicalScopeNames] command.
 */
@Serializable
public data class GlobalLexicalScopeNamesResponse(
  public val names: List<String>,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.queryObjects] command.
 */
@Serializable
public data class QueryObjectsRequest(
  /**
   * Identifier of the prototype to return objects for.
   */
  public val prototypeObjectId: RemoteObjectId,
  /**
   * Symbolic group name that can be used to release the results.
   */
  public val objectGroup: String? = null,
) {
  public class Builder(
    /**
     * Identifier of the prototype to return objects for.
     */
    public val prototypeObjectId: RemoteObjectId,
  ) {
    /**
     * Symbolic group name that can be used to release the results.
     */
    public var objectGroup: String? = null

    public fun build(): QueryObjectsRequest = QueryObjectsRequest(prototypeObjectId, objectGroup)
  }
}

/**
 * Response type for the [RuntimeDomain.queryObjects] command.
 */
@Serializable
public data class QueryObjectsResponse(
  /**
   * Array with objects.
   */
  public val objects: RemoteObject,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.releaseObject] command.
 */
@Serializable
public data class ReleaseObjectRequest(
  /**
   * Identifier of the object to release.
   */
  public val objectId: RemoteObjectId,
)

/**
 * A dummy response object for the [RuntimeDomain.releaseObject] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ReleaseObjectResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.releaseObjectGroup] command.
 */
@Serializable
public data class ReleaseObjectGroupRequest(
  /**
   * Symbolic object group name.
   */
  public val objectGroup: String,
)

/**
 * A dummy response object for the [RuntimeDomain.releaseObjectGroup] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ReleaseObjectGroupResponse

/**
 * A dummy response object for the [RuntimeDomain.runIfWaitingForDebugger] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object RunIfWaitingForDebuggerResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.runScript] command.
 */
@Serializable
public data class RunScriptRequest(
  /**
   * Id of the script to run.
   */
  public val scriptId: ScriptId,
  /**
   * Specifies in which execution context to perform script run. If the parameter is omitted the
   * evaluation will be performed in the context of the inspected page.
   */
  public val executionContextId: ExecutionContextId? = null,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Determines whether Command Line API should be available during the evaluation.
   */
  public val includeCommandLineAPI: Boolean? = null,
  /**
   * Whether the result is expected to be a JSON object which should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  public val generatePreview: Boolean? = null,
  /**
   * Whether execution should `await` for resulting value and return once awaited promise is
   * resolved.
   */
  public val awaitPromise: Boolean? = null,
) {
  public class Builder(
    /**
     * Id of the script to run.
     */
    public val scriptId: ScriptId,
  ) {
    /**
     * Specifies in which execution context to perform script run. If the parameter is omitted the
     * evaluation will be performed in the context of the inspected page.
     */
    public var executionContextId: ExecutionContextId? = null

    /**
     * Symbolic group name that can be used to release multiple objects.
     */
    public var objectGroup: String? = null

    /**
     * In silent mode exceptions thrown during evaluation are not reported and do not pause
     * execution. Overrides `setPauseOnException` state.
     */
    public var silent: Boolean? = null

    /**
     * Determines whether Command Line API should be available during the evaluation.
     */
    public var includeCommandLineAPI: Boolean? = null

    /**
     * Whether the result is expected to be a JSON object which should be sent by value.
     */
    public var returnByValue: Boolean? = null

    /**
     * Whether preview should be generated for the result.
     */
    public var generatePreview: Boolean? = null

    /**
     * Whether execution should `await` for resulting value and return once awaited promise is
     * resolved.
     */
    public var awaitPromise: Boolean? = null

    public fun build(): RunScriptRequest = RunScriptRequest(scriptId, executionContextId,
        objectGroup, silent, includeCommandLineAPI, returnByValue, generatePreview, awaitPromise)
  }
}

/**
 * Response type for the [RuntimeDomain.runScript] command.
 */
@Serializable
public data class RunScriptResponse(
  /**
   * Run result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Request object containing input parameters for the [RuntimeDomain.setAsyncCallStackDepth]
 * command.
 */
@Serializable
public data class SetAsyncCallStackDepthRequest(
  /**
   * Maximum depth of async call stacks. Setting to `0` will effectively disable collecting async
   * call stacks (default).
   */
  public val maxDepth: Int,
)

/**
 * A dummy response object for the [RuntimeDomain.setAsyncCallStackDepth] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetAsyncCallStackDepthResponse

/**
 * Request object containing input parameters for the
 * [RuntimeDomain.setCustomObjectFormatterEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetCustomObjectFormatterEnabledRequest(
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [RuntimeDomain.setCustomObjectFormatterEnabled] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetCustomObjectFormatterEnabledResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.setMaxCallStackSizeToCapture]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetMaxCallStackSizeToCaptureRequest(
  public val size: Int,
)

/**
 * A dummy response object for the [RuntimeDomain.setMaxCallStackSizeToCapture] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetMaxCallStackSizeToCaptureResponse

/**
 * A dummy response object for the [RuntimeDomain.terminateExecution] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object TerminateExecutionResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.addBinding] command.
 */
@Serializable
@ExperimentalChromeApi
public data class AddBindingRequest(
  public val name: String,
  /**
   * If specified, the binding would only be exposed to the specified
   * execution context. If omitted and `executionContextName` is not set,
   * the binding is exposed to all execution contexts of the target.
   * This parameter is mutually exclusive with `executionContextName`.
   * Deprecated in favor of `executionContextName` due to an unclear use case
   * and bugs in implementation (crbug.com/1169639). `executionContextId` will be
   * removed in the future.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val executionContextId: ExecutionContextId? = null,
  /**
   * If specified, the binding is exposed to the executionContext with
   * matching name, even for contexts created after the binding is added.
   * See also `ExecutionContext.name` and `worldName` parameter to
   * `Page.addScriptToEvaluateOnNewDocument`.
   * This parameter is mutually exclusive with `executionContextId`.
   */
  @ExperimentalChromeApi
  public val executionContextName: String? = null,
) {
  public class Builder(
    public val name: String,
  ) {
    /**
     * If specified, the binding would only be exposed to the specified
     * execution context. If omitted and `executionContextName` is not set,
     * the binding is exposed to all execution contexts of the target.
     * This parameter is mutually exclusive with `executionContextName`.
     * Deprecated in favor of `executionContextName` due to an unclear use case
     * and bugs in implementation (crbug.com/1169639). `executionContextId` will be
     * removed in the future.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var executionContextId: ExecutionContextId? = null

    /**
     * If specified, the binding is exposed to the executionContext with
     * matching name, even for contexts created after the binding is added.
     * See also `ExecutionContext.name` and `worldName` parameter to
     * `Page.addScriptToEvaluateOnNewDocument`.
     * This parameter is mutually exclusive with `executionContextId`.
     */
    @ExperimentalChromeApi
    public var executionContextName: String? = null

    public fun build(): AddBindingRequest = AddBindingRequest(name, executionContextId,
        executionContextName)
  }
}

/**
 * A dummy response object for the [RuntimeDomain.addBinding] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
@ExperimentalChromeApi
public object AddBindingResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.removeBinding] command.
 */
@Serializable
@ExperimentalChromeApi
public data class RemoveBindingRequest(
  public val name: String,
)

/**
 * A dummy response object for the [RuntimeDomain.removeBinding] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object RemoveBindingResponse

/**
 * Request object containing input parameters for the [RuntimeDomain.getExceptionDetails] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetExceptionDetailsRequest(
  /**
   * The error object for which to resolve the exception details.
   */
  public val errorObjectId: RemoteObjectId,
)

/**
 * Response type for the [RuntimeDomain.getExceptionDetails] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetExceptionDetailsResponse(
  public val exceptionDetails: ExceptionDetails? = null,
)

/**
 * Runtime domain exposes JavaScript runtime by means of remote evaluation and mirror objects.
 * Evaluation results are returned as mirror object that expose object type, string representation
 * and unique identifier that can be used for further object reference. Original objects are
 * maintained in memory unless they are either explicitly released or are released along with the
 * other objects in their object group.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime)
 */
public class RuntimeDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out RuntimeEvent>> =
      mapOf(
      "Runtime.bindingCalled" to serializer<RuntimeEvent.BindingCalled>(),
      "Runtime.consoleAPICalled" to serializer<RuntimeEvent.ConsoleAPICalled>(),
      "Runtime.exceptionRevoked" to serializer<RuntimeEvent.ExceptionRevoked>(),
      "Runtime.exceptionThrown" to serializer<RuntimeEvent.ExceptionThrown>(),
      "Runtime.executionContextCreated" to serializer<RuntimeEvent.ExecutionContextCreated>(),
      "Runtime.executionContextDestroyed" to serializer<RuntimeEvent.ExecutionContextDestroyed>(),
      "Runtime.executionContextsCleared" to serializer<RuntimeEvent.ExecutionContextsCleared>(),
      "Runtime.inspectRequested" to serializer<RuntimeEvent.InspectRequested>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<RuntimeEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Notification is issued every time when binding is called.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-bindingCalled)
   */
  @ExperimentalChromeApi
  public fun bindingCalledEvents(): Flow<RuntimeEvent.BindingCalled> =
      session.typedEvents("Runtime.bindingCalled")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("bindingCalledEvents()"),
  )
  public fun bindingCalled(): Flow<RuntimeEvent.BindingCalled> = bindingCalledEvents()

  /**
   * Issued when console API was called.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-consoleAPICalled)
   */
  public fun consoleAPICalledEvents(): Flow<RuntimeEvent.ConsoleAPICalled> =
      session.typedEvents("Runtime.consoleAPICalled")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("consoleAPICalledEvents()"),
  )
  public fun consoleAPICalled(): Flow<RuntimeEvent.ConsoleAPICalled> = consoleAPICalledEvents()

  /**
   * Issued when unhandled exception was revoked.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-exceptionRevoked)
   */
  public fun exceptionRevokedEvents(): Flow<RuntimeEvent.ExceptionRevoked> =
      session.typedEvents("Runtime.exceptionRevoked")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("exceptionRevokedEvents()"),
  )
  public fun exceptionRevoked(): Flow<RuntimeEvent.ExceptionRevoked> = exceptionRevokedEvents()

  /**
   * Issued when exception was thrown and unhandled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-exceptionThrown)
   */
  public fun exceptionThrownEvents(): Flow<RuntimeEvent.ExceptionThrown> =
      session.typedEvents("Runtime.exceptionThrown")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("exceptionThrownEvents()"),
  )
  public fun exceptionThrown(): Flow<RuntimeEvent.ExceptionThrown> = exceptionThrownEvents()

  /**
   * Issued when new execution context is created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-executionContextCreated)
   */
  public fun executionContextCreatedEvents(): Flow<RuntimeEvent.ExecutionContextCreated> =
      session.typedEvents("Runtime.executionContextCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("executionContextCreatedEvents()"),
  )
  public fun executionContextCreated(): Flow<RuntimeEvent.ExecutionContextCreated> =
      executionContextCreatedEvents()

  /**
   * Issued when execution context is destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-executionContextDestroyed)
   */
  public fun executionContextDestroyedEvents(): Flow<RuntimeEvent.ExecutionContextDestroyed> =
      session.typedEvents("Runtime.executionContextDestroyed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("executionContextDestroyedEvents()"),
  )
  public fun executionContextDestroyed(): Flow<RuntimeEvent.ExecutionContextDestroyed> =
      executionContextDestroyedEvents()

  /**
   * Issued when all executionContexts were cleared in browser
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-executionContextsCleared)
   */
  public fun executionContextsClearedEvents(): Flow<RuntimeEvent.ExecutionContextsCleared> =
      session.typedEvents("Runtime.executionContextsCleared")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("executionContextsClearedEvents()"),
  )
  public fun executionContextsCleared(): Flow<RuntimeEvent.ExecutionContextsCleared> =
      executionContextsClearedEvents()

  /**
   * Issued when object should be inspected (for example, as a result of inspect() command line API
   * call).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-inspectRequested)
   */
  public fun inspectRequestedEvents(): Flow<RuntimeEvent.InspectRequested> =
      session.typedEvents("Runtime.inspectRequested")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("inspectRequestedEvents()"),
  )
  public fun inspectRequested(): Flow<RuntimeEvent.InspectRequested> = inspectRequestedEvents()

  /**
   * Add handler to promise with given promise object id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-awaitPromise)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun awaitPromise(input: AwaitPromiseRequest): AwaitPromiseResponse =
      session.request("Runtime.awaitPromise", input)

  /**
   * Add handler to promise with given promise object id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-awaitPromise)
   *
   * @param promiseObjectId Identifier of the promise.
   */
  @JvmOverloads
  public suspend inline fun awaitPromise(promiseObjectId: RemoteObjectId,
      optionalArgs: AwaitPromiseRequest.Builder.() -> Unit = {}): AwaitPromiseResponse {
    val builder = AwaitPromiseRequest.Builder(promiseObjectId)
    val input = builder.apply(optionalArgs).build()
    return awaitPromise(input)
  }

  /**
   * Calls function with given declaration on the given object. Object group of the result is
   * inherited from the target object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-callFunctionOn)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun callFunctionOn(input: CallFunctionOnRequest): CallFunctionOnResponse =
      session.request("Runtime.callFunctionOn", input)

  /**
   * Calls function with given declaration on the given object. Object group of the result is
   * inherited from the target object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-callFunctionOn)
   *
   * @param functionDeclaration Declaration of the function to call.
   */
  @JvmOverloads
  public suspend inline fun callFunctionOn(functionDeclaration: String,
      optionalArgs: CallFunctionOnRequest.Builder.() -> Unit = {}): CallFunctionOnResponse {
    val builder = CallFunctionOnRequest.Builder(functionDeclaration)
    val input = builder.apply(optionalArgs).build()
    return callFunctionOn(input)
  }

  /**
   * Compiles expression.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-compileScript)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun compileScript(input: CompileScriptRequest): CompileScriptResponse =
      session.request("Runtime.compileScript", input)

  /**
   * Compiles expression.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-compileScript)
   *
   * @param expression Expression to compile.
   * @param sourceURL Source url to be set for the script.
   * @param persistScript Specifies whether the compiled script should be persisted.
   */
  @JvmOverloads
  public suspend inline fun compileScript(
    expression: String,
    sourceURL: String,
    persistScript: Boolean,
    optionalArgs: CompileScriptRequest.Builder.() -> Unit = {},
  ): CompileScriptResponse {
    val builder = CompileScriptRequest.Builder(expression, sourceURL, persistScript)
    val input = builder.apply(optionalArgs).build()
    return compileScript(input)
  }

  /**
   * Disables reporting of execution contexts creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Runtime.disable", Unit)

  /**
   * Discards collected exceptions and console API calls.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-discardConsoleEntries)
   */
  public suspend fun discardConsoleEntries(): DiscardConsoleEntriesResponse =
      session.request("Runtime.discardConsoleEntries", Unit)

  /**
   * Enables reporting of execution contexts creation by means of `executionContextCreated` event.
   * When the reporting gets enabled the event will be sent immediately for each existing execution
   * context.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Runtime.enable", Unit)

  /**
   * Evaluates expression on global object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-evaluate)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun evaluate(input: EvaluateRequest): EvaluateResponse =
      session.request("Runtime.evaluate", input)

  /**
   * Evaluates expression on global object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-evaluate)
   *
   * @param expression Expression to evaluate.
   */
  @JvmOverloads
  public suspend inline fun evaluate(expression: String,
      optionalArgs: EvaluateRequest.Builder.() -> Unit = {}): EvaluateResponse {
    val builder = EvaluateRequest.Builder(expression)
    val input = builder.apply(optionalArgs).build()
    return evaluate(input)
  }

  /**
   * Returns the isolate id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getIsolateId)
   */
  @ExperimentalChromeApi
  public suspend fun getIsolateId(): GetIsolateIdResponse = session.request("Runtime.getIsolateId",
      Unit)

  /**
   * Returns the JavaScript heap usage.
   * It is the total usage of the corresponding isolate not scoped to a particular Runtime.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getHeapUsage)
   */
  @ExperimentalChromeApi
  public suspend fun getHeapUsage(): GetHeapUsageResponse = session.request("Runtime.getHeapUsage",
      Unit)

  /**
   * Returns properties of a given object. Object group of the result is inherited from the target
   * object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getProperties)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getProperties(input: GetPropertiesRequest): GetPropertiesResponse =
      session.request("Runtime.getProperties", input)

  /**
   * Returns properties of a given object. Object group of the result is inherited from the target
   * object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getProperties)
   *
   * @param objectId Identifier of the object to return properties for.
   */
  @JvmOverloads
  public suspend inline fun getProperties(objectId: RemoteObjectId,
      optionalArgs: GetPropertiesRequest.Builder.() -> Unit = {}): GetPropertiesResponse {
    val builder = GetPropertiesRequest.Builder(objectId)
    val input = builder.apply(optionalArgs).build()
    return getProperties(input)
  }

  /**
   * Returns all let, const and class variables from global scope.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-globalLexicalScopeNames)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun globalLexicalScopeNames(input: GlobalLexicalScopeNamesRequest):
      GlobalLexicalScopeNamesResponse = session.request("Runtime.globalLexicalScopeNames", input)

  /**
   * Returns all let, const and class variables from global scope.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-globalLexicalScopeNames)
   */
  @JvmOverloads
  public suspend inline
      fun globalLexicalScopeNames(optionalArgs: GlobalLexicalScopeNamesRequest.Builder.() -> Unit =
      {}): GlobalLexicalScopeNamesResponse {
    val builder = GlobalLexicalScopeNamesRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return globalLexicalScopeNames(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-queryObjects)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun queryObjects(input: QueryObjectsRequest): QueryObjectsResponse =
      session.request("Runtime.queryObjects", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-queryObjects)
   *
   * @param prototypeObjectId Identifier of the prototype to return objects for.
   */
  @JvmOverloads
  public suspend inline fun queryObjects(prototypeObjectId: RemoteObjectId,
      optionalArgs: QueryObjectsRequest.Builder.() -> Unit = {}): QueryObjectsResponse {
    val builder = QueryObjectsRequest.Builder(prototypeObjectId)
    val input = builder.apply(optionalArgs).build()
    return queryObjects(input)
  }

  /**
   * Releases remote object with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-releaseObject)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun releaseObject(input: ReleaseObjectRequest): ReleaseObjectResponse =
      session.request("Runtime.releaseObject", input)

  /**
   * Releases remote object with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-releaseObject)
   *
   * @param objectId Identifier of the object to release.
   */
  public suspend fun releaseObject(objectId: RemoteObjectId): ReleaseObjectResponse {
    val input = ReleaseObjectRequest(objectId)
    return releaseObject(input)
  }

  /**
   * Releases all remote objects that belong to a given group.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-releaseObjectGroup)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun releaseObjectGroup(input: ReleaseObjectGroupRequest):
      ReleaseObjectGroupResponse = session.request("Runtime.releaseObjectGroup", input)

  /**
   * Releases all remote objects that belong to a given group.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-releaseObjectGroup)
   *
   * @param objectGroup Symbolic object group name.
   */
  public suspend fun releaseObjectGroup(objectGroup: String): ReleaseObjectGroupResponse {
    val input = ReleaseObjectGroupRequest(objectGroup)
    return releaseObjectGroup(input)
  }

  /**
   * Tells inspected instance to run if it was waiting for debugger to attach.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-runIfWaitingForDebugger)
   */
  public suspend fun runIfWaitingForDebugger(): RunIfWaitingForDebuggerResponse =
      session.request("Runtime.runIfWaitingForDebugger", Unit)

  /**
   * Runs script with given id in a given context.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-runScript)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun runScript(input: RunScriptRequest): RunScriptResponse =
      session.request("Runtime.runScript", input)

  /**
   * Runs script with given id in a given context.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-runScript)
   *
   * @param scriptId Id of the script to run.
   */
  @JvmOverloads
  public suspend inline fun runScript(scriptId: ScriptId,
      optionalArgs: RunScriptRequest.Builder.() -> Unit = {}): RunScriptResponse {
    val builder = RunScriptRequest.Builder(scriptId)
    val input = builder.apply(optionalArgs).build()
    return runScript(input)
  }

  /**
   * Enables or disables async call stacks tracking.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setAsyncCallStackDepth)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setAsyncCallStackDepth(input: SetAsyncCallStackDepthRequest):
      SetAsyncCallStackDepthResponse = session.request("Runtime.setAsyncCallStackDepth", input)

  /**
   * Enables or disables async call stacks tracking.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setAsyncCallStackDepth)
   *
   * @param maxDepth Maximum depth of async call stacks. Setting to `0` will effectively disable
   * collecting async
   * call stacks (default).
   */
  public suspend fun setAsyncCallStackDepth(maxDepth: Int): SetAsyncCallStackDepthResponse {
    val input = SetAsyncCallStackDepthRequest(maxDepth)
    return setAsyncCallStackDepth(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setCustomObjectFormatterEnabled)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setCustomObjectFormatterEnabled(input: SetCustomObjectFormatterEnabledRequest):
      SetCustomObjectFormatterEnabledResponse =
      session.request("Runtime.setCustomObjectFormatterEnabled", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setCustomObjectFormatterEnabled)
   */
  @ExperimentalChromeApi
  public suspend fun setCustomObjectFormatterEnabled(enabled: Boolean):
      SetCustomObjectFormatterEnabledResponse {
    val input = SetCustomObjectFormatterEnabledRequest(enabled)
    return setCustomObjectFormatterEnabled(input)
  }

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setMaxCallStackSizeToCapture)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setMaxCallStackSizeToCapture(input: SetMaxCallStackSizeToCaptureRequest):
      SetMaxCallStackSizeToCaptureResponse = session.request("Runtime.setMaxCallStackSizeToCapture",
      input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setMaxCallStackSizeToCapture)
   */
  @ExperimentalChromeApi
  public suspend fun setMaxCallStackSizeToCapture(size: Int): SetMaxCallStackSizeToCaptureResponse {
    val input = SetMaxCallStackSizeToCaptureRequest(size)
    return setMaxCallStackSizeToCapture(input)
  }

  /**
   * Terminate current or next JavaScript execution.
   * Will cancel the termination when the outer-most script execution ends.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-terminateExecution)
   */
  @ExperimentalChromeApi
  public suspend fun terminateExecution(): TerminateExecutionResponse =
      session.request("Runtime.terminateExecution", Unit)

  /**
   * If executionContextId is empty, adds binding with the given name on the
   * global objects of all inspected contexts, including those created later,
   * bindings survive reloads.
   * Binding function takes exactly one argument, this argument should be string,
   * in case of any other input, function throws an exception.
   * Each binding function call produces Runtime.bindingCalled notification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-addBinding)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun addBinding(input: AddBindingRequest): AddBindingResponse =
      session.request("Runtime.addBinding", input)

  /**
   * If executionContextId is empty, adds binding with the given name on the
   * global objects of all inspected contexts, including those created later,
   * bindings survive reloads.
   * Binding function takes exactly one argument, this argument should be string,
   * in case of any other input, function throws an exception.
   * Each binding function call produces Runtime.bindingCalled notification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-addBinding)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun addBinding(name: String,
      optionalArgs: AddBindingRequest.Builder.() -> Unit = {}): AddBindingResponse {
    val builder = AddBindingRequest.Builder(name)
    val input = builder.apply(optionalArgs).build()
    return addBinding(input)
  }

  /**
   * This method does not remove binding function from global object but
   * unsubscribes current runtime agent from Runtime.bindingCalled notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-removeBinding)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun removeBinding(input: RemoveBindingRequest): RemoveBindingResponse =
      session.request("Runtime.removeBinding", input)

  /**
   * This method does not remove binding function from global object but
   * unsubscribes current runtime agent from Runtime.bindingCalled notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-removeBinding)
   */
  @ExperimentalChromeApi
  public suspend fun removeBinding(name: String): RemoveBindingResponse {
    val input = RemoveBindingRequest(name)
    return removeBinding(input)
  }

  /**
   * This method tries to lookup and populate exception details for a
   * JavaScript Error object.
   * Note that the stackTrace portion of the resulting exceptionDetails will
   * only be populated if the Runtime domain was enabled at the time when the
   * Error was thrown.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getExceptionDetails)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getExceptionDetails(input: GetExceptionDetailsRequest):
      GetExceptionDetailsResponse = session.request("Runtime.getExceptionDetails", input)

  /**
   * This method tries to lookup and populate exception details for a
   * JavaScript Error object.
   * Note that the stackTrace portion of the resulting exceptionDetails will
   * only be populated if the Runtime domain was enabled at the time when the
   * Error was thrown.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getExceptionDetails)
   *
   * @param errorObjectId The error object for which to resolve the exception details.
   */
  @ExperimentalChromeApi
  public suspend fun getExceptionDetails(errorObjectId: RemoteObjectId):
      GetExceptionDetailsResponse {
    val input = GetExceptionDetailsRequest(errorObjectId)
    return getExceptionDetails(input)
  }
}
